const dataMigrationHelpers = require("./helpers/dataMigrationHelpers.js");
const db = require("./helpers/dynamoDBHelpers.js");
const logger = require("./logger.js");
const constants = require("./helpers/constants.js");
const commonHelpers = require("./helpers/commonHelpers.js");

// time interval in ms between running the cleaner
const CLEANER_CHECK_INTERVAL = 10000; // every 10 seconds

// only clean this channel, or across all channels if this is null
let channelIDToClean = null;

module.exports.startCleaner = function(channelID = null) {
    channelIDToClean = channelID;
    if (channelID != null) {
        logger.info(`Cleaner started in restricted mode. It will only operate on channel ${channelID}.`);
    } else {
        logger.info("Cleaner started in non-restricted mode. It will operate on all channels.");
    }
    setInterval(cleanLiveData, CLEANER_CHECK_INTERVAL);
}

async function cleanLiveData() {
    const gameCleanerSuccess = await cleanUpOldGames();

    if (!gameCleanerSuccess) {
        logger.error("An error occurred while cleaning up old games.");
        return false;
    }

    return true;
}

async function cleanUpOldGames() {
    const dateCutoffForOldGames = new Date(Date.now() - constants.TIME_UNTIL_GAME_CONSIDERED_INACTIVE_MS).toISOString();
    const filterExpression = 'updatedAt <= :t';
    const expressionAttributeValues = {':t' : dateCutoffForOldGames};
    const getLiveGamesResponse = await db.dynamoScan(db.DynamoTables.LiveGames, filterExpression, expressionAttributeValues);

    if (!getLiveGamesResponse.success) {
        logger.error("Failed to scan LiveGames to clean up old games.");
        return false;
    }

    const recordGameResultsPromises = [];
    const migratePlayersPromises = [];
    const removePlayerInputsPromises = [];

    for (const game of getLiveGamesResponse.items) {
        // Only run the cleaner on one channel if it was specified
        if (channelIDToClean != null && game.channelID != channelIDToClean) {
            return true;
        }

        logger.info(`Game ${game.gameID} was found inactive. Ending it and migrating its data to the history tables.`);

        const playerRecordsForGameResult = await getPlayerRecordsForGivenGame(game.gameID);
        if (!playerRecordsForGameResult.success) {
            logger.error(`Failed to look up players in game ${game.gameID}.`);
            return false;
        }

        // Migrate player data to history
        logger.info(`Migrating ${playerRecordsForGameResult.count} player(s) from game ${game.gameID} to PlayerHistory.`);
        for (const playerRecord of playerRecordsForGameResult.items) {
            let result = constants.GAME_ABORTED_DUE_TO_INACTIVITY;
            const gameHasWinner = game.storedState.gameWinnerMetadata != null;
            if (gameHasWinner) {
                const isWinner = commonHelpers.userIsWinnerOfGameState(playerRecord.opaqueUserID, game.storedState);
                result = isWinner? constants.GAME_END_WON : constants.GAME_END_LOST;
            }
            migratePlayersPromises.push(dataMigrationHelpers.migratePlayerRecordFromLiveToHistory(playerRecord, result));
        }

        // Remove player inputs
        game.storedState.playerStates.forEach((player) => {
            removePlayerInputsPromises.push(removePlayerInputsFromLive(player.playerMetadata.opaqueUserID, game.gameID));
        })

        // Migrate game data to history
        migratePlayersPromises.push(dataMigrationHelpers.migrateGameRecordFromLiveToHistory(game));
    }

    const cleanUpOldGamesResult = await Promise.all([...recordGameResultsPromises, ...migratePlayersPromises, ...removePlayerInputsPromises]);
    if (cleanUpOldGamesResult.includes(false)) {
        logger.error("Live data cleaner ran into an issue while removing old games.");
        return false;
    }

    return true;
}

async function getPlayerRecordsForGivenGame(gameID) {
    const filterExpression = 'gameID = :id';
    const expressionAttributeValues = {':id' : gameID};
    return db.dynamoScan(db.DynamoTables.LivePlayers, filterExpression, expressionAttributeValues);
}

async function removePlayerInputsFromLive(opaqueUserID, gameID) {
    const deletePlayerInputsResponse = await db.dynamoDelete(
        db.DynamoTables.LivePlayerInputs, 
        {gameID: gameID, opaqueUserID: opaqueUserID}
    );
    if (!deletePlayerInputsResponse.success) {
        logger.error(`An error occurred while removing inputs for player ${opaqueUserID} for game ${gameID} from the LivePlayerInputs table.`);
        return false;
    }
    return true;
}