const matchmakingHelpers = require("./helpers/matchmakingHelpers.js");
const commonHelpers = require("./helpers/commonHelpers.js");
const twitchHelpers = require("./helpers/twitchHelpers.js");
const constants = require("./helpers/constants.js");
const logger = require("./logger.js");

// time interval in ms between checking whether we should start some games
const MATCHMAKER_CHECK_INTERVAL = 3000;
// delay in ms between MIN_PLAYERS_PER_GAME joining matchmaking and a game automatically starting
const AUTOMATIC_MATCH_START_DELAY = 15000;
module.exports.MAX_PLAYERS_PER_GAME = 25;
module.exports.MIN_PLAYERS_PER_GAME = 2;

// only matchmake users in this channel, or across all channels if this is null
let channelIDToMatchmake = null;

module.exports.startMatchmaker = function(channelID = null) {
    channelIDToMatchmake = channelID;
    if (channelID != null) {
        logger.info(`Matchmaker started in restricted mode. It will only operate on channel ${channelID}.`);
    } else {
        logger.info("Matchmaker started in non-restricted mode. It will operate on all channels.");
    }
    setInterval(module.exports.matchmakeAllChannels, MATCHMAKER_CHECK_INTERVAL);
}

module.exports.matchmakeAllChannels = async function() {
    let liveChannels = await matchmakingHelpers.getLiveChannels();

    if (!liveChannels.success) {
        logger.error("An error occurred while matchmaking all channels. Unable to get the list of currently live channels.");
        return;
    }

    liveChannels.items.forEach(channel => {
        matchmakeChannel(channel.channelID, channel.automaticMatchStartTimestamp);
    })
}

async function matchmakeChannel(channelID, automaticMatchStartTimestamp) {
    // Only run the matchmaker on one channel if it was specified
    if (channelIDToMatchmake != null && channelID != channelIDToMatchmake) {
        return;
    }

    let getPlayersMatchmakingResponse = await matchmakingHelpers.getPlayersMatchmaking(channelID);
    if (!getPlayersMatchmakingResponse.success) {
        logger.error(`An error occurred while matchmaking channel ${channelID}. Unable to get the set of players currently matchmaking on channel.`);
        return;
    }

    let playersMatchmaking = getPlayersMatchmakingResponse.items;
    let gamesStarted = false;

    // immediately create matches when we have over MAX_PLAYERS_PER_GAME matchmaking
    const startGamePromises = [];

    while (playersMatchmaking.length >= module.exports.MAX_PLAYERS_PER_GAME) {
        logger.info(`There were ${playersMatchmaking.length} >= ${module.exports.MAX_PLAYERS_PER_GAME} players matchmaking on channel ${channelID}, starting a game.`);
        // sort by matchmaking start timestamp to ensure players who have been waiting the longest get into a match
        playersMatchmaking.sort((p1, p2) => p1.matchmakingStartTimestamp - p2.matchmakingStartTimestamp);
        let playerBatch = playersMatchmaking.splice(0, module.exports.MAX_PLAYERS_PER_GAME);
        startGamePromises.push(matchmakingHelpers.startGameForPlayers(channelID, playerBatch));
        gamesStarted = true;
    }

    await Promise.all(startGamePromises);

    // automatically schedule (or act on) an automatic start if we don't have enough players for a full game
    if (!gamesStarted && playersMatchmaking.length >= module.exports.MIN_PLAYERS_PER_GAME) {
        if (automaticMatchStartTimestamp < 0) {
            automaticMatchStartTimestamp = Date.now() + AUTOMATIC_MATCH_START_DELAY;
            await matchmakingHelpers.setAutomaticMatchStartTimestamp(channelID, automaticMatchStartTimestamp);
        } 
        else if (Date.now() >= automaticMatchStartTimestamp) {
            logger.info(`Automatically starting a game for the ${playersMatchmaking.length} players matchmaking on channel ${channelID}.`);
            await matchmakingHelpers.startGameForPlayers(channelID, playersMatchmaking);
            gamesStarted = true;
        }
    }

    // cancel automatic match start if we just started games or are currently lacking the minimum number of players
    if (gamesStarted || playersMatchmaking.length < module.exports.MIN_PLAYERS_PER_GAME) {
        automaticMatchStartTimestamp = constants.NULL_TIMESTAMP;
        await matchmakingHelpers.setAutomaticMatchStartTimestamp(channelID, automaticMatchStartTimestamp);
    }

    // Get the number of players playing in the channel
    const getNumPlayersInChannelResponse = await commonHelpers.getNumActivePlayersInChannel(channelID);

    let numPlayersInChannel = 0;
    if (!getNumPlayersInChannelResponse.error) {
        numPlayersInChannel = getNumPlayersInChannelResponse.numPlayers;
    }

    twitchHelpers.broadcast({
        channelID: channelID,
        message: { 
            type: "matchmakingUpdate",
            automaticMatchStartTimestamp: (automaticMatchStartTimestamp == constants.NULL_TIMESTAMP? null : automaticMatchStartTimestamp),
            matchmakingPlayersCount: getPlayersMatchmakingResponse.count,
            activePlayersCount: numPlayersInChannel,
            gamesStartedSinceLastUpdate: gamesStarted,
            maxPlayersPerGame: module.exports.MAX_PLAYERS_PER_GAME,
        },
    });
}