const twitchHelpers = require("../helpers/twitchHelpers.js");
const JWT = require("jsonwebtoken");
const logger = require("../logger.js");

module.exports = function(req, res, next) {
    // Do not require authentication for root path (used in the health check on our EC2s)
    if (req.path == '/') {
        return next();
    }

    const secret = twitchHelpers.getExtensionSecret();

    if (secret == null) {
        logger.error("An error occurred while obtaining the extension secret.");
        res.statusMessage = "Something went wrong during authentication.";
        res.status(500).end();
        return;
    }

    const jwt = req.header("X-Game-Authorization");
    let decodedJWT = null;

    try {
        decodedJWT = JWT.verify(jwt, secret);
    } catch(error) {
        logger.error(`An error occurred while decoding the user's JWT: ${error}`);
        res.statusMessage = "User authentication failed: missing or invalid JWT.";
        res.status(403).end();
        return;
    }

    logger.info(`The decoded JWT contains: ${JSON.stringify(decodedJWT)}`);
    // See https://dev.twitch.tv/docs/extensions/reference/#jwt-schema
    // for more details on the data the JWT contains. 
    res.locals.channelID = decodedJWT.channel_id;
    res.locals.userID = decodedJWT.first_party_user_id || decodedJWT.user_id;
    res.locals.opaqueUserID = decodedJWT.opaque_user_id;
    res.locals.role = decodedJWT.role;
    return next();
}