const express = require('express');
const router = express.Router();
const commonHelpers = require("../helpers/commonHelpers.js");
const constants = require("../helpers/constants.js");
const logger = require("../logger.js");

router.get('/getSession', async function(req, res, next) {
  const response = {
    ongoingSessionExists: false,
    matchmaking: false,
    matchmakingStartTimestamp: constants.NULL_TIMESTAMP,
    numberOfPlayersInChannel: 0,
    gameState: null,
    currentTimeOnServer: Date.now()
  }

  // Check if there is an ongoing session on the channel.
  const channelHasOngoingSessionPromise = commonHelpers.channelHasOngoingSession(res.locals.channelID);

  // Get the user's session state in the active session on the channel.
  const userSessionStatePromise = commonHelpers.getUserSessionState(res.locals.channelID, res.locals.opaqueUserID);

  const [channelHasOngoingSession, userSessionState] = await Promise.all([channelHasOngoingSessionPromise, userSessionStatePromise]);

  if (channelHasOngoingSession.error) {
    logger.error(`An error occurred during /getSession. Unable to determine whether there is an active session for channel ${res.locals.channelID}.`);
    res.status(500).end();
    return;
  }

  if (channelHasOngoingSession.error || userSessionState.error) {
    logger.error(`An error occurred during /getSession. Unable to get the user session state for user ${res.locals.opaqueUserID} on channel ${res.locals.channelID}.`);
    res.status(500).end();
    return;
  }

  response.ongoingSessionExists = channelHasOngoingSession.result;
  response.matchmaking = userSessionState.matchmaking;
  const gameID = userSessionState.gameID;

  // Get the user's game state in the active session on the channel.
  const userGameState = await commonHelpers.getGameState(gameID);

  if (userGameState.error) {
    logger.error(`An error occurred during /getSession. Unable to get the user game state for gameID ${gameID}`);
    res.status(500).end();
    return;
  }

  response.gameState = userGameState.gameState;
  res.json(response);
});

module.exports = router;