const express = require('express');
const router = express.Router();
const twitchHelpers = require("../helpers/twitchHelpers.js");
const constants = require("../helpers/constants.js");
const matchmakingHelpers = require("../helpers/matchmakingHelpers.js");
const db = require("../helpers/dynamoDBHelpers.js");
const uuidv1 = require('uuid/v1');
const commonHelpers = require("../helpers/commonHelpers.js");
const logger = require("../logger.js");

router.post('/startSession', async function(req, res, next) {
  const sessionID = uuidv1();

  const dbResponse = await db.dynamoPut(db.DynamoTables.LiveSessions, {
    channelID: res.locals.channelID,
    sessionID: sessionID,
    automaticMatchStartTimestamp: constants.NULL_TIMESTAMP,
  });

  if (!dbResponse.success) {
    logger.error(`An error occurred during /startSession on channel ${res.locals.channelID}. Unable to create an entry in the LiveSessions table.`);
    res.status(500).end();
    return;
  }

  twitchHelpers.broadcast({
    channelID: res.locals.channelID,
    message: {
      type: "sessionUpdate",
    }
  });

  res.status(200).end();
});

router.post('/requestSessionEnd', async function(req, res, next) {
  const endTime = commonHelpers.getISOTimestamp();

  // Get the currently active session.
  const getLiveSessionResponse = await db.dynamoGet(db.DynamoTables.LiveSessions, {
    channelID: res.locals.channelID
  });

  if (!getLiveSessionResponse.success) {
    logger.error(`An error occurred during /requestSessionEnd on channel ${res.locals.channelID}. The read from the LiveSessions table was unsuccessful.`);
    res.status(500).end();
    return;
  }

  if (!getLiveSessionResponse.itemFound) {
    logger.warn(`Request session end was called on a non existent session for channel ${res.locals.channelID}.`);
    res.status(200).end();
    return;
  }

  // Collect the data we want to record about the session.
  const newSessionHistoryData = {
    sessionID: getLiveSessionResponse.item.sessionID,
    channelID: getLiveSessionResponse.item.channelID,
    startTime: getLiveSessionResponse.item.createdAt,
    endTime: endTime
  };

  // Delete the session from Live Sessions.
  const deleteLiveSessionResponse = await db.dynamoDelete(db.DynamoTables.LiveSessions, {channelID: res.locals.channelID});

  if (!deleteLiveSessionResponse.success) {
    logger.error(`An error occurred during /requestSessionEnd. The delete from the LiveSessions table was unsuccessful for channel ${res.locals.channelID}.`);
    res.status(500).end();
    return;
  }

  // Remove all users from Live Players who are still matchmaking in the channel.
  const sessionEndCancelMatchmakingForChannelSuccess = await matchmakingHelpers.sessionEndCancelMatchmakingForChannel(res.locals.channelID, endTime);

  if (!sessionEndCancelMatchmakingForChannelSuccess) {
    logger.error(`An error occurred during /requestSessionEnd. Unable to cancel matchmaking for all currently matchmaking players on channel ${res.locals.channelID}.`);
    res.status(500).end();
    return;
  }

  // Tell the users on the channel that there has been an update to the Sumo session.
  twitchHelpers.broadcast({
    channelID: res.locals.channelID,
    message: { 
        type: "sessionUpdate",
    },  
  });

  // Record the historical data to the Session History table.
  commonHelpers.addSessionHistoryEntry(newSessionHistoryData);
  
  res.status(200).end();
});

module.exports = router;
