const express = require('express');
const router = express.Router();
const db = require("../helpers/dynamoDBHelpers.js");
const dataMigrationHelpers = require("../helpers/dataMigrationHelpers.js");
const commonHelpers = require("../helpers/commonHelpers.js");
const constants = require("../helpers/constants.js");
const logger = require("../logger.js");

router.post('/startMatchmaking', async function(req, res, next) {
  const channelID = res.locals.channelID;

  const channelHasOngoingSessionResponse = await commonHelpers.channelHasOngoingSession(channelID);

  if (channelHasOngoingSessionResponse.error) {
      logger.error(`Unable to determine whether an active session exists on channel ${channelID}. Rejecting request to start matchmaking.`);
      res.status(500).end();
      return;
  }

  if (!channelHasOngoingSessionResponse.result) {
      logger.error(`There is no active session on channel ${channelID}. Rejecting request to start matchmaking.`);
      res.status(403).end();
      return;
  }

  const dbResponse = await db.dynamoPut(db.DynamoTables.LivePlayers, {
    channelID: res.locals.channelID,
    opaqueUserID: res.locals.opaqueUserID,
    userID: res.locals.userID ? res.locals.userID : constants.NO_USER_ID,
    gameID: constants.NO_GAME_ID,
    matchmakingStartTimestamp: Date.now()
  });

  if (!dbResponse.success) {
    logger.error(`An error occurred during /startMatchmaking. Unable to add the player to the LivePlayers table.`);
    res.status(500).end();
    return;
  }

  res.status(200).end();
});

router.post('/submitGameInput', async function(req, res, next) {
  if (req.body == null || req.body.input == null) {
    logger.error("An error occurred during /submitGameInput. Unable to find input in the request body.");
    res.status(500).end();
    return;
  }
  
  // Get the user's gameID on the current channel.
  const userSessionState = await commonHelpers.getUserSessionState(res.locals.channelID, res.locals.opaqueUserID);

  if (userSessionState.error) {
    logger.error("An error occurred during /submitGameInput. Unable to get the user session state.");
    res.status(500).end();
    return;
  }

  if (userSessionState.gameID == constants.NO_GAME_ID) {
    logger.error(`An error occurred during /submitGameInput. User ${res.locals.opaqueUserID} is not in an active game in channel ${res.locals.channelID}.`);
    res.status(403).end();
    return;
  }

  /* Store the input in DynamoDB.
  Note: since the composite key for this table is gameID and opaqueUserID, each player
  will never have more than one input record per game stored in the database. Instead,
  the database will only keep track of the last input for each player. */
  const dbResponse = await db.dynamoPut(db.DynamoTables.LivePlayerInputs, {
    gameID: userSessionState.gameID,
    opaqueUserID: res.locals.opaqueUserID,
    input: req.body.input.input,
    roundNumber: req.body.input.roundNumber,
  });

  if (!dbResponse.success) {
    logger.error("An error occurred during /submitGameInput. Unable to add to the LivePlayerInputs table.");
    res.status(500).end();
    return;
  }

  logger.info(`The user's input was: ${JSON.stringify(req.body.input)}`);

  res.json({ submittedInput: req.body.input });
});

router.post('/leaveGame', async function(req, res, next) {
  const success = await dataMigrationHelpers.migratePlayerIDFromLiveToHistory(res.locals.channelID, res.locals.opaqueUserID);
  
  if (!success) {
    logger.error(`An error occurred during /leaveGame for channelID: ${res.locals.channelID}, opaqueUserID: ${res.locals.opaqueUserID}`);
    res.status(500).end();
    return;
  }

  res.status(200).end();
});

router.post('/cancelMatchmaking', async function(req, res, next) {
  const success = await dataMigrationHelpers.migratePlayerIDFromLiveToHistory(res.locals.channelID, res.locals.opaqueUserID, constants.MATCHMAKING_CANCELLED_BY_USER);
  
  if (!success) {
    logger.error(`An error occurred during /cancelMatchmaking for channelID: ${res.locals.channelID}, opaqueUserID: ${res.locals.opaqueUserID}`);
    res.status(500).end();
    return;
  }

  res.status(200).end();
});

module.exports = router;