package api

import (
	. "code.justin.tv/samus/gateway/util"
	log "github.com/sirupsen/logrus"

	"encoding/json"
	"fmt"
	"net/http"

	"strings"

	"code.justin.tv/samus/gateway/backend"
	samus_gateway "code.justin.tv/samus/gateway/client"
	"github.com/pkg/errors"
	"goji.io/pat"
	"golang.org/x/net/context"
)

var gameNameToID = map[string]string{
	"callofduty":      "ATVI",
	"rainbowsixsiege": "UBI",
	"rocketleague":    "PSYNX",
}

func (s *Server) CreateAccountLink(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	userID := pat.Param(ctx, "userId")
	validationErrMsg := ValidateTuid(userID)
	if validationErrMsg != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, validationErrMsg)
		return
	}
	gameName := getGameName(r.URL.Path, ctx)

	gameID, found := gameNameToID[gameName]

	if !found {
		s.serveError(ctx, w, r, http.StatusNotFound, errors.New("trying to create link for unregistered game"))
		return
	}

	var requestBody samus_gateway.CreateAccountLinkRequestBody
	err := json.NewDecoder(r.Body).Decode(&requestBody)
	if err != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, err)
	}

	log.Infof("Creating account link for %s", gameName)

	accountLink := backend.GameAccountLink{
		UserID: userID,
		GameID: gameID,
		GameAccount: backend.GameAccount{
			AccountID:   requestBody.GameAccount.AccountID,
			DisplayName: requestBody.GameAccount.DisplayName,
		},
	}

	log.Info(fmt.Sprintf("Creating account link [%+v]", accountLink))
	status, err := s.Backend.CreateAccountLink(ctx, accountLink)

	if status == http.StatusConflict {
		s.serveError(ctx, w, r, status, errors.New("an account is already linked"))
		return
	} else if status != http.StatusOK {
		log.WithError(err).Errorf("could not create account link: %+v", err)
		s.serveError(ctx, w, r, status, errors.New("could not create account link"))
		return
	}

	log.Info(fmt.Sprintf("Created account link [%+v]", accountLink))
	s.serveJSON(ctx, w, r, samus_gateway.CreateAccountLinkResponse{
		AccountLink: convert(accountLink),
	})
	return
}

func (s *Server) CreateAccountLinkV2(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	userID := pat.Param(ctx, "userId")
	validationErrMsg := ValidateTuid(userID)
	if validationErrMsg != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, validationErrMsg)
		return
	}
	vendorID := pat.Param(ctx, "vendorId")

	var requestBody samus_gateway.CreateAccountLinkRequestBody
	err := json.NewDecoder(r.Body).Decode(&requestBody)
	if err != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, err)
	}

	accountLink := backend.GameAccountLink{
		UserID: userID,
		GameID: vendorID,
		GameAccount: backend.GameAccount{
			AccountID:   requestBody.GameAccount.AccountID,
			DisplayName: requestBody.GameAccount.DisplayName,
		},
	}

	log.Infof("Creating account link [%+v]\nContext: [%+v]", accountLink, ctx)
	status, err := s.Backend.CreateAccountLink(ctx, accountLink)

	if status == http.StatusConflict {
		s.serveError(ctx, w, r, status, errors.New("an account is already linked"))
		return
	} else if status != http.StatusOK {
		log.WithError(err).Errorf("could not create account link: %+v", err)
		s.serveError(ctx, w, r, status, errors.New("could not create account link"))
		return
	}

	log.Infof("Created account link [%+v]\nContext: [%+v]", accountLink, ctx)
	s.serveJSON(ctx, w, r, samus_gateway.CreateAccountLinkResponse{
		AccountLink: convert(accountLink),
	})
	return
}

func (s *Server) GetAccountLink(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	userID := pat.Param(ctx, "userId")
	validationErrMsg := ValidateTuid(userID)
	if validationErrMsg != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, validationErrMsg)
		return
	}
	gameName := getGameName(r.URL.Path, ctx)

	gameID, found := gameNameToID[gameName]

	if !found {
		s.serveError(ctx, w, r, http.StatusNotFound, errors.New("trying to get link for unregistered game"))
		return
	}

	link, status, err := s.Backend.GetAccountLink(ctx, userID, gameID)

	if status == http.StatusNotFound {
		s.serveError(ctx, w, r, status, errors.New("no link found"))
		return
	} else if status != http.StatusOK {
		log.WithError(err).Errorf("link lookup failed: %+v", err)
		s.serveError(ctx, w, r, status, errors.New("could not get account link"))
		return
	}

	s.serveJSON(ctx, w, r, samus_gateway.GetAccountLinkResponse{
		AccountLink: convert(*link),
	})
	return
}

// directly gets account link by the vendorId (e.g. publishers such as ATVI, UBI etc)
// this API is to be called by SACS for getting publisher account of known vendor, and does not need to be exposed in visage.
func (s *Server) GetAccountLinkV2(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	userID := pat.Param(ctx, "userId")
	validationErrMsg := ValidateTuid(userID)
	if validationErrMsg != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, validationErrMsg)
		return
	}
	vendorID := pat.Param(ctx, "vendorId")

	link, status, err := s.Backend.GetAccountLink(ctx, userID, vendorID)

	if status == http.StatusNotFound {
		s.serveError(ctx, w, r, status, errors.New("no link found"))
		return
	} else if status != http.StatusOK {
		log.WithError(err).Errorf("link lookup failed: %+v", err)
		s.serveError(ctx, w, r, status, errors.New("could not get account link"))
		return
	}

	s.serveJSON(ctx, w, r, samus_gateway.GetAccountLinkResponse{
		AccountLink: convert(*link),
	})
	return
}

func (s *Server) DeleteAccountLink(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	userID := pat.Param(ctx, "userId")
	validationErrMsg := ValidateTuid(userID)
	if validationErrMsg != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, validationErrMsg)
		return
	}
	gameName := getGameName(r.URL.Path, ctx)

	gameID, found := gameNameToID[gameName]

	if !found {
		s.serveError(ctx, w, r, http.StatusNotFound, errors.New("trying to delete link for unregistered game"))
		return
	}

	log.Infof("Deleting account link for user %s for game %s\nContext: [%+v]", userID, gameName, ctx)

	previous, status, err := s.Backend.DeleteAccountLink(ctx, userID, gameID)
	if status != http.StatusOK {
		log.WithError(err).Errorf("failed to delete account link: %+v", err)
		s.serveError(ctx, w, r, status, errors.New("failed to delete account link"))
		return
	}

	log.Infof("Deleted account link for user %s for game %s: %v\nContext: [%+v]", userID, gameName, previous, ctx)
	response := samus_gateway.DeleteAccountLinkResponse{}
	if previous != nil {
		deleted := convert(*previous)
		response.Deleted = &deleted
	}

	s.serveJSON(ctx, w, r, response)
	return
}

func (s *Server) DeleteAccountLinkV2(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	userID := pat.Param(ctx, "userId")
	validationErrMsg := ValidateTuid(userID)
	if validationErrMsg != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, validationErrMsg)
		return
	}
	vendorID := pat.Param(ctx, "vendorId")

	log.Infof("Deleting account link for user %s for game %s\nContext: [%+v]", userID, vendorID, ctx)

	previous, status, err := s.Backend.DeleteAccountLink(ctx, userID, vendorID)
	if status != http.StatusOK {
		log.WithError(err).Errorf("failed to delete account link: %+v", err)
		s.serveError(ctx, w, r, status, errors.New("failed to delete account link"))
		return
	}

	log.Infof("Deleted account link for user %s for game %s: %+v\nContext: [%+v]", userID, vendorID, previous, ctx)
	response := samus_gateway.DeleteAccountLinkResponse{}
	if previous != nil {
		deleted := convert(*previous)
		response.Deleted = &deleted
	}

	s.serveJSON(ctx, w, r, response)
	return
}

func convert(link backend.GameAccountLink) samus_gateway.GameAccountLink {
	return samus_gateway.GameAccountLink{
		UserID: link.UserID,
		GameAccount: samus_gateway.GameAccount{
			AccountID:   link.GameAccount.AccountID,
			DisplayName: link.GameAccount.DisplayName,
		},
	}
}

/*
	Creating to keep parity with current behavior but we should remove this after visage changes have been deployed
*/
func getGameName(path string, ctx context.Context) string {
	gameName := "callofduty"
	splitPath := strings.Split(path, "/")
	splitPathLength := len(splitPath)

	if splitPathLength > 5 {
		gameName = pat.Param(ctx, "gameName")
	}

	return gameName
}
