package api

import (
	"net/http"
	"testing"

	"fmt"

	"context"
	"net/http/httptest"

	"strings"

	"code.justin.tv/common/config"
	"code.justin.tv/samus/gateway/backend"
	samus_gateway "code.justin.tv/samus/gateway/client"
	samus_config "code.justin.tv/samus/gateway/configuration"
	"code.justin.tv/samus/gateway/metrics"
	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

const twitchUserID = "1231231234"
const gameName = "callofduty"
const gameID = "ATVI"
const accountID = "55556666"
const displayName = "zachwazhere"

func TestGetGameName(t *testing.T) {

	Convey("GetGameNameForAccountLinkAPIs", t, func() {
		Convey("When getting game name based on url with no game name", func() {

			userID := "1231231234"
			gameName := "callofduty"
			path := fmt.Sprintf("/api/users/%s/link", userID)
			name := getGameName(path, context.TODO())

			So(name, ShouldEqual, gameName)
		})
	})
}

func TestCreateAccountLink(t *testing.T) {
	Convey("CreateAccountLink API", t, func() {
		log.SetLevel(log.DebugLevel)
		b := &backend.BackendMock{}
		metricsConf := &metrics.MetricsConfig{
			Environment:   "UnitTest",
			MetricsRegion: "PDX",
		}
		configs := samus_config.SamusConfigurations{MetricsConfig: metricsConf}
		s, err := NewServer(config.Statsd(), config.RollbarErrorLogger(), b, configs)
		So(err, ShouldBeNil)

		testServer := httptest.NewServer(s)
		defer testServer.Close()

		Convey("When HTTP status OK", func() {
			b.On("CreateAccountLink", mock.Anything, mock.Anything).Return(http.StatusOK, nil)

			resp, err := http.Post(fmt.Sprintf("%v/api/users/%s/link/%s", testServer.URL, twitchUserID, gameName), "application/json", strings.NewReader(fmt.Sprintf("{\"gameAccount\": { \"accountId\": \"%s\", \"displayName\": \"%s\"}}", accountID, displayName)))
			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			createAccountLinkResponse := samus_gateway.CreateAccountLinkResponse{}
			ConvertToResponseInterface(resp, &createAccountLinkResponse)

			accountLink := createAccountLinkResponse.AccountLink
			So(accountLink, ShouldNotBeNil)

			So(accountLink.UserID, ShouldEqual, twitchUserID)

			gameAccount := accountLink.GameAccount
			So(gameAccount, ShouldNotBeNil)

			So(gameAccount.DisplayName, ShouldEqual, displayName)
			So(gameAccount.AccountID, ShouldEqual, accountID)
		})

		Convey("When HTTP status Conflict", func() {
			b.On("CreateAccountLink", mock.Anything, mock.Anything).Return(http.StatusConflict, nil)

			resp, err := http.Post(fmt.Sprintf("%v/api/users/%s/link/%s", testServer.URL, twitchUserID, gameName), "application/json", strings.NewReader(fmt.Sprintf("{\"gameAccount\": { \"accountId\": \"%s\", \"displayName\": \"%s\"}}", accountID, displayName)))
			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusConflict)
		})

		Convey("When HTTP status Error", func() {
			b.On("CreateAccountLink", mock.Anything, mock.Anything).Return(http.StatusInternalServerError, errors.New("An error occurred"))

			resp, err := http.Post(fmt.Sprintf("%v/api/users/%s/link/%s", testServer.URL, twitchUserID, gameName), "application/json", strings.NewReader(fmt.Sprintf("{\"gameAccount\": { \"accountId\": \"%s\", \"displayName\": \"%s\"}}", accountID, displayName)))
			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusInternalServerError)
		})
	})
}

func TestGetAccountLink(t *testing.T) {
	Convey("GetAccountLink API", t, func() {
		log.SetLevel(log.DebugLevel)
		b := &backend.BackendMock{}
		metricsConf := &metrics.MetricsConfig{
			Environment:   "UnitTest",
			MetricsRegion: "PDX",
		}
		configs := samus_config.SamusConfigurations{MetricsConfig: metricsConf}
		s, err := NewServer(config.Statsd(), config.RollbarErrorLogger(), b, configs)
		So(err, ShouldBeNil)

		testServer := httptest.NewServer(s)
		defer testServer.Close()

		Convey("When HTTP status OK", func() {
			b.On("GetAccountLink", mock.Anything, twitchUserID, gameID).Return(&backend.GameAccountLink{
				UserID: twitchUserID,
				GameID: gameID,
				GameAccount: backend.GameAccount{
					AccountID:   accountID,
					DisplayName: displayName,
				},
			}, http.StatusOK, nil)

			resp, err := http.Get(fmt.Sprintf("%v/api/users/%s/link/%s", testServer.URL, twitchUserID, gameName))
			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			getAccountLinkResponse := samus_gateway.GetAccountLinkResponse{}
			ConvertToResponseInterface(resp, &getAccountLinkResponse)

			accountLink := getAccountLinkResponse.AccountLink
			So(accountLink, ShouldNotBeNil)

			So(accountLink.UserID, ShouldEqual, twitchUserID)

			gameAccount := accountLink.GameAccount
			So(gameAccount, ShouldNotBeNil)

			So(gameAccount.DisplayName, ShouldEqual, displayName)
			So(gameAccount.AccountID, ShouldEqual, accountID)
		})

		Convey("When HTTP status Not Found", func() {
			b.On("GetAccountLink", mock.Anything, twitchUserID, gameID).Return(nil, http.StatusNotFound, nil)

			resp, err := http.Get(fmt.Sprintf("%v/api/users/%s/link/%s", testServer.URL, twitchUserID, gameName))
			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusNotFound)
		})

		Convey("When HTTP status Error", func() {
			b.On("GetAccountLink", mock.Anything, twitchUserID, gameID).Return(nil, http.StatusInternalServerError, errors.New("An error occurred"))

			resp, err := http.Get(fmt.Sprintf("%v/api/users/%s/link/%s", testServer.URL, twitchUserID, gameName))
			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusInternalServerError)
		})
	})
}

func TestDeleteAccountLink(t *testing.T) {
	Convey("DeleteAccountLink API", t, func() {
		log.SetLevel(log.DebugLevel)
		b := &backend.BackendMock{}
		metricsConf := &metrics.MetricsConfig{
			Environment:   "UnitTest",
			MetricsRegion: "PDX",
		}
		configs := samus_config.SamusConfigurations{MetricsConfig: metricsConf}
		s, err := NewServer(config.Statsd(), config.RollbarErrorLogger(), b, configs)
		So(err, ShouldBeNil)

		testServer := httptest.NewServer(s)
		defer testServer.Close()

		Convey("When HTTP status OK", func() {
			b.On("DeleteAccountLink", mock.Anything, twitchUserID, gameID).Return(&backend.GameAccountLink{
				UserID: twitchUserID,
				GameID: gameID,
				GameAccount: backend.GameAccount{
					AccountID:   accountID,
					DisplayName: displayName,
				},
			}, http.StatusOK, nil)

			client := &http.Client{}
			req, err := http.NewRequest("DELETE", fmt.Sprintf("%v/api/users/%s/link/%s", testServer.URL, twitchUserID, gameName), nil)
			resp, err := client.Do(req)

			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			deleteAccountLinkResponse := samus_gateway.DeleteAccountLinkResponse{}
			ConvertToResponseInterface(resp, &deleteAccountLinkResponse)

			accountLink := deleteAccountLinkResponse.Deleted
			So(accountLink, ShouldNotBeNil)

			So(accountLink.UserID, ShouldEqual, twitchUserID)

			gameAccount := accountLink.GameAccount
			So(gameAccount, ShouldNotBeNil)

			So(gameAccount.DisplayName, ShouldEqual, displayName)
			So(gameAccount.AccountID, ShouldEqual, accountID)
		})

		Convey("When HTTP status Not Found", func() {
			b.On("DeleteAccountLink", mock.Anything, twitchUserID, gameID).Return(nil, http.StatusNotFound, nil)

			client := &http.Client{}
			req, err := http.NewRequest("DELETE", fmt.Sprintf("%v/api/users/%s/link/%s", testServer.URL, twitchUserID, gameName), nil)
			resp, err := client.Do(req)

			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusNotFound)
		})

		Convey("When HTTP status Error", func() {
			b.On("DeleteAccountLink", mock.Anything, twitchUserID, gameID).Return(nil, http.StatusInternalServerError, errors.New("An error occurred"))

			client := &http.Client{}
			req, err := http.NewRequest("DELETE", fmt.Sprintf("%v/api/users/%s/link/%s", testServer.URL, twitchUserID, gameName), nil)
			resp, err := client.Do(req)

			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusInternalServerError)
		})
	})
}

func TestCreateAccountLinkV2(t *testing.T) {
	Convey("CreateAccountLinkV2 API", t, func() {
		log.SetLevel(log.DebugLevel)
		b := &backend.BackendMock{}
		metricsConf := &metrics.MetricsConfig{
			Environment:   "UnitTest",
			MetricsRegion: "PDX",
		}
		configs := samus_config.SamusConfigurations{MetricsConfig: metricsConf}
		s, err := NewServer(config.Statsd(), config.RollbarErrorLogger(), b, configs)
		So(err, ShouldBeNil)

		testServer := httptest.NewServer(s)
		defer testServer.Close()

		Convey("When HTTP status OK", func() {
			b.On("CreateAccountLink", mock.Anything, mock.Anything).Return(http.StatusOK, nil)

			resp, err := http.Post(fmt.Sprintf("%v/api/v2/users/%s/link/%s", testServer.URL, twitchUserID, gameID), "application/json", strings.NewReader(fmt.Sprintf("{\"gameAccount\": { \"accountId\": \"%s\", \"displayName\": \"%s\"}}", accountID, displayName)))
			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			createAccountLinkResponse := samus_gateway.CreateAccountLinkResponse{}
			ConvertToResponseInterface(resp, &createAccountLinkResponse)

			accountLink := createAccountLinkResponse.AccountLink
			So(accountLink, ShouldNotBeNil)

			So(accountLink.UserID, ShouldEqual, twitchUserID)

			gameAccount := accountLink.GameAccount
			So(gameAccount, ShouldNotBeNil)

			So(gameAccount.DisplayName, ShouldEqual, displayName)
			So(gameAccount.AccountID, ShouldEqual, accountID)
		})

		Convey("When HTTP status Conflict", func() {
			b.On("CreateAccountLink", mock.Anything, mock.Anything).Return(http.StatusConflict, nil)

			resp, err := http.Post(fmt.Sprintf("%v/api/v2/users/%s/link/%s", testServer.URL, twitchUserID, gameID), "application/json", strings.NewReader(fmt.Sprintf("{\"gameAccount\": { \"accountId\": \"%s\", \"displayName\": \"%s\"}}", accountID, displayName)))
			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusConflict)
		})

		Convey("When HTTP status Error", func() {
			b.On("CreateAccountLink", mock.Anything, mock.Anything).Return(http.StatusInternalServerError, errors.New("An error occurred"))

			resp, err := http.Post(fmt.Sprintf("%v/api/v2/users/%s/link/%s", testServer.URL, twitchUserID, gameID), "application/json", strings.NewReader(fmt.Sprintf("{\"gameAccount\": { \"accountId\": \"%s\", \"displayName\": \"%s\"}}", accountID, displayName)))
			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusInternalServerError)
		})
	})
}

func TestGetAccountLinkV2(t *testing.T) {
	Convey("GetAccountLink API", t, func() {
		log.SetLevel(log.DebugLevel)
		b := &backend.BackendMock{}
		metricsConf := &metrics.MetricsConfig{
			Environment:   "UnitTest",
			MetricsRegion: "PDX",
		}
		configs := samus_config.SamusConfigurations{MetricsConfig: metricsConf}
		s, err := NewServer(config.Statsd(), config.RollbarErrorLogger(), b, configs)
		So(err, ShouldBeNil)

		testServer := httptest.NewServer(s)
		defer testServer.Close()

		Convey("When HTTP status OK", func() {
			b.On("GetAccountLink", mock.Anything, twitchUserID, gameID).Return(&backend.GameAccountLink{
				UserID: twitchUserID,
				GameID: gameID,
				GameAccount: backend.GameAccount{
					AccountID:   accountID,
					DisplayName: displayName,
				},
			}, http.StatusOK, nil)

			resp, err := http.Get(fmt.Sprintf("%v/api/v2/users/%s/link/%s", testServer.URL, twitchUserID, gameID))
			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			getAccountLinkResponse := samus_gateway.GetAccountLinkResponse{}
			ConvertToResponseInterface(resp, &getAccountLinkResponse)

			accountLink := getAccountLinkResponse.AccountLink
			So(accountLink, ShouldNotBeNil)

			So(accountLink.UserID, ShouldEqual, twitchUserID)

			gameAccount := accountLink.GameAccount
			So(gameAccount, ShouldNotBeNil)

			So(gameAccount.DisplayName, ShouldEqual, displayName)
			So(gameAccount.AccountID, ShouldEqual, accountID)
		})

		Convey("When HTTP status Not Found", func() {
			b.On("GetAccountLink", mock.Anything, twitchUserID, gameID).Return(nil, http.StatusNotFound, nil)

			resp, err := http.Get(fmt.Sprintf("%v/api/v2/users/%s/link/%s", testServer.URL, twitchUserID, gameID))
			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusNotFound)
		})

		Convey("When HTTP status Error", func() {
			b.On("GetAccountLink", mock.Anything, twitchUserID, gameID).Return(nil, http.StatusInternalServerError, errors.New("An error occurred"))

			resp, err := http.Get(fmt.Sprintf("%v/api/v2/users/%s/link/%s", testServer.URL, twitchUserID, gameID))
			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusInternalServerError)
		})
	})
}

func TestDeleteAccountLinkV2(t *testing.T) {
	Convey("DeleteAccountLink API", t, func() {
		log.SetLevel(log.DebugLevel)
		b := &backend.BackendMock{}
		metricsConf := &metrics.MetricsConfig{
			Environment:   "UnitTest",
			MetricsRegion: "PDX",
		}
		configs := samus_config.SamusConfigurations{MetricsConfig: metricsConf}
		s, err := NewServer(config.Statsd(), config.RollbarErrorLogger(), b, configs)
		So(err, ShouldBeNil)

		testServer := httptest.NewServer(s)
		defer testServer.Close()

		Convey("When HTTP status OK", func() {
			b.On("DeleteAccountLink", mock.Anything, twitchUserID, gameID).Return(&backend.GameAccountLink{
				UserID: twitchUserID,
				GameID: gameID,
				GameAccount: backend.GameAccount{
					AccountID:   accountID,
					DisplayName: displayName,
				},
			}, http.StatusOK, nil)

			client := &http.Client{}
			req, err := http.NewRequest("DELETE", fmt.Sprintf("%v/api/v2/users/%s/link/%s", testServer.URL, twitchUserID, gameID), nil)
			resp, err := client.Do(req)

			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			deleteAccountLinkResponse := samus_gateway.DeleteAccountLinkResponse{}
			ConvertToResponseInterface(resp, &deleteAccountLinkResponse)

			accountLink := deleteAccountLinkResponse.Deleted
			So(accountLink, ShouldNotBeNil)

			So(accountLink.UserID, ShouldEqual, twitchUserID)

			gameAccount := accountLink.GameAccount
			So(gameAccount, ShouldNotBeNil)

			So(gameAccount.DisplayName, ShouldEqual, displayName)
			So(gameAccount.AccountID, ShouldEqual, accountID)
		})

		Convey("When HTTP status Not Found", func() {
			b.On("DeleteAccountLink", mock.Anything, twitchUserID, gameID).Return(nil, http.StatusNotFound, nil)

			client := &http.Client{}
			req, err := http.NewRequest("DELETE", fmt.Sprintf("%v/api/v2/users/%s/link/%s", testServer.URL, twitchUserID, gameID), nil)
			resp, err := client.Do(req)

			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusNotFound)
		})

		Convey("When HTTP status Error", func() {
			b.On("DeleteAccountLink", mock.Anything, twitchUserID, gameID).Return(nil, http.StatusInternalServerError, errors.New("An error occurred"))

			client := &http.Client{}
			req, err := http.NewRequest("DELETE", fmt.Sprintf("%v/api/v2/users/%s/link/%s", testServer.URL, twitchUserID, gameID), nil)
			resp, err := client.Do(req)

			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusInternalServerError)
		})
	})
}
