package api

import (
	"encoding/json"
	"fmt"
	"net/http"
	"strconv"
	"time"

	"goji.io"
	"goji.io/pat"

	"code.justin.tv/common/golibs/errorlogger"
	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/samus/gateway/api/apidef"
	"code.justin.tv/samus/gateway/backend"
	config "code.justin.tv/samus/gateway/configuration"
	"code.justin.tv/samus/gateway/decorators"
	"code.justin.tv/samus/gateway/metrics"
	"code.justin.tv/samus/gateway/middleware"
	"github.com/cactus/go-statsd-client/statsd"
	log "github.com/sirupsen/logrus"
	"golang.org/x/net/context"
)

// Server struct
type Server struct {
	*goji.Mux
	Backend     backend.Backender
	Stats       statsd.Statter
	ErrorLogger errorlogger.ErrorLogger
}

const StatsEndpointKey = "endpoint"
const StatsEndpointStatusCodeFilter = "endpoints.%s.status_code.%d"

// NewServer Initializes server with handler functions
func NewServer(stats statsd.Statter, errorLogger errorlogger.ErrorLogger, backend backend.Backender, samusConfigs config.SamusConfigurations) (*Server, error) {

	server := twitchhttp.NewServer()
	log.Info("Creating Handlers")
	s := &Server{
		server,
		backend,
		stats,
		errorLogger,
	}

	metricsClient, err := metrics.NewMetricsClient(samusConfigs.MetricsConfig)
	var handlerMetrics *middleware.HandlerMetrics

	if err != nil {
		log.WithError(err)
		handlerMetrics = middleware.NewHandlerMetrics(nil)
	} else {
		handlerMetrics = middleware.NewHandlerMetrics(metricsClient)
	}

	// middleware layers
	s.Use(middleware.AccessLogger)
	s.UseC(handlerMetrics.Metrics)
	s.UseC(middleware.Recoverer)

	s.HandleC(pat.Get("/healthcheck"), decorators.DecorateAndAdapt(s.createHandler(s.HealthCheck, "health_check"), apidef.HealthCheck))

	s.HandleC(pat.Get("/samus/users/:userId/status"), decorators.DecorateAndAdapt(s.createHandler(s.PrimeStatus, "prime_status"), apidef.PrimeStatus))
	s.HandleC(pat.Get("/api/users/:userId/samus/status"), decorators.DecorateAndAdapt(s.createHandler(s.PrimeStatus, "prime_status"), apidef.PrimeStatus))

	s.HandleC(pat.Get("/samus/users/:userId/subscription_credit/balance"), decorators.DecorateAndAdapt(s.createHandler(s.SubCreditBalance, "sub_credit_balance"), apidef.SubCreditBalance))
	s.HandleC(pat.Get("/api/users/:userId/samus/balance"), decorators.DecorateAndAdapt(s.createHandler(s.SubCreditBalance, "sub_credit_balance"), apidef.SubCreditBalance))

	s.HandleC(pat.Put("/samus/users/:userId/subscription_credit/spend/order/:orderId"), decorators.DecorateAndAdapt(s.createHandler(s.SubCreditSpend, "sub_credit_spend"), apidef.SubCreditSpend))

	//"spend_samus_credit" API migrated from web/web
	s.HandleC(pat.Put("/samus/users/:userId/subscription_credit/spend/:broadcasterId"), decorators.DecorateAndAdapt(s.createHandler(s.SpendSubscriptionCredit, "spend_subscription_credit"), apidef.SpendSubscriptionCredit))
	s.HandleC(pat.Put("/api/users/:userId/subscription_credit/spend/:broadcasterId"), decorators.DecorateAndAdapt(s.createHandler(s.SpendSubscriptionCredit, "spend_subscription_credit"), apidef.SpendSubscriptionCredit))
	s.HandleC(pat.Delete("/samus/users/:userId/subscription_credit/spend/order/:broadcasterId"), decorators.DecorateAndAdapt(s.createHandler(s.UnspendSubscriptionCredit, "spend_subscription_credit"), apidef.UnspendSubscriptionCredit))

	s.HandleC(pat.Get("/api/users/:userId/prime/settings"), decorators.DecorateAndAdapt(s.createHandler(s.GetPrimeSettings, "get_prime_settings"), apidef.GetPrimeSettings))
	s.HandleC(pat.Post("/api/users/:userId/prime/settings"), decorators.DecorateAndAdapt(s.createHandler(s.SetPrimeSettings, "set_prime_settings"), apidef.SetPrimeSettings))

	s.HandleC(pat.Get("/api/string"), decorators.DecorateAndAdapt(s.createHandler(s.GetDynamicStrings, "get_dynamic_strings"), apidef.GetDynamicStrings))

	s.HandleC(pat.Get("/api/offers/blacklist"), decorators.DecorateAndAdapt(s.createHandler(s.GetOfferBlacklist, "get_offer_blacklist"), apidef.GetOfferBlacklist))
	s.HandleC(pat.Get("/api/prime/offers/eligibility"), decorators.DecorateAndAdapt(s.createHandler(s.GetCurrentOffersWithEligibility, "get_current_offers_with_eligibility"), apidef.GetCurrentOffersWithEligibility))
	s.HandleC(pat.Get("/api/prime/offers/:countryCode"), decorators.DecorateAndAdapt(s.createHandler(s.GetCurrentOffers, "get_current_offers"), apidef.GetCurrentOffers))
	s.HandleC(pat.Get("/api/prime/offers"), decorators.DecorateAndAdapt(s.createHandler(s.GetCurrentPrimeOffers, "get_current_prime_offers"), apidef.GetCurrentPrimeOffers))
	// TODO: once we confirm that the following path works, make it the default offers path by renaming it to "/api/prime/offers" (https://jira.agscollab.com/browse/TPDAM-1129)
	s.HandleC(pat.Get("/api/prime/offersv2"), decorators.DecorateAndAdapt(s.createHandler(s.GetCurrentOffersForUser, "get_current_prime_offers_for_user"), apidef.GetCurrentOffersForUser))
	s.HandleC(pat.Post("/api/users/:userId/prime/claim/offer/:offerId"), decorators.DecorateAndAdapt(s.createHandler(s.ClaimOffer, "claim_offer"), apidef.ClaimOffer))
	s.HandleC(pat.Post("/api/users/:userId/prime/place/order/:offerId"), decorators.DecorateAndAdapt(s.createHandler(s.PlaceOrder, "place_order"), apidef.PlaceOrder))
	s.HandleC(pat.Get("/api/users/:userId/prime/claim/offer/:offerId"), decorators.DecorateAndAdapt(s.createHandler(s.GetPrimeEntitlement, "get_prime_entitlement"), apidef.GetPrimeEntitlement))
	s.HandleC(pat.Get("/api/users/:userId/prime/orders"), decorators.DecorateAndAdapt(s.createHandler(s.GetOrdersByCustomer, "get_orders_by_customer"), apidef.GetOrdersByCustomer))
	s.HandleC(pat.Get("/api/users/:userId/prime/inventory"), decorators.DecorateAndAdapt(s.createHandler(s.ListInventory, "list_inventory"), apidef.ListInventory))
	s.HandleC(pat.Put("/api/users/:userId/prime/claim/offer/:offerId"), decorators.DecorateAndAdapt(s.createHandler(s.SetPrimeEntitlement, "set_prime_entitlement"), apidef.SetPrimeEntitlement))
	s.HandleC(pat.Post("/api/users/:userId/prime/unclaim/offer/:offerId"), decorators.DecorateAndAdapt(s.createHandler(s.ClearOfferClaimCodeForUser, "clear_offer_claim_code_for_user"), apidef.ClearOfferClaimCodeForUser))

	s.HandleC(pat.Get("/api/users/:userId/samus/credit"), decorators.DecorateAndAdapt(s.createHandler(s.CanSpendPrimeCredit, "can_spend_prime_credit"), apidef.CanSpendPrimeCredit))

	s.HandleC(pat.Post("/api/users/:userId/link"), decorators.DecorateAndAdapt(s.createHandler(s.CreateAccountLink, "create_account_link"), apidef.CreateAccountLink))
	s.HandleC(pat.Get("/api/users/:userId/link"), decorators.DecorateAndAdapt(s.createHandler(s.GetAccountLink, "get_account_link"), apidef.GetAccountLink))
	s.HandleC(pat.Delete("/api/users/:userId/link"), decorators.DecorateAndAdapt(s.createHandler(s.DeleteAccountLink, "delete_account_link"), apidef.DeleteAccountLink))

	s.HandleC(pat.Post("/api/users/:userId/link/:gameName"), decorators.DecorateAndAdapt(s.createHandler(s.CreateAccountLink, "create_account_link"), apidef.CreateAccountLink))
	s.HandleC(pat.Get("/api/users/:userId/link/:gameName"), decorators.DecorateAndAdapt(s.createHandler(s.GetAccountLink, "get_account_link"), apidef.GetAccountLink))
	s.HandleC(pat.Delete("/api/users/:userId/link/:gameName"), decorators.DecorateAndAdapt(s.createHandler(s.DeleteAccountLink, "delete_account_link"), apidef.DeleteAccountLink))

	s.HandleC(pat.Post("/api/v2/users/:userId/link/:vendorId"), decorators.DecorateAndAdapt(s.createHandler(s.CreateAccountLinkV2, "create_account_link_v2"), apidef.CreateAccountLinkV2))
	s.HandleC(pat.Get("/api/v2/users/:userId/link/:vendorId"), decorators.DecorateAndAdapt(s.createHandler(s.GetAccountLinkV2, "get_account_link_v2"), apidef.GetAccountLinkV2))
	s.HandleC(pat.Delete("/api/v2/users/:userId/link/:vendorId"), decorators.DecorateAndAdapt(s.createHandler(s.DeleteAccountLinkV2, "delete_account_link_v2"), apidef.DeleteAccountLinkV2))

	return s, nil
}

func (s *Server) createHandler(handler func(context.Context, http.ResponseWriter, *http.Request), statName string) func(context.Context, http.ResponseWriter, *http.Request) {
	return func(ctx context.Context, w http.ResponseWriter, r *http.Request) {
		start := time.Now()
		updatedCtx := context.WithValue(ctx, StatsEndpointKey, statName)
		handler(updatedCtx, w, r)
		duration := time.Since(start)
		err := s.Stats.Inc(fmt.Sprintf("endpoints.%s", statName), 1, 0.1)
		err = s.Stats.TimingDuration(fmt.Sprintf("endpoints.%s", statName), duration, 0.1)
		if err != nil {
		}
	}
}

func (s *Server) serveJSONError(ctx context.Context, w http.ResponseWriter, r *http.Request, status int, v interface{}, err error) {
	statsName := ctx.Value(StatsEndpointKey).(string)
	errr := s.Stats.Inc(fmt.Sprintf(StatsEndpointStatusCodeFilter, statsName, status), 1, 1)
	if errr != nil {
	}

	if s.ErrorLogger != nil && status > 200 {
		s.ErrorLogger.RequestError(r, err)
	}

	content, err := json.MarshalIndent(v, "", "  ")
	if err != nil {
		if s.ErrorLogger != nil {
			s.ErrorLogger.RequestError(r, err)
		}
		http.Error(w, err.Error(), http.StatusInternalServerError)
		err := s.Stats.Inc(fmt.Sprintf(StatsEndpointStatusCodeFilter, statsName, http.StatusInternalServerError), 1, 1)
		if err != nil {
		}
		return
	}
	w.Header().Set("Content-Length", strconv.Itoa(len(content)))
	w.Header().Set("Content-Type", "application/json")
	w.WriteHeader(status)
	_, err = w.Write(content)
	if err != nil {
		if s.ErrorLogger != nil {
			s.ErrorLogger.RequestError(r, err)
		}
		http.Error(w, err.Error(), http.StatusInternalServerError)
		log.WithError(err).Error("Error generating JSON Response: ", err.Error())
		err = s.Stats.Inc(fmt.Sprintf(StatsEndpointStatusCodeFilter, statsName, http.StatusInternalServerError), 1, 1)
		if err != nil {
		}
		return
	}
	err = s.Stats.Inc(fmt.Sprintf(StatsEndpointStatusCodeFilter, statsName, http.StatusOK), 1, 1)
	if err != nil {
	}
}

func (s *Server) serveJSON(ctx context.Context, w http.ResponseWriter, r *http.Request, v interface{}) {
	statsName := ctx.Value(StatsEndpointKey).(string)
	content, err := json.MarshalIndent(v, "", "  ")
	if err != nil {
		if s.ErrorLogger != nil {
			s.ErrorLogger.RequestError(r, err)
		}
		http.Error(w, err.Error(), http.StatusInternalServerError)
		err := s.Stats.Inc(fmt.Sprintf(StatsEndpointStatusCodeFilter, statsName, http.StatusInternalServerError), 1, 1)
		if err != nil {
		}
		return
	}
	w.Header().Set("Content-Length", strconv.Itoa(len(content)))
	w.Header().Set("Content-Type", "application/json")
	_, err = w.Write(content)
	if err != nil {
		if s.ErrorLogger != nil {
			s.ErrorLogger.RequestError(r, err)
		}
		http.Error(w, err.Error(), http.StatusInternalServerError)
		log.WithError(err).Error("Error generating JSON Response: ", err.Error())
		err = s.Stats.Inc(fmt.Sprintf(StatsEndpointStatusCodeFilter, statsName, http.StatusInternalServerError), 1, 1)
		if err != nil {
		}
		return
	}
	err = s.Stats.Inc(fmt.Sprintf(StatsEndpointStatusCodeFilter, statsName, http.StatusOK), 1, 1)
	if err != nil {
	}
}

// serveError vends a non-200 status error code and reports the error to Rollbar
func (s *Server) serveError(ctx context.Context, w http.ResponseWriter, r *http.Request, status int, err error) {
	statsName := ctx.Value(StatsEndpointKey).(string)
	if status == 0 {
		status = http.StatusInternalServerError
	}

	http.Error(w, err.Error(), status)
	errr := s.Stats.Inc(fmt.Sprintf(StatsEndpointStatusCodeFilter, statsName, status), 1, 1)
	if errr != nil {
	}

	if s.ErrorLogger != nil && status > 200 {
		s.ErrorLogger.RequestError(r, err)
	}
}
