package apidef

import (
	goji_middleware "goji.io/middleware"
	"golang.org/x/net/context"

	log "github.com/sirupsen/logrus"
)

// HandlerName is the name of a handler.
type HandlerName string

// NamedHandler is an interface representing an API handler with a name.
type NamedHandler interface {
	GetHandlerName() HandlerName
}

// Public APIs
const (
	HealthCheck                     HandlerName = "HealthCheck"
	PrimeStatus                     HandlerName = "PrimeStatus"
	SubCreditBalance                HandlerName = "SubCreditBalance"
	SubCreditSpend                  HandlerName = "SubCreditSpend"
	SubCreditUnspend                HandlerName = "SubCreditUnspend"
	GetPrimeSettings                HandlerName = "GetPrimeSettings"
	SetPrimeSettings                HandlerName = "SetPrimeSettings"
	GetDynamicStrings               HandlerName = "GetDynamicStrings"
	GetOfferBlacklist               HandlerName = "GetOfferBlacklist"
	GetCurrentOffers                HandlerName = "GetCurrentOffers"
	GetCurrentOffersForUser         HandlerName = "GetCurrentOffersForUser"
	GetCurrentPrimeOffers           HandlerName = "GetCurrentPrimeOffers"
	GetCurrentOffersWithEligibility HandlerName = "GetCurrentOffersWithEligibility"
	ClaimOffer                      HandlerName = "ClaimOffer"
	PlaceOrder                      HandlerName = "PlaceOrder"
	GetOrdersByCustomer             HandlerName = "GetOrdersByCustomer"
	ListInventory                   HandlerName = "ListInventory"
	GetPrimeEntitlement             HandlerName = "GetPrimeEntitlement"
	SetPrimeEntitlement             HandlerName = "SetPrimeEntitlement"
	ClearOfferClaimCodeForUser      HandlerName = "ClearOfferClaimCodeForUser"
	CanSpendPrimeCredit             HandlerName = "CanSpendPrimeCredit"
	CreateAccountLink               HandlerName = "CreateAccountLink"
	GetAccountLink                  HandlerName = "GetAccountLink"
	DeleteAccountLink               HandlerName = "DeleteAccountLink"
	SpendSubscriptionCredit         HandlerName = "SpendSubscriptionCredit"
	UnspendSubscriptionCredit       HandlerName = "UnspendSubscriptionCredit"
	GetAccountLinkV2                HandlerName = "GetAccountLinkV2"
	CreateAccountLinkV2             HandlerName = "CreateAccountLinkV2"
	DeleteAccountLinkV2             HandlerName = "DeleteAccountLinkV2"
)

// GetNamedHandler returns the NamedHandler that is handling the request with the specified context. This method will
// return nil if there is no handler associated with this request. This could happen if a bad url (e.g. /foobar) is
// requested
func GetNamedHandler(ctx context.Context) NamedHandler {
	handler := goji_middleware.Handler(ctx)
	if handler != nil {
		namedHandler, ok := handler.(NamedHandler)
		if !ok {
			log.Errorf("Handler is not of type NamedHandler. Pattern: %s", goji_middleware.Pattern(ctx))
			return nil
		}
		return namedHandler
	}
	return nil
}
