package api

import (
	"net/http"

	"encoding/json"

	"github.com/pkg/errors"

	"code.justin.tv/samus/gateway/backend"
	. "code.justin.tv/samus/gateway/util"
	log "github.com/sirupsen/logrus"
	"goji.io/pat"
	"golang.org/x/net/context"
)

// SubCreditSpendInput hold format of spend credit api body
type SubCreditSpendInput struct {
	ProductID string   `json:"productId"`
	Channels  []string `json:"channels"`
}

func (s *Server) SubCreditBalance(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	userID := pat.Param(ctx, "userId")
	validationErrMsg := ValidateTuid(userID)
	if validationErrMsg != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, validationErrMsg)
		return
	}
	log.Debug("Checking Credit Balance for userId : ", userID)

	resp, status, err := s.Backend.Balance(ctx, userID)
	if err != nil {
		s.serveError(ctx, w, r, status, err)
		return
	}
	log.Debug("Credit Balance for userId :", userID+" is", resp.CreditBalance)
	s.serveJSON(ctx, w, r, resp)
}

func (s *Server) SubCreditSpend(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	userID := pat.Param(ctx, "userId")
	validationErrMsg := ValidateTuid(userID)
	if validationErrMsg != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, validationErrMsg)
		return
	}
	orderID := pat.Param(ctx, "orderId")
	log.Debug("Spending Credit for UserId: ", userID, " OrderId: ", orderID)

	//Validate client request body
	var subCreditSpendInputBody SubCreditSpendInput
	err := json.NewDecoder(r.Body).Decode(&subCreditSpendInputBody)
	if err != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, err)
		return
	}

	//Generate request body
	buf, err := json.Marshal(subCreditSpendInputBody)
	if err != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, err)
	}
	log.Debug("Request Body : ", string(buf[:]))
	resp, status, err := s.Backend.SubCreditSpend(ctx, userID, orderID, buf)
	if err != nil {
		s.serveError(ctx, w, r, status, err)
		return
	}
	s.serveJSON(ctx, w, r, resp)
}

func (s *Server) CanSpendPrimeCredit(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	userID := pat.Param(ctx, "userId")
	validationErrMsg := ValidateTuid(userID)
	if validationErrMsg != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, validationErrMsg)
		return
	}
	productID := r.URL.Query().Get("productId")
	if !(len(productID) > 0) {
		err := errors.New("productId is a required query parameter")
		s.serveError(ctx, w, r, http.StatusBadRequest, err)
		return
	}

	if isProductBlacklisted(productID) {
		s.serveJSON(ctx, w, r, &backend.CanSpendPrimeCreditResponse{
			CanSpendPrimeCredit: false,
			WillRenew:           false,
			RenewalDate:         ""})
		return
	}

	resp, status, err := s.Backend.CanSpendPrimeCredit(ctx, userID, productID)
	if err != nil {
		s.serveError(ctx, w, r, status, err)
		return
	}
	s.serveJSON(ctx, w, r, resp)
}

func isProductBlacklisted(productID string) bool {
	var blacklistedProductMap = map[string]bool{
		"1189177": true, // OWL 2019
	}

	return blacklistedProductMap[productID]
}
