package api

import (
	"net/http"
	"testing"

	"fmt"

	"net/http/httptest"

	"strings"

	"code.justin.tv/common/config"
	"code.justin.tv/samus/gateway/backend"
	samus_config "code.justin.tv/samus/gateway/configuration"
	"code.justin.tv/samus/gateway/metrics"
	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

const validCreditUserID = "11111"
const invalidCreditUserID = "123HAX"
const validOrderID, validBroadcasterID = "1234", "1234"
const invalidOrderID, invalidBroadcasterID = "456ORZ", "456ORZ"
const productID = "coolProduct"
const TransactionID = "amzn1.monad3.1468959699787.1.3.Samus.1607122017dHoRxmHeTGSVZ1J/drm9xA.RA3n3IeeS8KGCR6wWuQS9Q.2"

// TestSubCreditBalance for subCreditBalance stress test
func TestSubCreditBalance(t *testing.T) {

	Convey("Credit Balance API ", t, func() {
		log.SetLevel(log.DebugLevel)
		b := &backend.BackendMock{}
		metricsConf := &metrics.MetricsConfig{
			Environment:   "UnitTest",
			MetricsRegion: "PDX",
		}
		configs := samus_config.SamusConfigurations{MetricsConfig: metricsConf}
		s, err := NewServer(config.Statsd(), config.RollbarErrorLogger(), b, configs)
		So(err, ShouldBeNil)

		testServer := httptest.NewServer(s)
		defer testServer.Close()

		Convey("When Balance = 1", func() {

			b.On("Balance", mock.Anything, validCreditUserID).Return(&backend.BalanceResponse{CreditBalance: 1}, http.StatusOK, nil)

			resp, err := http.Get(fmt.Sprintf("%v/samus/users/%s/subscription_credit/balance", testServer.URL, validCreditUserID))
			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			balanceResponse := backend.BalanceResponse{}
			ConvertToResponseInterface(resp, &balanceResponse)

			So(balanceResponse.CreditBalance, ShouldEqual, 1)
		})

		Convey("When userID is non-numeric, status 400 bad request", func() {

			resp, err := http.Get(fmt.Sprintf("%v/samus/users/%s/subscription_credit/balance", testServer.URL, invalidCreditUserID))
			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusBadRequest)

			balanceResponse := backend.BalanceResponse{}
			ConvertToResponseInterface(resp, &balanceResponse)

			So(balanceResponse.CreditBalance, ShouldEqual, 0)
		})
	})
}

// TestSubCreditSpend for subCreditSpend stress test
func TestSubCreditSpend(t *testing.T) {

	Convey("Spend Credit API", t, func() {
		log.SetLevel(log.DebugLevel)
		b := &backend.BackendMock{}
		metricsConf := &metrics.MetricsConfig{
			Environment:   "UnitTest",
			MetricsRegion: "PDX",
		}
		configs := samus_config.SamusConfigurations{MetricsConfig: metricsConf}
		s, err := NewServer(config.Statsd(), config.RollbarErrorLogger(), b, configs)
		So(err, ShouldBeNil)

		testServer := httptest.NewServer(s)
		defer testServer.Close()

		Convey("When HTTP status OK", func() {
			b.On("SubCreditSpend", mock.Anything, validCreditUserID, validOrderID, mock.Anything).Return(&backend.SubCreditSpendResponse{CreditBalance: 0, UserID: validCreditUserID, TransactionID: TransactionID}, http.StatusOK, nil)

			client := &http.Client{}
			req, err := http.NewRequest("PUT", fmt.Sprintf("%v/samus/users/%s/subscription_credit/spend/order/%s", testServer.URL, validCreditUserID, validOrderID), strings.NewReader("{\"productId\":\"SomeProductId\",\"channels\":[\"channel1\",\"channel2\"]}"))
			resp, err := client.Do(req)

			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)

			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			subCreditSpendResponse := backend.SubCreditSpendResponse{}
			ConvertToResponseInterface(resp, &subCreditSpendResponse)

			So(subCreditSpendResponse.CreditBalance, ShouldEqual, 0)
			So(subCreditSpendResponse.UserID, ShouldEqual, validCreditUserID)
			So(subCreditSpendResponse.TransactionID, ShouldEqual, TransactionID)
		})

		Convey("When userID is non-numeric, status 400 bad request", func() {

			client := &http.Client{}
			req, err := http.NewRequest("PUT", fmt.Sprintf("%v/samus/users/%s/subscription_credit/spend/order/%s", testServer.URL, invalidCreditUserID, validOrderID),
				strings.NewReader("<golang> really </golang>"))
			resp, err := client.Do(req)

			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusBadRequest)
		})

		Convey("When broadcasterID/orderID is non-numeric, status 400 bad request", func() {

			client := &http.Client{}
			req, err := http.NewRequest("PUT", fmt.Sprintf("%v/samus/users/%s/subscription_credit/spend/order/%s", testServer.URL, validCreditUserID, invalidOrderID),
				strings.NewReader("<golang> really </golang>"))
			resp, err := client.Do(req)

			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusBadRequest)
		})
	})
}

// TestSubCreditUnspend for SubCreditUnspend stress test
func TestSubCreditUnspend(t *testing.T) {
	Convey("Unspend API", t, func() {
		log.SetLevel(log.DebugLevel)
		b := &backend.BackendMock{}
		metricsConf := &metrics.MetricsConfig{
			Environment:   "UnitTest",
			MetricsRegion: "PDX",
		}
		configs := samus_config.SamusConfigurations{MetricsConfig: metricsConf}
		s, err := NewServer(config.Statsd(), config.RollbarErrorLogger(), b, configs)
		So(err, ShouldBeNil)

		testServer := httptest.NewServer(s)
		defer testServer.Close()

		Convey("When HTTP status OK", func() {
			testTransactionID := "TEST_TRANSACTION_ID"
			b.On("UnspendSubscriptionCredit", mock.Anything, validCreditUserID, mock.Anything).Return(&backend.UnspendSubscriptionCreditResponse{TransactionID: testTransactionID}, nil)

			client := &http.Client{}
			req, err := http.NewRequest("DELETE", fmt.Sprintf("%v/samus/users/%s/subscription_credit/spend/order/%s", testServer.URL, validCreditUserID, validOrderID), nil)
			resp, err := client.Do(req)

			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			unspendCreditResponse := backend.UnspendSubscriptionCreditResponse{}
			ConvertToResponseInterface(resp, &unspendCreditResponse)

			So(unspendCreditResponse.TransactionID, ShouldEqual, testTransactionID)
		})

		Convey("When userID is non-numeric, status 400 bad request", func() {

			client := &http.Client{}
			req, err := http.NewRequest("DELETE", fmt.Sprintf("%v/samus/users/%s/subscription_credit/spend/order/%s", testServer.URL, invalidCreditUserID, validOrderID), nil)
			resp, err := client.Do(req)

			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusBadRequest)
		})

		Convey("When broadcasterID/orderID is non-numeric, status 400 bad request", func() {

			client := &http.Client{}
			req, err := http.NewRequest("DELETE", fmt.Sprintf("%v/samus/users/%s/subscription_credit/spend/order/%s", testServer.URL, validCreditUserID, invalidOrderID), nil)
			resp, err := client.Do(req)

			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusBadRequest)
		})
	})
}

// TestSubCreditCanSpendPrimeCredit for new CanSpendPrimeCredit API logic
func TestSubCreditCanSpendPrimeCredit(t *testing.T) {
	Convey("CanSpendPrimeCredit API", t, func() {
		log.SetLevel(log.DebugLevel)
		b := &backend.BackendMock{}
		metricsConf := &metrics.MetricsConfig{
			Environment:   "UnitTest",
			MetricsRegion: "PDX",
		}
		configs := samus_config.SamusConfigurations{MetricsConfig: metricsConf}
		s, err := NewServer(config.Statsd(), config.RollbarErrorLogger(), b, configs)
		So(err, ShouldBeNil)

		testServer := httptest.NewServer(s)
		defer testServer.Close()

		Convey("When HTTP status OK", func() {
			b.On("CanSpendPrimeCredit", mock.Anything, validCreditUserID, productID).Return(&backend.CanSpendPrimeCreditResponse{CanSpendPrimeCredit: false, WillRenew: false}, http.StatusOK, nil)

			resp, err := http.Get(fmt.Sprintf("%v/api/users/%s/samus/credit?productId=%s", testServer.URL, validCreditUserID, productID))
			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			canSpendPrimeCreditResponse := backend.CanSpendPrimeCreditResponse{}
			ConvertToResponseInterface(resp, &canSpendPrimeCreditResponse)

			So(canSpendPrimeCreditResponse.CanSpendPrimeCredit, ShouldEqual, false)
		})

		Convey("When productId is missing", func() {
			resp, err := http.Get(fmt.Sprintf("%v/api/users/%s/samus/credit", testServer.URL, validCreditUserID))
			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusBadRequest)
		})

		Convey("When userID is non-numeric, status 400 bad request", func() {

			resp, err := http.Get(fmt.Sprintf("%v/api/users/%s/samus/credit?productId=%s", testServer.URL, invalidCreditUserID, productID))
			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusBadRequest)

			canSpendPrimeCreditResponse := backend.CanSpendPrimeCreditResponse{}
			ConvertToResponseInterface(resp, &canSpendPrimeCreditResponse)

			So(canSpendPrimeCreditResponse.CanSpendPrimeCredit, ShouldEqual, false)
		})
	})
}

func TestSpendSubscriptionCredit(t *testing.T) {
	Convey("api/credit:SpendSubscriptionCredit", t, func() {
		log.SetLevel(log.DebugLevel)

		b := &backend.BackendMock{}

		metricsConf := &metrics.MetricsConfig{
			Environment:   "UnitTest",
			MetricsRegion: "PDX",
		}

		client := &http.Client{}

		configs := samus_config.SamusConfigurations{MetricsConfig: metricsConf}

		s, err := NewServer(config.Statsd(), config.RollbarErrorLogger(), b, configs)

		So(err, ShouldBeNil)

		testServer := httptest.NewServer(s)

		defer testServer.Close()

		Convey("When HTTP status OK", func() {

			b.On("SpendSubscriptionCredit", mock.Anything, validCreditUserID, validBroadcasterID).Return(&backend.SpendSubscriptionCreditResponse{
				UserID:                    validCreditUserID,
				BroadcasterID:             validBroadcasterID,
				SubscriptionCreditBalance: 0,
			}, nil)

			req, err := http.NewRequest("PUT", fmt.Sprintf("%v/samus/users/%s/subscription_credit/spend/%s", testServer.URL, validCreditUserID, validBroadcasterID), strings.NewReader("{\"metadata\":null}"))

			// Critical
			resp, err := client.Do(req)

			So(err, ShouldBeNil)

			So(resp.StatusCode, ShouldEqual, http.StatusOK)
		})

		Convey("When userID is non-numeric, status 400 bad request", func() {
			req, err := http.NewRequest("PUT", fmt.Sprintf("%v/samus/users/%s/subscription_credit/spend/%s", testServer.URL, invalidCreditUserID, validBroadcasterID), strings.NewReader("{\"metadata\":null}"))

			// Critical
			resp, err := client.Do(req)

			So(err, ShouldBeNil)

			So(resp.StatusCode, ShouldEqual, http.StatusBadRequest)
		})

		Convey("When broadcasterID/orderID is non-numeric, status 400 bad request", func() {
			req, err := http.NewRequest("PUT", fmt.Sprintf("%v/samus/users/%s/subscription_credit/spend/%s", testServer.URL, validCreditUserID, invalidBroadcasterID), strings.NewReader("{\"metadata\":null}"))

			// Critical
			resp, err := client.Do(req)

			So(err, ShouldBeNil)

			So(resp.StatusCode, ShouldEqual, http.StatusBadRequest)
		})

		Convey("When user is already subscribed to a channel, status 409 conflict", func() {
			b.On("SpendSubscriptionCredit", mock.Anything, validCreditUserID, validBroadcasterID).Return(nil, errors.New("PAYMENT_CONFLICT"))

			req, err := http.NewRequest("PUT", fmt.Sprintf("%v/samus/users/%s/subscription_credit/spend/%s", testServer.URL, validCreditUserID, validBroadcasterID), strings.NewReader("{\"metadata\":null}"))

			// Critical
			resp, err := client.Do(req)

			So(err, ShouldBeNil)

			So(resp.StatusCode, ShouldEqual, http.StatusConflict)
		})
	})
}
