package api

import (
	"net/http"
	"testing"

	"fmt"

	"net/http/httptest"

	"code.justin.tv/common/config"
	"code.justin.tv/samus/gateway/backend"
	samus_config "code.justin.tv/samus/gateway/configuration"
	"code.justin.tv/samus/gateway/metrics"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestDynamicStrings(t *testing.T) {

	Convey("Dynamic Strings API", t, func() {
		log.SetLevel(log.DebugLevel)
		b := &backend.BackendMock{}
		metricsConf := &metrics.MetricsConfig{
			Environment:   "UnitTest",
			MetricsRegion: "PDX",
		}
		configs := samus_config.SamusConfigurations{MetricsConfig: metricsConf}
		s, err := NewServer(config.Statsd(), config.RollbarErrorLogger(), b, configs)
		So(err, ShouldBeNil)

		testServer := httptest.NewServer(s)
		defer testServer.Close()

		Convey("When getting string", func() {
			stringIds := "a,b"
			countryCode := "US"
			locale := "en"
			userIDStrings := "123456"
			dateOverride := "2017-10-01T00:00:00Z"

			stringMap := make(map[string]backend.DynamicString)
			stringA := backend.DynamicString{
				ID:             "a",
				Text:           "textA",
				ExternalUrl:    "externalUrl",
				IsExternalLink: true,
			}
			stringMap["a"] = stringA
			stringB := backend.DynamicString{
				ID:             "b",
				Text:           "textB",
				ExternalUrl:    "",
				IsExternalLink: false,
			}
			stringMap["b"] = stringB

			b.On("GetDynamicStrings", mock.Anything, stringIds, countryCode, locale, userIDStrings, dateOverride).Return(&backend.GetDynamicStringResponse{DynamicStringMap: stringMap}, http.StatusOK, nil)

			resp, err := http.Get(fmt.Sprintf("%v/api/string?stringIds=%s&countryCode=%s&locale=%s&userId=%s&dateOverride=%s", testServer.URL, stringIds, countryCode, locale, userIDStrings, dateOverride))
			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			getDynamicStringResponse := backend.GetDynamicStringResponse{}
			ConvertToResponseInterface(resp, &getDynamicStringResponse)

			So(getDynamicStringResponse.DynamicStringMap["a"].ID, ShouldEqual, stringA.ID)
			So(getDynamicStringResponse.DynamicStringMap["a"].Text, ShouldEqual, stringA.Text)
			So(getDynamicStringResponse.DynamicStringMap["a"].IsExternalLink, ShouldEqual, stringA.IsExternalLink)
			So(getDynamicStringResponse.DynamicStringMap["a"].ExternalUrl, ShouldEqual, stringA.ExternalUrl)

			So(getDynamicStringResponse.DynamicStringMap["b"].ID, ShouldEqual, stringB.ID)
			So(getDynamicStringResponse.DynamicStringMap["b"].Text, ShouldEqual, stringB.Text)
			So(getDynamicStringResponse.DynamicStringMap["b"].IsExternalLink, ShouldEqual, stringB.IsExternalLink)
			So(getDynamicStringResponse.DynamicStringMap["b"].ExternalUrl, ShouldEqual, stringB.ExternalUrl)
		})
	})
}
