package api

import (
	"net/http"
	"strconv"
	"strings"

	"encoding/json"
	"fmt"

	samus_gateway "code.justin.tv/samus/gateway/client"
	. "code.justin.tv/samus/gateway/util"
	log "github.com/sirupsen/logrus"
	"goji.io/pat"
	"golang.org/x/net/context"
)

const ClientID = "TWITCH"

type ClaimOfferInput struct {
	Metadata map[string]string `json:"metadata"` //TODO: Update this to match what were expecting
}

type PlaceOrderInput struct {
	Metadata map[string]string `json:"metadata"`
}
type GetOrdersByCustomerInput struct {
	Metadata map[string]string `json:"metadata"`
}
type ListInventoryInput struct {
	Metadata map[string]string `json:"metadata"`
}

func (s *Server) GetOfferBlacklist(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	log.Info("Getting Offer Blacklist")

	resp, status, err := s.Backend.GetOfferBlacklist(ctx)
	if err != nil {
		s.serveError(ctx, w, r, status, err)
		return
	}

	s.serveJSON(ctx, w, r, resp)
}

func (s *Server) GetCurrentPrimeOffers(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	countryCode := r.URL.Query().Get("countryCode")
	locale := r.URL.Query().Get("locale")
	userID := r.URL.Query().Get("userId")

	// userId can be empty, but if not, validate
	if userID != "" {
		validationErrMsg := ValidateTuid(userID)
		if validationErrMsg != nil {
			s.serveError(ctx, w, r, http.StatusBadRequest, validationErrMsg)
			log.WithError(validationErrMsg).Error("GetCurrentPrimeOffers: error calling GetCurrentPrimeOffers", validationErrMsg)
			return
		}
	}

	dateOverride := r.URL.Query().Get("dateOverride")
	resp, status, err := s.Backend.GetCurrentOffers(ctx, userID, countryCode, locale, dateOverride, ClientID)
	if err != nil {
		s.serveError(ctx, w, r, status, err)
		log.WithError(err).Error("GetCurrentPrimeOffers: error calling GetCurrentOffers", err)
		return
	}

	s.serveJSON(ctx, w, r, resp)
}

// Delete this once visage code is updated
func (s *Server) GetCurrentOffers(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	countryCode := pat.Param(ctx, "countryCode")
	locale := r.URL.Query().Get("locale")
	userID := r.URL.Query().Get("userId")

	// userId can be empty, but if not, validate
	if userID != "" {
		validationErrMsg := ValidateTuid(userID)
		if validationErrMsg != nil {
			s.serveError(ctx, w, r, http.StatusBadRequest, validationErrMsg)
			log.WithError(validationErrMsg).Error("GetCurrentOffers: invalid tuid", validationErrMsg)
			return
		}
	}

	dateOverride := r.URL.Query().Get("dateOverride")

	logMessage := fmt.Sprintf("Getting offers for %s, locale: %s, tuid: %s, dateOverride: %s", countryCode, locale, userID, dateOverride)
	log.Info(logMessage)

	resp, status, err := s.Backend.GetCurrentOffers(ctx, userID, countryCode, locale, dateOverride, ClientID)
	if err != nil {
		s.serveError(ctx, w, r, status, err)
		log.WithError(err).Error("GetCurrentOffers: error calling GetCurrentOffers", err)
		return
	}

	s.serveJSON(ctx, w, r, resp)
}

func (s *Server) GetCurrentOffersForUser(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	countryCode := r.URL.Query().Get("countryCode")
	locale := r.URL.Query().Get("locale")
	userID := r.URL.Query().Get("userId")
	dateOverride := r.URL.Query().Get("dateOverride")

	// userId can be empty, but if not, validate
	if userID != "" {
		validationErrMsg := ValidateTuid(userID)
		if validationErrMsg != nil {
			s.serveError(ctx, w, r, http.StatusBadRequest, validationErrMsg)
			log.WithError(validationErrMsg).Error("GetCurrentOffersForUser: invalid tuid", validationErrMsg)
			return
		}
	}

	resp, status, err := s.Backend.GetCurrentOffersForUser(ctx, userID, countryCode, locale, dateOverride, ClientID)
	if err != nil {
		s.serveError(ctx, w, r, status, err)
		log.WithError(err).Error("GetCurrentOffersForUser: error calling GetCurrentOffersForUser", err)
		return
	}

	s.serveJSON(ctx, w, r, resp)
}

func (s *Server) GetCurrentOffersWithEligibility(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	countryCode := r.URL.Query().Get("countryCode")
	locale := r.URL.Query().Get("locale")
	userID := r.URL.Query().Get("userId")
	dateOverride := r.URL.Query().Get("dateOverride")

	// userId can be empty, but if not, validate
	if userID != "" {
		validationErrMsg := ValidateTuid(userID)
		if validationErrMsg != nil {
			s.serveError(ctx, w, r, http.StatusBadRequest, validationErrMsg)
			log.WithError(validationErrMsg).Error("GetCurrentOffersWithEligibility: invalid tuid", validationErrMsg)
			return
		}
	}

	resp, status, err := s.Backend.GetCurrentOffersWithEligibility(ctx, userID, countryCode, locale, dateOverride, ClientID)
	if err != nil {
		s.serveError(ctx, w, r, status, err)
		log.WithError(err).Error("GetCurrentOffersWithEligibility: error calling GetCurrentOffersWithEligibility", err)
		return
	}

	s.serveJSON(ctx, w, r, resp)
}

func (s *Server) ClaimOffer(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	userID := pat.Param(ctx, "userId")
	validationErrMsg := ValidateTuid(userID)
	if validationErrMsg != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, validationErrMsg)
		log.WithError(validationErrMsg).Error("ClaimOffer: invalid tuid", validationErrMsg)
		return
	}

	offerID := pat.Param(ctx, "offerId")
	validationErrMsg = ValidateOffer(offerID)
	if validationErrMsg != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, validationErrMsg)
		log.WithError(validationErrMsg).Error("ClaimOffer: invalid offerid", validationErrMsg)
		return
	}

	locale := r.URL.Query().Get("locale")
	logMessage := fmt.Sprintf("Claiming %s for user %s in locale %s", offerID, userID, locale)
	log.Info(logMessage)

	//Validate client request body
	var claimOfferInput ClaimOfferInput
	err := json.NewDecoder(r.Body).Decode(&claimOfferInput)
	if err != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, err)
		log.WithError(err).Error("ClaimOffer: invalid client request body input", err)
	}
	//Generate request body
	buf, err := json.Marshal(claimOfferInput)
	if err != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, err)
		log.WithError(err).Error("ClaimOffer: error generating request body", err)
	}
	log.Debug("Request Body : ", string(buf[:]))

	resp, status, err := s.Backend.ClaimOffer(ctx, userID, offerID, locale, buf)
	if err != nil {
		claimOfferError := samus_gateway.ErrorResponse{
			Code: err.Error(),
		}
		s.serveJSONError(ctx, w, r, status, claimOfferError, err)
		log.WithError(err).Error("ClaimOffer: error claiming offer", err)
		return
	}

	errorDynamo := s.Backend.UpdatePrimeEntitlement(ctx, userID, offerID, locale)
	if errorDynamo != nil {
		errorMsg := fmt.Sprintf("Ignoring: fail to store entitlement to Dynamo for, %v, %v", userID, errorDynamo)
		log.WithError(errorDynamo).Error(errorMsg)
		// TODO: add metric via cloudwatch.Put...
	}

	s.serveJSON(ctx, w, r, resp)
}

func (s *Server) PlaceOrder(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	offerID := pat.Param(ctx, "offerId")
	userID := pat.Param(ctx, "userId")
	validationErrMsg := ValidateTuid(userID)
	if validationErrMsg != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, validationErrMsg)
		log.WithError(validationErrMsg).Error("PlaceOrder: invalid tuid", validationErrMsg)
		return
	}

	var requestBody samus_gateway.PlaceOrderRequestBody
	err := json.NewDecoder(r.Body).Decode(&requestBody)
	if err != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, err)
		log.WithError(err).Error("PlaceOrder: error decoding request body", err)
	}

	resp, status, err := s.Backend.PlaceOrder(ctx, userID, offerID, requestBody.IdempotenceKey, requestBody.AttributionChannel, requestBody.DateOverride, ClientID)
	if err != nil {
		placeOrderError := samus_gateway.ErrorResponse{
			Code: err.Error(),
		}
		s.serveJSONError(ctx, w, r, status, placeOrderError, err)
		log.WithError(err).Error("PlaceOrder: error calling PlaceOrder", err)
		return
	}
	s.serveJSON(ctx, w, r, resp)
}

func (s *Server) GetOrdersByCustomer(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	userID := pat.Param(ctx, "userId")
	validationErrMsg := ValidateTuid(userID)
	if validationErrMsg != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, validationErrMsg)
		return
	}

	nextToken := r.URL.Query().Get("nextToken")
	pageSize := r.URL.Query().Get("pageSize")
	pageSizeInt, err := strconv.Atoi(pageSize)
	offerId := r.URL.Query().Get("offerId")
	orderId := r.URL.Query().Get("orderId")

	resp, status, err := s.Backend.GetOrdersByCustomer(ctx, userID, nextToken, pageSizeInt, offerId, orderId, ClientID)
	if err != nil {
		getOrdersByCustomerError := samus_gateway.ErrorResponse{
			Code: err.Error(),
		}
		s.serveJSONError(ctx, w, r, status, getOrdersByCustomerError, err)
		return
	}
	s.serveJSON(ctx, w, r, resp)
}

func (s *Server) ListInventory(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	userID := pat.Param(ctx, "userId")
	validationErrMsg := ValidateTuid(userID)
	if validationErrMsg != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, validationErrMsg)
		return
	}

	itemIds := r.URL.Query().Get("itemIds")
	itemIdsList := strings.Split(itemIds, ",")
	amazonCustomerId := r.URL.Query().Get("amazonCustomerId")
	entitlementAccountType := r.URL.Query().Get("entitlementAccountType")
	nextToken := r.URL.Query().Get("nextToken")
	maxResults := r.URL.Query().Get("maxResults")
	maxResultsInt, err := strconv.Atoi(maxResults)
	entitlementStatusFilters := r.URL.Query().Get("entitlementStatusFilters")
	entitlementStatusFiltersList := strings.Split(entitlementStatusFilters, ",")

	resp, status, err := s.Backend.ListInventory(ctx, userID, amazonCustomerId, entitlementAccountType, itemIdsList, nextToken, maxResultsInt, entitlementStatusFiltersList, ClientID)
	if err != nil {
		listInventoryError := samus_gateway.ErrorResponse{
			Code: err.Error(),
		}
		s.serveJSONError(ctx, w, r, status, listInventoryError, err)
		return
	}
	s.serveJSON(ctx, w, r, resp)
}

func (s *Server) GetPrimeEntitlement(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	userID := pat.Param(ctx, "userId")
	validationErrMsg := ValidateTuid(userID)
	if validationErrMsg != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, validationErrMsg)
		log.WithError(validationErrMsg).Error("GetPrimeEntitlement: invalid tuid", validationErrMsg)
		return
	}
	offerID := pat.Param(ctx, "offerId")
	locale := r.URL.Query().Get("locale")
	logMessage := fmt.Sprintf("GetPrimeEntitlement %s for user %s in locale %s", offerID, userID, locale)
	log.Info(logMessage)

	resp, status, err := s.Backend.GetPrimeEntitlement(ctx, userID, offerID, locale)
	if err != nil {
		s.serveError(ctx, w, r, status, err)
		log.WithError(err).Error("GetPrimeEntitlement: error calling GetPrimeEntitlement", err)
		return
	}

	s.serveJSON(ctx, w, r, resp)
}

func (s *Server) SetPrimeEntitlement(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	userID := pat.Param(ctx, "userId")
	validationErrMsg := ValidateTuid(userID)
	if validationErrMsg != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, validationErrMsg)
		log.WithError(validationErrMsg).Error("SetPrimeEntitlement: invalid tuid", validationErrMsg)
		return
	}
	offerID := pat.Param(ctx, "offerId")
	hasEntitlement := r.URL.Query().Get("hasEntitlement")
	logMessage := fmt.Sprintf("SetPrimeEntitlement %s for user %s hasEntitlement %s", offerID, userID, hasEntitlement)
	log.Info(logMessage)

	resp, status, err := s.Backend.SetPrimeEntitlement(ctx, userID, offerID, hasEntitlement)
	if err != nil {
		s.serveError(ctx, w, r, status, err)
		log.WithError(err).Error("SetPrimeEntitlement: error calling SetPrimeEntitlement", err)
		return
	}

	s.serveJSON(ctx, w, r, resp)
}

func (s *Server) ClearOfferClaimCodeForUser(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	userID := pat.Param(ctx, "userId")
	validationErrMsg := ValidateTuid(userID)
	if validationErrMsg != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, validationErrMsg)
		log.WithError(validationErrMsg).Error("SetPrimeEntitlement: invalid tuid", validationErrMsg)
		return
	}

	offerID := pat.Param(ctx, "offerId")

	queryParams := r.URL.Query()
	marketplaceID := queryParams.Get("marketplaceID")
	csAgent := queryParams.Get("csAgent")
	csContactID := queryParams.Get("csContactID")

	resp, status, err := s.Backend.ClearOfferClaimCodeForUser(ctx, userID, marketplaceID, offerID, csAgent, csContactID)
	if err != nil {
		log.Info(fmt.Sprintf("[ClearOfferClaimCodeForUser] error: %v", err))
		s.serveError(ctx, w, r, status, err)
		log.WithError(err).Error("ClearOfferClaimCodeForUser: error calling ClearOfferClaimCodeForUser", err)
		return
	}

	s.serveJSON(ctx, w, r, resp)
}
