package api

import (
	"net/http"
	"testing"

	"fmt"

	"net/http/httptest"

	"strings"

	"errors"

	"code.justin.tv/common/config"
	"code.justin.tv/samus/gateway/backend"
	samus_gateway "code.justin.tv/samus/gateway/client"
	"code.justin.tv/samus/gateway/clients"
	samus_config "code.justin.tv/samus/gateway/configuration"
	"code.justin.tv/samus/gateway/metrics"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

const countryCode = "US"
const validOffersUserID = "1234567213"
const invalidOffersUserID = "I'm an invalid TUID!"
const locale = "en"
const offerID = "84b44d9d-8153-75f2-00d9-46bfaa309d48"
const dateOverride = "2017-12-10T12:00:00Z"
const csAgent = "testCsAgent"
const csContactID = "testCsContactID"
const marketplaceID = "testMarketplaceID"

//TestPrimeSettings for PrimeSettings API validation
func TestOfferBlacklist(t *testing.T) {

	Convey("Get Offer Blacklist API", t, func() {
		log.SetLevel(log.DebugLevel)
		bConf := backend.BackendConf{
			Stats:                      config.Statsd(),
			SamusSWSHost:               "",
			SamusSWSCertPath:           "",
			RootCertPath:               "",
			RedisHostPort:              "",
			RailsHost:                  "",
			DynamoRegion:               "",
			EntitlementDynamoTableName: "",
			UserDynamoTableName:        "",
		}
		metricsConf := &metrics.MetricsConfig{
			Environment:   "UnitTest",
			MetricsRegion: "PDX",
		}
		configs := samus_config.SamusConfigurations{MetricsConfig: metricsConf}
		b, err := backend.NewBackend(&bConf)
		s, err := NewServer(config.Statsd(), config.RollbarErrorLogger(), b, configs)
		So(err, ShouldBeNil)

		testServer := httptest.NewServer(s)
		defer testServer.Close()

		Convey("When getting blacklist", func() {
			resp, err := http.Get(fmt.Sprintf("%v/api/offers/blacklist", testServer.URL))
			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			offerBlacklistResponse := backend.OfferBlacklistResponse{}
			ConvertToResponseInterface(resp, &offerBlacklistResponse)

			So(offerBlacklistResponse.Blacklist["9aadfcdd-da46-8d47-07f4-ee8885482da3"], ShouldContain, "tw")
		})
	})
}

func TestClaimOffer(t *testing.T) {

	Convey("Claim Offer API", t, func() {
		log.SetLevel(log.DebugLevel)
		b := &backend.BackendMock{}
		metricsConf := &metrics.MetricsConfig{
			Environment:   "UnitTest",
			MetricsRegion: "PDX",
		}
		configs := samus_config.SamusConfigurations{MetricsConfig: metricsConf}
		s, err := NewServer(config.Statsd(), config.RollbarErrorLogger(), b, configs)
		So(err, ShouldBeNil)

		testServer := httptest.NewServer(s)
		defer testServer.Close()

		Convey("When locale is en", func() {

			metadata := samus_gateway.OfferClaimMetadata{
				OfferID:   offerID,
				CodeLabel: "codeLabel",
				CodeValue: "codeValue",
			}
			claimOfferResponse := backend.PrimeEntitlementResponse{
				OfferID:            offerID,
				OfferClaimMetadata: metadata,
				ClaimMethod:        "CLAIM_CODE",
				OfferClaimData:     "offerClaimData",
				ClaimInstruction:   "claimInstruction",
			}

			b.On("ClaimOffer", mock.Anything, validOffersUserID, offerID, locale, mock.Anything).Return(&claimOfferResponse, http.StatusOK, nil)
			b.On("UpdatePrimeEntitlement", mock.Anything, validOffersUserID, offerID, locale).Return(nil)
			client := &http.Client{}
			req, err := http.NewRequest("POST", fmt.Sprintf("%v/api/users/%s/prime/claim/offer/%s?locale=en", testServer.URL, validOffersUserID, offerID), strings.NewReader("{\"metadata\":null}"))
			resp, err := client.Do(req)

			log.Info("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			response := backend.PrimeEntitlementResponse{}
			ConvertToResponseInterface(resp, &response)

			So(response, ShouldNotBeNil)
			So(response.OfferID, ShouldEqual, offerID)
		})

		Convey("When store entitlement to dynamo fails silent", func() {

			metadata := samus_gateway.OfferClaimMetadata{
				OfferID:   offerID,
				CodeLabel: "codeLabel",
				CodeValue: "codeValue",
			}
			claimOfferResponse := backend.PrimeEntitlementResponse{
				OfferID:            offerID,
				OfferClaimMetadata: metadata,
				ClaimMethod:        "CLAIM_CODE",
				OfferClaimData:     "offerClaimData",
				ClaimInstruction:   "claimInstruction",
			}

			b.On("ClaimOffer", mock.Anything, validOffersUserID, offerID, locale, mock.Anything).Return(&claimOfferResponse, http.StatusOK, nil)
			b.On("UpdatePrimeEntitlement", mock.Anything, validOffersUserID, offerID, locale).Return(errors.New("Failed to update prime entitlement"))
			client := &http.Client{}
			req, err := http.NewRequest("POST", fmt.Sprintf("%v/api/users/%s/prime/claim/offer/%s?locale=en", testServer.URL, validOffersUserID, offerID), strings.NewReader("{\"metadata\":null}"))
			resp, err := client.Do(req)

			log.Info("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			response := backend.PrimeEntitlementResponse{}
			ConvertToResponseInterface(resp, &response)

			So(response, ShouldNotBeNil)
			So(response.OfferID, ShouldEqual, offerID)
		})
	})
}

func TestGetCurrentOffers(t *testing.T) {

	Convey("Get Current Offers API", t, func() {
		log.SetLevel(log.DebugLevel)
		b := &backend.BackendMock{}
		metricsConf := &metrics.MetricsConfig{
			Environment:   "UnitTest",
			MetricsRegion: "PDX",
		}
		configs := samus_config.SamusConfigurations{MetricsConfig: metricsConf}
		s, err := NewServer(config.Statsd(), config.RollbarErrorLogger(), b, configs)
		So(err, ShouldBeNil)

		testServer := httptest.NewServer(s)
		defer testServer.Close()

		Convey("When countryCode is US with userID", func() {
			currentPrimeOffers := []samus_gateway.OfferWithNoClaimData{}
			currentPrimeOffer := samus_gateway.OfferWithNoClaimData{
				ContentCategories:       []string{},
				ExternalOfferUri:        "externalOfferUri",
				PublisherName:           "publisherName",
				ApplicableGame:          "applicableGame",
				ContentDeliveryMethod:   "claimCode",
				Assets:                  []samus_gateway.OfferAsset{},
				ContentClaimInstruction: "claimIntructions",
				Priority:                1,
				OfferDescription:        "offerDescription",
				OfferTitle:              "offerTitle",
				OfferID:                 offerID,
				EndTime:                 "endTime",
				StartTime:               "startTime",
			}
			currentPrimeOffers = append(currentPrimeOffers, currentPrimeOffer)
			currentOffersResponse := backend.CurrentOffersResponse{
				CurrentPrimeOffers: currentPrimeOffers,
			}

			b.On("GetCurrentOffers", mock.Anything, validOffersUserID, countryCode, locale, "", ClientID).Return(&currentOffersResponse, http.StatusOK, nil)

			resp, err := http.Get(fmt.Sprintf("%v/api/prime/offers/US?locale=en&userId=%v", testServer.URL, validOffersUserID))
			log.Info("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			currentOffersResponse2 := backend.CurrentOffersResponse{}
			ConvertToResponseInterface(resp, &currentOffersResponse2)

			So(currentOffersResponse2.CurrentPrimeOffers, ShouldNotBeNil)
			So(len(currentOffersResponse2.CurrentPrimeOffers), ShouldBeGreaterThan, 0)
		})

		Convey("When countryCode is US with empty validOffersUserID", func() {
			currentPrimeOffers := []samus_gateway.OfferWithNoClaimData{}
			currentPrimeOffer := samus_gateway.OfferWithNoClaimData{
				ContentCategories:       []string{},
				ExternalOfferUri:        "externalOfferUri",
				PublisherName:           "publisherName",
				ApplicableGame:          "applicableGame",
				ContentDeliveryMethod:   "claimCode",
				Assets:                  []samus_gateway.OfferAsset{},
				ContentClaimInstruction: "claimIntructions",
				Priority:                1,
				OfferDescription:        "offerDescription",
				OfferTitle:              "offerTitle",
				OfferID:                 offerID,
				EndTime:                 "endTime",
				StartTime:               "startTime",
			}
			currentPrimeOffers = append(currentPrimeOffers, currentPrimeOffer)
			currentOffersResponse := backend.CurrentOffersResponse{
				CurrentPrimeOffers: currentPrimeOffers,
			}

			b.On("GetCurrentOffers", mock.Anything, "", countryCode, locale, "", ClientID).Return(&currentOffersResponse, http.StatusOK, nil)

			resp, err := http.Get(fmt.Sprintf("%v/api/prime/offers/US?locale=en", testServer.URL))
			log.Info("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			currentOffersResponse2 := backend.CurrentOffersResponse{}
			ConvertToResponseInterface(resp, &currentOffersResponse2)

			So(currentOffersResponse2.CurrentPrimeOffers, ShouldNotBeNil)
			So(len(currentOffersResponse2.CurrentPrimeOffers), ShouldBeGreaterThan, 0)
		})

		Convey("When countryCode is US with userID and dateOverride", func() {
			currentPrimeOffers := []samus_gateway.OfferWithNoClaimData{}
			currentPrimeOffer := samus_gateway.OfferWithNoClaimData{
				ContentCategories:       []string{},
				ExternalOfferUri:        "externalOfferUri",
				PublisherName:           "publisherName",
				ApplicableGame:          "applicableGame",
				ContentDeliveryMethod:   "claimCode",
				Assets:                  []samus_gateway.OfferAsset{},
				ContentClaimInstruction: "claimIntructions",
				Priority:                1,
				OfferDescription:        "offerDescription",
				OfferTitle:              "offerTitle",
				OfferID:                 offerID,
				EndTime:                 "endTime",
				StartTime:               "startTime",
			}
			currentPrimeOffers = append(currentPrimeOffers, currentPrimeOffer)
			currentOffersResponse := backend.CurrentOffersResponse{
				CurrentPrimeOffers: currentPrimeOffers,
			}

			b.On("GetCurrentOffers", mock.Anything, validOffersUserID, countryCode, locale, dateOverride, ClientID).Return(&currentOffersResponse, http.StatusOK, nil)

			resp, err := http.Get(fmt.Sprintf("%v/api/prime/offers/US?locale=en&userId=%v&dateOverride=%s", testServer.URL, validOffersUserID, dateOverride))
			log.Info("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			currentOffersResponse2 := backend.CurrentOffersResponse{}
			ConvertToResponseInterface(resp, &currentOffersResponse2)

			So(currentOffersResponse2.CurrentPrimeOffers, ShouldNotBeNil)
			So(len(currentOffersResponse2.CurrentPrimeOffers), ShouldBeGreaterThan, 0)
		})
	})
}

func TestGetCurrentOffersForUser(t *testing.T) {

	Convey("Get Current Offers API", t, func() {
		log.SetLevel(log.DebugLevel)
		b := &backend.BackendMock{}
		metricsConf := &metrics.MetricsConfig{
			Environment:   "UnitTest",
			MetricsRegion: "PDX",
		}
		configs := samus_config.SamusConfigurations{MetricsConfig: metricsConf}
		s, err := NewServer(config.Statsd(), config.RollbarErrorLogger(), b, configs)
		So(err, ShouldBeNil)

		testServer := httptest.NewServer(s)
		defer testServer.Close()

		currentPrimeOffers := []samus_gateway.PrimeOffer{}
		currentPrimeOffer := samus_gateway.PrimeOffer{
			OfferWithClaimData: samus_gateway.OfferWithClaimData{
				ContentCategories:       []string{},
				ExternalOfferUri:        "externalOfferUri",
				PublisherName:           "publisherName",
				ApplicableGame:          "applicableGame",
				ContentDeliveryMethod:   "claimCode",
				Assets:                  []samus_gateway.OfferAsset{},
				ContentClaimInstruction: "claimIntructions",
				Priority:                1,
				OfferDescription:        "offerDescription",
				OfferTitle:              "offerTitle",
				OfferID:                 offerID,
				EndTime:                 "endTime",
				StartTime:               "startTime",
			},
			ClaimHint: "AVAILABLE",
		}
		currentPrimeOffers = append(currentPrimeOffers, currentPrimeOffer)
		currentOffersResponse := samus_gateway.CurrentPrimeOffersResponse{
			PrimeOffers: currentPrimeOffers,
		}

		b.On("GetCurrentOffersForUser", mock.Anything, mock.Anything, countryCode, locale, mock.Anything, ClientID).Return(&currentOffersResponse, http.StatusOK, nil)

		Convey("succeeds with the correct arguments when userID is valid", func() {
			resp, err := http.Get(fmt.Sprintf("%v/api/prime/offersv2?countryCode=%v&locale=%v&dateOverride=%v&userId=%v", testServer.URL, countryCode, locale, dateOverride, validOffersUserID))
			log.Info("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			// check that a call to the backend was made using the correct arguments (i.e. query parameters were parsed and forwarded)
			b.AssertCalled(t, "GetCurrentOffersForUser", mock.Anything, validOffersUserID, countryCode, locale, dateOverride, ClientID)

			currentOffersResponse2 := samus_gateway.CurrentPrimeOffersResponse{}
			ConvertToResponseInterface(resp, &currentOffersResponse2)

			So(currentOffersResponse2.PrimeOffers, ShouldNotBeNil)
			So(len(currentOffersResponse2.PrimeOffers), ShouldEqual, 1)
		})

		Convey("succeeds with the correct arguments when no userID is provided", func() {
			resp, err := http.Get(fmt.Sprintf("%v/api/prime/offersv2?countryCode=%v&locale=%v", testServer.URL, countryCode, locale))
			log.Info("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			// check that a call to the backend was made using the correct arguments (empty userID and date override)
			b.AssertCalled(t, "GetCurrentOffersForUser", mock.Anything, "", countryCode, locale, "", ClientID)

			currentOffersResponse2 := samus_gateway.CurrentPrimeOffersResponse{}
			ConvertToResponseInterface(resp, &currentOffersResponse2)

			So(currentOffersResponse2.PrimeOffers, ShouldNotBeNil)
			So(len(currentOffersResponse2.PrimeOffers), ShouldEqual, 1)
		})

		Convey("fails when userID is invalid", func() {
			resp, err := http.Get(fmt.Sprintf("%v/api/prime/offersv2?countryCode=%v&locale=%v&dateOverride=%v&userId=%v", testServer.URL, countryCode, locale, dateOverride, invalidOffersUserID))
			log.Info("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusBadRequest)

			// check that no call to the backend was made
			b.AssertNotCalled(t, "GetCurrentOffersForUser", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything)

			currentOffersResponse2 := samus_gateway.CurrentPrimeOffersResponse{}
			ConvertToResponseInterface(resp, &currentOffersResponse2)

			So(currentOffersResponse2.PrimeOffers, ShouldBeNil)
		})
	})
}

func TestGetPrimeEntitlement(t *testing.T) {

	Convey("Get Prime Entitlement API", t, func() {
		log.SetLevel(log.DebugLevel)
		b := &backend.BackendMock{}
		metricsConf := &metrics.MetricsConfig{
			Environment:   "UnitTest",
			MetricsRegion: "PDX",
		}
		configs := samus_config.SamusConfigurations{MetricsConfig: metricsConf}

		s, err := NewServer(config.Statsd(), config.RollbarErrorLogger(), b, configs)
		So(err, ShouldBeNil)

		testServer := httptest.NewServer(s)
		defer testServer.Close()

		Convey("When successful when locale is en_US ", func() {

			metadata := samus_gateway.OfferClaimMetadata{
				OfferID:   offerID,
				CodeLabel: "codeLabel",
				CodeValue: "codeValue",
			}
			entitlementResponse := backend.PrimeEntitlementResponse{
				OfferID:            offerID,
				OfferClaimMetadata: metadata,
				ClaimMethod:        "CLAIM_CODE",
				OfferClaimData:     "offerClaimData",
				ClaimInstruction:   "claimInstruction",
			}

			b.On("GetPrimeEntitlement", mock.Anything, validOffersUserID, offerID, locale, mock.Anything).Return(&entitlementResponse, http.StatusOK, nil)
			client := &http.Client{}
			req, err := http.NewRequest("GET", fmt.Sprintf("%v/api/users/%s/prime/claim/offer/%s?locale=en", testServer.URL, validOffersUserID, offerID), strings.NewReader("{\"metadata\":null}"))
			resp, err := client.Do(req)

			log.Info("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			response := backend.PrimeEntitlementResponse{}
			ConvertToResponseInterface(resp, &response)

			So(response, ShouldNotBeNil)
			So(response.OfferID, ShouldEqual, offerID)
		})

	})
}
func TestSetPrimeEntitlement(t *testing.T) {

	Convey("Set Prime Entitlement API", t, func() {
		log.SetLevel(log.DebugLevel)
		b := &backend.BackendMock{}
		metricsConf := &metrics.MetricsConfig{
			Environment:   "UnitTest",
			MetricsRegion: "PDX",
		}
		configs := samus_config.SamusConfigurations{MetricsConfig: metricsConf}

		s, err := NewServer(config.Statsd(), config.RollbarErrorLogger(), b, configs)
		So(err, ShouldBeNil)

		testServer := httptest.NewServer(s)
		defer testServer.Close()

		Convey("When successful when hasEntitlement false ", func() {

			entitlementResponse := backend.PrimeEntitlementResponse{
				OfferID:            offerID,
				OfferClaimMetadata: samus_gateway.OfferClaimMetadata{},
				ClaimMethod:        "",
				OfferClaimData:     "",
				ClaimInstruction:   "",
			}

			b.On("SetPrimeEntitlement", mock.Anything, validOffersUserID, offerID, "false").Return(&entitlementResponse, http.StatusOK, nil)
			client := &http.Client{}
			req, err := http.NewRequest("PUT", fmt.Sprintf("%v/api/users/%s/prime/claim/offer/%s?hasEntitlement=false", testServer.URL, validOffersUserID, offerID), strings.NewReader("{\"metadata\":null}"))
			resp, err := client.Do(req)

			log.Info("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			response := backend.PrimeEntitlementResponse{}
			ConvertToResponseInterface(resp, &response)

			So(response, ShouldNotBeNil)
			So(response.OfferID, ShouldEqual, offerID)
		})
	})
}

func TestClearOfferClaimCodeForUser(t *testing.T) {
	client := &http.Client{}

	Convey("TestClearOfferClaimCodeForUser API", t, func() {
		log.SetLevel(log.DebugLevel)
		b := &backend.BackendMock{}
		metricsConf := &metrics.MetricsConfig{
			Environment:   "UnitTest",
			MetricsRegion: "PDX",
		}
		configs := samus_config.SamusConfigurations{MetricsConfig: metricsConf}

		s, err := NewServer(config.Statsd(), config.RollbarErrorLogger(), b, configs)
		So(err, ShouldBeNil)

		testServer := httptest.NewServer(s)
		defer testServer.Close()

		Convey("When successfully cleared offer claim code", func() {
			clearOfferClaimCodeForuserResponse := clients.ClearOfferClaimCodeForUserResponse{
				Success: true,
			}

			b.On("ClearOfferClaimCodeForUser", mock.Anything, validOffersUserID, mock.Anything, offerID, mock.Anything, mock.Anything).Return(&clearOfferClaimCodeForuserResponse, http.StatusOK, nil)
			req, err := http.NewRequest("POST", fmt.Sprintf("%v/api/users/%s/prime/unclaim/offer/%s", testServer.URL, validOffersUserID, offerID), strings.NewReader("{\"metadata\":null}"))
			resp, err := client.Do(req)

			log.Info("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			response := clients.ClearOfferClaimCodeForUserResponse{}
			ConvertToResponseInterface(resp, &response)

			So(response, ShouldNotBeNil)
			So(response.Success, ShouldEqual, true)
		})

		Convey("When there was an internal error", func() {
			clearOfferClaimCodeForUserResponse := clients.ClearOfferClaimCodeForUserResponse{
				Success: true,
			}

			b.On("ClearOfferClaimCodeForUser", mock.Anything, validOffersUserID, mock.Anything, offerID, mock.Anything, mock.Anything).Return(&clearOfferClaimCodeForUserResponse, http.StatusOK, nil)
			req, err := http.NewRequest("POST", fmt.Sprintf("%v/api/users/%s/prime/unclaim/offer/%s", testServer.URL, validOffersUserID, offerID), strings.NewReader("{\"metadata\":null}"))
			resp, err := client.Do(req)

			log.Info("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			response := clients.ClearOfferClaimCodeForUserResponse{}
			ConvertToResponseInterface(resp, &response)

			So(response, ShouldNotBeNil)
			So(response.Success, ShouldEqual, true)
		})

		Convey("fails when TUID is invalid", func() {
			req, err := http.NewRequest("POST", fmt.Sprintf("%v/api/users/%s/prime/unclaim/offer/%s", testServer.URL, invalidOffersUserID, offerID), strings.NewReader("{\"metadata\":null}"))
			resp, err := client.Do(req)

			log.Info("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusBadRequest)

			// check that no call to the backend was made
			b.AssertNotCalled(t, "ClearOfferClaimCodeForUser", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything)
		})
	})
}

func TestGetCurrentOffersWithEligibility(t *testing.T) {

	Convey("Get Current Offers With Eligibility API", t, func() {
		log.SetLevel(log.DebugLevel)
		b := &backend.BackendMock{}
		metricsConf := &metrics.MetricsConfig{
			Environment:   "UnitTest",
			MetricsRegion: "PDX",
		}
		configs := samus_config.SamusConfigurations{MetricsConfig: metricsConf}
		s, err := NewServer(config.Statsd(), config.RollbarErrorLogger(), b, configs)
		So(err, ShouldBeNil)

		testServer := httptest.NewServer(s)
		defer testServer.Close()

		currentPrimeOffer := samus_gateway.OfferWithNoClaimData{
			ContentCategories:       []string{},
			ExternalOfferUri:        "externalOfferUri",
			PublisherName:           "publisherName",
			ApplicableGame:          "applicableGame",
			ContentDeliveryMethod:   "claimCode",
			Assets:                  []samus_gateway.OfferAsset{},
			ContentClaimInstruction: "claimIntructions",
			Priority:                1,
			OfferDescription:        "offerDescription",
			OfferTitle:              "offerTitle",
			OfferID:                 offerID,
			EndTime:                 "endTime",
			StartTime:               "startTime",
			Tags:                    samus_gateway.OfferTag{},
		}

		Convey("GetCurrentOffersWithEligibility: Authenticated Twitch user", func() {
			expectedResp := samus_gateway.GetCurrentOffersWithEligibilityResponse{
				CurrentOffers: []samus_gateway.OfferWithEligibility{
					samus_gateway.OfferWithEligibility{
						OfferWithNoClaimData: currentPrimeOffer,
						OfferEligibility: samus_gateway.OfferEligibility{
							OfferState:        samus_gateway.OfferLive,
							CanClaim:          true,
							PrimeGamingRule:   true,
							LinkedAccountRule: true,
						},
					},
				},
			}

			b.On("GetCurrentOffersWithEligibility", mock.Anything, mock.Anything, countryCode, locale, mock.Anything, ClientID).Return(&expectedResp, http.StatusOK, nil)

			resp, err := http.Get(fmt.Sprintf("%v/api/prime/offers/eligibility?countryCode=%v&locale=%v&dateOverride=%v&userId=%v", testServer.URL, countryCode, locale, dateOverride, validOffersUserID))
			log.Info("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			// check that a call to the backend was made using the correct arguments (i.e. query parameters were parsed and forwarded)
			b.AssertCalled(t, "GetCurrentOffersWithEligibility", mock.Anything, validOffersUserID, countryCode, locale, dateOverride, ClientID)

			actualResp := samus_gateway.GetCurrentOffersWithEligibilityResponse{}
			ConvertToResponseInterface(resp, &actualResp)

			So(actualResp.CurrentOffers, ShouldNotBeNil)
			So(len(actualResp.CurrentOffers), ShouldEqual, 1)
			So(actualResp.CurrentOffers[0].OfferID, ShouldEqual, expectedResp.CurrentOffers[0].OfferID)
			So(actualResp.CurrentOffers[0].OfferEligibility, ShouldNotBeNil)
			So(actualResp.CurrentOffers[0].OfferEligibility.OfferState, ShouldEqual, expectedResp.CurrentOffers[0].OfferEligibility.OfferState)
		})

		Convey("GetCurrentOffersWithEligibility: Unauthenticated Twitch user", func() {
			expectedResp := samus_gateway.GetCurrentOffersWithEligibilityResponse{
				CurrentOffers: []samus_gateway.OfferWithEligibility{
					samus_gateway.OfferWithEligibility{
						OfferWithNoClaimData: currentPrimeOffer,
						OfferEligibility: samus_gateway.OfferEligibility{
							CanClaim: false,
						},
					},
				},
			}

			b.On("GetCurrentOffersWithEligibility", mock.Anything, mock.Anything, countryCode, locale, mock.Anything, ClientID).Return(&expectedResp, http.StatusOK, nil)

			resp, err := http.Get(fmt.Sprintf("%v/api/prime/offers/eligibility?countryCode=%v&locale=%v&dateOverride=%v", testServer.URL, countryCode, locale, dateOverride))
			log.Info("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			// check that a call to the backend was made using the correct arguments (i.e. query parameters were parsed and forwarded)
			b.AssertCalled(t, "GetCurrentOffersWithEligibility", mock.Anything, "", countryCode, locale, dateOverride, ClientID)

			actualResp := samus_gateway.GetCurrentOffersWithEligibilityResponse{}
			ConvertToResponseInterface(resp, &actualResp)

			So(actualResp.CurrentOffers, ShouldNotBeNil)
			So(len(actualResp.CurrentOffers), ShouldEqual, 1)
			So(actualResp.CurrentOffers[0].OfferID, ShouldEqual, expectedResp.CurrentOffers[0].OfferID)
			So(actualResp.CurrentOffers[0].OfferEligibility, ShouldNotBeNil)
			So(actualResp.CurrentOffers[0].OfferEligibility.CanClaim, ShouldEqual, expectedResp.CurrentOffers[0].OfferEligibility.CanClaim)
		})

		Convey("GetCurrentOffersWithEligibility: Fails when userID is invalid", func() {
			resp, err := http.Get(fmt.Sprintf("%v/api/prime/offers/eligibility?countryCode=%v&locale=%v&dateOverride=%v&userId=%v", testServer.URL, countryCode, locale, dateOverride, invalidOffersUserID))
			log.Info("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusBadRequest)

			// check that no call to the backend was made
			b.AssertNotCalled(t, "GetCurrentOffersWithEligibility", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything)

			actualResp := samus_gateway.GetCurrentOffersWithEligibilityResponse{}
			ConvertToResponseInterface(resp, &actualResp)

			So(actualResp.CurrentOffers, ShouldBeNil)
		})
	})
}
