package api

import (
	"net/http"
	"testing"

	"fmt"

	"net/http/httptest"

	"encoding/json"
	"io/ioutil"

	"errors"

	"code.justin.tv/common/config"
	"code.justin.tv/samus/gateway/backend"
	samus_config "code.justin.tv/samus/gateway/configuration"
	"code.justin.tv/samus/gateway/metrics"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

//TestPrimeStatus for PrimeStatus API validation
func TestPrimeStatus(t *testing.T) {

	Convey("Prime Status API", t, func() {
		log.SetLevel(log.DebugLevel)
		b := &backend.BackendMock{}
		metricsConf := &metrics.MetricsConfig{
			Environment:   "UnitTest",
			MetricsRegion: "PDX",
		}
		configs := samus_config.SamusConfigurations{MetricsConfig: metricsConf}

		s, err := NewServer(config.Statsd(), config.RollbarErrorLogger(), b, configs)
		So(err, ShouldBeNil)

		testServer := httptest.NewServer(s)
		defer testServer.Close()

		Convey("When user is prime", func() {

			userIDPrime := "11111"
			userIDNoNPrime := "22222"
			userIDError := "33333"
			b.On("Status", mock.Anything, userIDPrime).Return(&backend.StatusResponse{UserID: userIDPrime, TwitchPrime: true}, http.StatusOK, nil)

			resp, err := http.Get(fmt.Sprintf("%v/samus/users/%s/status", testServer.URL, userIDPrime))
			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			statusResponse := backend.StatusResponse{}
			ConvertToResponseInterface(resp, &statusResponse)

			So(statusResponse.UserID, ShouldEqual, userIDPrime)
			So(statusResponse.TwitchPrime, ShouldEqual, true)

			Convey("When user is not prime", func() {
				b.On("Status", mock.Anything, userIDNoNPrime).Return(&backend.StatusResponse{UserID: userIDNoNPrime, TwitchPrime: false}, http.StatusOK, nil)

				resp, err := http.Get(fmt.Sprintf("%v/samus/users/%s/status", testServer.URL, userIDNoNPrime))
				log.Debug("Raw HTTP response is:", resp)
				So(err, ShouldBeNil)
				So(resp.StatusCode, ShouldEqual, http.StatusOK)

				statusResponse := backend.StatusResponse{}
				ConvertToResponseInterface(resp, &statusResponse)

				So(statusResponse.UserID, ShouldEqual, userIDNoNPrime)
				So(statusResponse.TwitchPrime, ShouldEqual, false)

				Convey("When Bad Gateway", func() {
					b.On("Status", mock.Anything, userIDError).Return(nil, http.StatusBadGateway, errors.New("Error for test"))

					resp, err := http.Get(fmt.Sprintf("%v/samus/users/%s/status", testServer.URL, userIDError))
					log.Debug("Raw HTTP response is:", resp)
					So(err, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusBadGateway)

					statusResponse := backend.StatusResponse{}
					ConvertToResponseInterface(resp, &statusResponse)
				})
			})
		})
	})
}

// ConvertToResponseInterface parases http.Response to a specified interface and make this piece of code reusable
func ConvertToResponseInterface(resp *http.Response, v interface{}) {
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		log.Debug("Error is:", err)
	}
	err = json.Unmarshal(body, &v)
	if err != nil {
		log.Debug("Unmarshal Error is:", err)
	}
	log.Debug("Converted Interface is :", v)
	log.Debug("Converted Response is: ", string(body[:]))
}
