package api

import (
	"net/http"
	"strings"
	"testing"

	"fmt"

	"net/http/httptest"

	"errors"

	"code.justin.tv/common/config"
	"code.justin.tv/samus/gateway/backend"
	samus_config "code.justin.tv/samus/gateway/configuration"
	"code.justin.tv/samus/gateway/metrics"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

//TestPrimeSettings for PrimeSettings API validation
func TestPrimeSettings(t *testing.T) {

	Convey("Prime Settings API", t, func() {
		log.SetLevel(log.DebugLevel)
		b := &backend.BackendMock{}
		metricsConf := &metrics.MetricsConfig{
			Environment:   "UnitTest",
			MetricsRegion: "PDX",
		}
		configs := samus_config.SamusConfigurations{MetricsConfig: metricsConf}
		s, err := NewServer(config.Statsd(), config.RollbarErrorLogger(), b, configs)
		So(err, ShouldBeNil)

		testServer := httptest.NewServer(s)
		defer testServer.Close()

		Convey("When getting settings", func() {

			userIDSettings := "11111"
			userIDNonSettings := "22222"
			userIDError := "33333"
			b.On("GetSettings", mock.Anything, userIDSettings).Return(&backend.SettingsResponse{UserID: userIDSettings, IsSubscriptionMessage: false}, http.StatusOK, nil)

			resp, err := http.Get(fmt.Sprintf("%v/api/users/%s/prime/settings", testServer.URL, userIDSettings))
			log.Debug("Raw HTTP response is:", resp)
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, http.StatusOK)

			settingsResponse := backend.SettingsResponse{}
			ConvertToResponseInterface(resp, &settingsResponse)

			So(settingsResponse.UserID, ShouldEqual, userIDSettings)
			So(settingsResponse.IsSubscriptionMessage, ShouldEqual, false)

			Convey("When user does not have settings", func() {
				b.On("GetSettings", mock.Anything, userIDNonSettings).Return(&backend.SettingsResponse{UserID: userIDNonSettings, IsSubscriptionMessage: true}, http.StatusOK, nil)

				resp, err := http.Get(fmt.Sprintf("%v/api/users/%s/prime/settings", testServer.URL, userIDNonSettings))
				log.Debug("Raw HTTP response is:", resp)
				So(err, ShouldBeNil)
				So(resp.StatusCode, ShouldEqual, http.StatusOK)

				settingsResponse := backend.SettingsResponse{}
				ConvertToResponseInterface(resp, &settingsResponse)

				So(settingsResponse.UserID, ShouldEqual, userIDNonSettings)
				So(settingsResponse.IsSubscriptionMessage, ShouldEqual, true)

				Convey("When Bad Gateway", func() {
					b.On("GetSettings", mock.Anything, userIDError).Return(nil, http.StatusBadGateway, errors.New("Error for test"))

					resp, err := http.Get(fmt.Sprintf("%v/api/users/%s/prime/settings", testServer.URL, userIDError))
					log.Debug("Raw HTTP response is:", resp)
					So(err, ShouldBeNil)
					So(resp.StatusCode, ShouldEqual, http.StatusBadGateway)

					settingsResponse := backend.SettingsResponse{}
					ConvertToResponseInterface(resp, &settingsResponse)
				})
			})
		})

		Convey("When updating settings", func() {
			userIDSettingsTrue := "11111"
			userIDSettingsFalse := "22222"

			Convey("When setting to true", func() {
				b.On("UpdateSettings", mock.Anything, userIDSettingsTrue, true).Return(&backend.SettingsResponse{UserID: userIDSettingsTrue, IsSubscriptionMessage: true}, http.StatusOK, nil)

				resp, err := http.Post(fmt.Sprintf("%v/api/users/%s/prime/settings", testServer.URL, userIDSettingsTrue), "application/json", strings.NewReader("{\"isSubscriptionMessage\": true}"))
				log.Debug("Raw HTTP response is:", resp)
				So(err, ShouldBeNil)
				So(resp.StatusCode, ShouldEqual, http.StatusOK)

				settingsResponse := backend.SettingsResponse{}
				ConvertToResponseInterface(resp, &settingsResponse)

				So(settingsResponse.UserID, ShouldEqual, userIDSettingsTrue)
				So(settingsResponse.IsSubscriptionMessage, ShouldEqual, true)
			})

			Convey("When setting to false", func() {
				b.On("UpdateSettings", mock.Anything, userIDSettingsFalse, false).Return(&backend.SettingsResponse{UserID: userIDSettingsFalse, IsSubscriptionMessage: false}, http.StatusOK, nil)

				resp, err := http.Post(fmt.Sprintf("%v/api/users/%s/prime/settings", testServer.URL, userIDSettingsFalse), "application/json", strings.NewReader("{\"isSubscriptionMessage\": false}"))
				log.Debug("Raw HTTP response is:", resp)
				So(err, ShouldBeNil)
				So(resp.StatusCode, ShouldEqual, http.StatusOK)

				settingsResponse := backend.SettingsResponse{}
				ConvertToResponseInterface(resp, &settingsResponse)

				So(settingsResponse.UserID, ShouldEqual, userIDSettingsFalse)
				So(settingsResponse.IsSubscriptionMessage, ShouldEqual, false)
			})
		})
	})
}
