package api

import (
	"net/http"

	"github.com/pkg/errors"

	samus_gateway "code.justin.tv/samus/gateway/client"
	. "code.justin.tv/samus/gateway/util"
	log "github.com/sirupsen/logrus"
	"goji.io/pat"
	"golang.org/x/net/context"
)

const (
	NoBalance            = "NO_BALANCE"
	HasPaidSub           = "HAS_PAID_SUB"
	NoProductsForChannel = "NO_PRODUCTS_FOR_CHANNEL"
	TooManyRecentSpends  = "TOO_MANY_RECENT_SPENDS"
	PaymentConflict      = "PAYMENT_CONFLICT"
)

type SpendSubscriptionCreditResponse struct {
	UserID                    string `json:"twitchUserId"`
	BroadcasterID             string `json:"broadcasterId"`
	SubscriptionCreditBalance int    `json:"subscriptionCreditBalance"`
}

// SpendSubscriptionCredit API/Activity routes to backend/spend_subscription_credit.go
func (s *Server) SpendSubscriptionCredit(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	// Get path parameters.
	userID := pat.Param(ctx, "userId")
	userIDValidationErrMsg := ValidateTuid(userID)
	if userIDValidationErrMsg != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, errors.Wrap(userIDValidationErrMsg, "UserId is not valid"))
		return
	}

	broadcasterID := pat.Param(ctx, "broadcasterId")
	broadcasterIDValidationErrMsg := ValidateTuid(broadcasterID)
	if broadcasterIDValidationErrMsg != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, errors.Wrap(broadcasterIDValidationErrMsg, "BroadcasterId is not valid"))
		return
	}

	logger := log.WithFields(log.Fields{
		"userID":        userID,
		"broadcasterID": broadcasterID,
	})

	logger.Infof("Attempting to spend subscription credit for Context: [%+v]", ctx)

	response, err := s.Backend.SpendSubscriptionCredit(ctx, userID, broadcasterID)
	if err != nil {
		logger.Error("[SpendSubscriptionCredit] ", "backend.SpendSubscriptionCredit : ", err.Error())
		if err.Error() == NoBalance {
			noBalance := samus_gateway.ErrorResponse{
				Code: samus_gateway.NoBalance,
			}
			s.serveJSONError(ctx, w, r, http.StatusBadRequest, noBalance, err)
			return
		}
		if err.Error() == HasPaidSub {
			hasPaidSub := samus_gateway.ErrorResponse{
				Code: samus_gateway.HasPaidSub,
			}
			s.serveJSONError(ctx, w, r, http.StatusBadRequest, hasPaidSub, err)
			return
		}
		if err.Error() == NoProductsForChannel {
			invalidProduct := samus_gateway.ErrorResponse{
				Code: samus_gateway.InvalidProduct,
			}
			s.serveJSONError(ctx, w, r, http.StatusBadRequest, invalidProduct, err)
			return
		}
		if err.Error() == TooManyRecentSpends {
			tooManyRecentSpends := samus_gateway.ErrorResponse{
				Code: samus_gateway.TooManyRecentSpends,
			}
			s.serveJSONError(ctx, w, r, http.StatusBadRequest, tooManyRecentSpends, err)
			return
		}
		if err.Error() == PaymentConflict {
			paymentConflict := samus_gateway.ErrorResponse{
				Code: samus_gateway.PaymentConflict,
			}
			s.serveJSONError(ctx, w, r, http.StatusConflict, paymentConflict, err)
			return
		}
		s.serveError(ctx, w, r, http.StatusInternalServerError, err)
		return
	} else {
		logger.Infof("Successfully spent subscription credit for Context: [%+v]", ctx)
	}

	bodyResponse := SpendSubscriptionCreditResponse{
		UserID:                    response.UserID,
		BroadcasterID:             response.BroadcasterID,
		SubscriptionCreditBalance: response.SubscriptionCreditBalance,
	}

	s.serveJSON(ctx, w, r, bodyResponse)
}
