package api

import (
	"net/http"

	errors2 "github.com/pkg/errors"

	samus_gateway "code.justin.tv/samus/gateway/client"
	. "code.justin.tv/samus/gateway/util"
	log "github.com/sirupsen/logrus"
	"goji.io/pat"
	"golang.org/x/net/context"
)

const (
	NoSpendHistory = "NO_SPEND_HISTORY"
)

type UnspendSubscriptionCreditResponse struct {
	TransactionId string `json:"transactionId"`
}

// UnspendSubscriptionCredit API/Activity routes to backend/unspend_subscription_credit.go
func (s *Server) UnspendSubscriptionCredit(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	// Get path parameters.
	userID := pat.Param(ctx, "userId")
	userIDValidationErrMsg := ValidateTuid(userID)
	if userIDValidationErrMsg != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, errors2.Wrap(userIDValidationErrMsg, "UserId is not valid"))
		return
	}

	broadcasterID := pat.Param(ctx, "broadcasterId")
	broadcasterIDValidationErrMsg := ValidateTuid(broadcasterID)
	if broadcasterIDValidationErrMsg != nil {
		s.serveError(ctx, w, r, http.StatusBadRequest, errors2.Wrap(broadcasterIDValidationErrMsg, "BroadcasterId is not valid"))
		return
	}

	logger := log.WithFields(log.Fields{
		"userID":        userID,
		"broadcasterID": broadcasterID,
	})

	response, err := s.Backend.UnspendSubscriptionCredit(ctx, userID, broadcasterID)
	if err != nil {
		logger.Error("[UnspendSubscriptionCredit] ", "backend.UnspendSubscriptionCredit : ", err.Error())
		if err.Error() == NoSpendHistory {
			noBalance := samus_gateway.ErrorResponse{
				Code: samus_gateway.NoSpendHistory,
			}
			s.serveJSONError(ctx, w, r, http.StatusBadRequest, noBalance, err)
			return
		}
		s.serveError(ctx, w, r, http.StatusInternalServerError, err)
		return
	}

	bodyResponse := UnspendSubscriptionCreditResponse{
		TransactionId: response.TransactionID,
	}

	s.serveJSON(ctx, w, r, bodyResponse)
}
