package backend

import (
	"net/http"

	"code.justin.tv/samus/gateway/dynamo"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"golang.org/x/net/context"
)

type GameAccountLink struct {
	UserID      string      `json:"userId"`
	GameID      string      `json:"gameId"`
	GameAccount GameAccount `json:"gameAccount"`
}

type GameAccount struct {
	AccountID   string `json:"accountId"`
	DisplayName string `json:"displayName"`
}

func (b *Backend) GetAccountLink(ctx context.Context, userID, gameID string) (*GameAccountLink, int, error) {
	link, err := b.accountLinkDao.Get(userID, gameID)
	if err != nil {
		return nil, http.StatusInternalServerError, err
	}

	if link == nil {
		return nil, http.StatusNotFound, nil
	}

	result := GameAccountLink{
		UserID: string(link.UserID),
		GameID: gameID,
		GameAccount: GameAccount{
			AccountID:   link.AccountId,
			DisplayName: link.DisplayName,
		},
	}

	return &result, http.StatusOK, nil
}

func (b *Backend) CreateAccountLink(ctx context.Context, accountLink GameAccountLink) (int, error) {
	link := dynamo.AccountLink{
		UserID:      dynamo.TwitchUserID(accountLink.UserID),
		GameID:      accountLink.GameID,
		AccountId:   accountLink.GameAccount.AccountID,
		DisplayName: accountLink.GameAccount.DisplayName,
	}
	err := b.accountLinkDao.Put(&link)

	if err != nil {
		awsError, isAwsError := err.(awserr.Error)
		if isAwsError && awsError.Code() == "ConditionalCheckFailedException" {
			return http.StatusConflict, err
		} else {
			return http.StatusInternalServerError, err
		}
	}

	return http.StatusOK, nil
}

func (b *Backend) DeleteAccountLink(ctx context.Context, userID, gameID string) (*GameAccountLink, int, error) {
	link, _, err := b.GetAccountLink(ctx, userID, gameID)
	if err != nil {
		return nil, http.StatusInternalServerError, err
	}

	err = b.accountLinkDao.Delete(userID, gameID)
	if err != nil {
		return nil, http.StatusInternalServerError, err
	}

	return link, http.StatusOK, nil
}
