package backend

import (
	"testing"

	"bytes"
	"io/ioutil"
	"net/http"

	"encoding/json"

	"code.justin.tv/common/config"
	"code.justin.tv/samus/gateway/clients/mocks"
	nitro "code.justin.tv/samus/nitro/rpc"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
	"golang.org/x/net/context"
)

func TestCanSpendPrimeCreditBalanceBackend(t *testing.T) {

	log.SetLevel(log.DebugLevel)

	Convey("When User does not have prime", t, func() {
		mockNitroClient := new(mocks.Nitro)
		samusSWSClientMock := new(mocks.HttpMock)
		productID := "tww2"

		b := Backend{stats: config.Statsd(),
			samusSWSClient: samusSWSClientMock,
			nitroClient:    mockNitroClient,
		}

		premiumStatusResponse := &nitro.GetPremiumStatusesResponse{
			HasPrime: false,
			HasTurbo: true,
		}

		mockNitroClient.On("GetPremiumStatuses", mock.Anything, mock.Anything).Return(premiumStatusResponse, nil)

		resp, s, e := b.CanSpendPrimeCredit(context.TODO(), userID, productID)

		So(resp.CanSpendPrimeCredit, ShouldEqual, false)
		So(s, ShouldEqual, 200)
		So(e, ShouldBeNil)
		mockNitroClient.AssertNumberOfCalls(t, "GetPremiumStatuses", 1)
	})

	Convey("When Balance API is 1", t, func() {
		mockNitroClient := new(mocks.Nitro)
		samusSWSClientMock := new(mocks.HttpMock)
		productID := "tww2"

		b := Backend{stats: config.Statsd(),
			samusSWSClient: samusSWSClientMock,
			nitroClient:    mockNitroClient,
		}

		premiumStatusResponse := &nitro.GetPremiumStatusesResponse{
			HasPrime: true,
			HasTurbo: true,
		}

		mockNitroClient.On("GetPremiumStatuses", mock.Anything, mock.Anything).Return(premiumStatusResponse, nil)

		balanceResponse := BalanceResponse{CreditBalance: 1}
		buf, err := json.Marshal(balanceResponse)
		if err != nil {
			log.Debug(err)
		}
		mockedResp := new(http.Response)
		mockedResp.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
		mockedResp.StatusCode = http.StatusOK
		samusSWSClientMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockedResp, nil)

		resp, s, e := b.CanSpendPrimeCredit(context.TODO(), userID, productID)

		So(resp.CanSpendPrimeCredit, ShouldEqual, true)
		So(s, ShouldEqual, 200)
		So(e, ShouldBeNil)
		mockNitroClient.AssertNumberOfCalls(t, "GetPremiumStatuses", 1)
		samusSWSClientMock.AssertNumberOfCalls(t, "Do", 1)
	})

	Convey("When Balance API is 0", t, func() {
		mockNitroClient := new(mocks.Nitro)
		samusSWSClientMock := new(mocks.HttpMock)
		productID := "tww2"

		b := Backend{stats: config.Statsd(),
			samusSWSClient: samusSWSClientMock,
			nitroClient:    mockNitroClient,
		}

		premiumStatusResponse := &nitro.GetPremiumStatusesResponse{
			HasPrime: true,
			HasTurbo: true,
		}

		mockNitroClient.On("GetPremiumStatuses", mock.Anything, mock.Anything).Return(premiumStatusResponse, nil)

		balanceResponse := BalanceResponse{CreditBalance: 0}
		buf, err := json.Marshal(balanceResponse)
		if err != nil {
			log.Debug(err)
		}
		mockedResp := new(http.Response)
		mockedResp.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
		mockedResp.StatusCode = http.StatusOK
		samusSWSClientMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockedResp, nil)

		resp, s, e := b.CanSpendPrimeCredit(context.TODO(), userID, productID)

		So(resp.CanSpendPrimeCredit, ShouldEqual, false)
		So(s, ShouldEqual, 200)
		So(e, ShouldBeNil)
		mockNitroClient.AssertNumberOfCalls(t, "GetPremiumStatuses", 1)
		samusSWSClientMock.AssertNumberOfCalls(t, "Do", 1)
	})

}
