package backend

import (
	"bytes"

	nitro "code.justin.tv/samus/nitro/rpc"
	log "github.com/sirupsen/logrus"
	"golang.org/x/net/context"
)

// SamusGrantCancelRequest holds request params for grant and cancel
type SamusGrantCancelRequest struct {
	UserID int `json:"user_id"`
}

// GrantSamusBenefitsResponse holds subscription details
type GrantSamusBenefitsResponse struct {
	ExtSubscriptionID int    `json:"ext_subscription_id"`
	PurchaseDateTime  string `json:"purchase_datetime"`
}

// CancelSamusBenefitsResponse holds canceled ticket id
type CancelSamusBenefitsResponse struct {
	TicketID int `json:"ticket_id"`
}

// Wrapper for premium status values
type PremiumStatusResponse struct {
	HasPrime bool
	HasTurbo bool
}

// GrantPrime calls nitro to grant Twitch Prime ticket product
func (b *Backend) GrantPrime(ctx context.Context, userID string) (*PremiumStatusResponse, error) {
	logger := log.WithFields(log.Fields{
		"userID": userID,
	})

	request := &nitro.GrantPremiumStatusRequest{
		TwitchUserID: userID,
		ProductName:  nitro.PremiumProduct_PRIME,
	}

	logger.Infof("Attempting to GrantPrime: [%+v]\nContext: [%+v]", request, ctx)

	response, err := b.nitroClient.GrantPremiumStatus(ctx, request)
	if err != nil {
		logger.Error("[GrantPrime] ", "Failed to GrantPrime from Nitro.", err)
		return nil, err
	}

	logger.Infof("Successfull call to GrantPrime: [%+v]\nContext: [%+v]", request, ctx)

	return &PremiumStatusResponse{
		HasPrime: response.HasPrime,
	}, nil
}

// CancelPrime calls nitro to cancel Twitch Prime ticket product
func (b *Backend) CancelPrime(ctx context.Context, userID string) (*PremiumStatusResponse, error) {
	logger := log.WithFields(log.Fields{
		"userID": userID,
	})

	request := &nitro.CancelPremiumStatusRequest{
		TwitchUserID: userID,
		ProductName:  nitro.PremiumProduct_PRIME,
	}

	logger.Infof("Attempting to CancelPrime: [%+v]\nContext: [%+v]", request, ctx)

	response, err := b.nitroClient.CancelPremiumStatus(ctx, request)
	if err != nil {
		logger.Error("[CancelPrime] ", "Failed to CancelPrime from Nitro.", err)
		return nil, err
	}

	logger.Infof("Successfull call to CancelPrime: [%+v]\nContext: [%+v]", request, ctx)

	return &PremiumStatusResponse{
		HasPrime: response.HasPrime,
	}, nil
}

// GrantSamusBenefits grants samus benetfits
func (b *Backend) GrantSamusBenefits(ctx context.Context, jsonStr []byte) (*GrantSamusBenefitsResponse, int, error) {

	req, err := b.entitlementsClient.NewRequest("POST", "/api/internal/samus_benefits/grant", bytes.NewBuffer(jsonStr))
	req.Header.Set("Content-Type", "application/json")
	if err != nil {
		return nil, 0, err
	}
	log.Infof("Attempting to GrantSamusBenefits: [%+v]\nContext: [%+v]", req, ctx)

	grantSamusBenefitsResponse := GrantSamusBenefitsResponse{}
	s, err := sendRequestAndParseResponse(b.entitlementsClient, ctx, req, &grantSamusBenefitsResponse)
	if err != nil {
		return nil, s, err
	}

	log.Infof("Successfull call to GrantSamusBenefits: [%+v]\nContext: [%+v]", req, ctx)

	return &grantSamusBenefitsResponse, 0, nil
}

// CancelSamusBenefits cancels samus benefits
func (b *Backend) CancelSamusBenefits(ctx context.Context, jsonStr []byte) (*CancelSamusBenefitsResponse, int, error) {

	req, err := b.entitlementsClient.NewRequest("POST", "/api/internal/samus_benefits/cancel", bytes.NewBuffer(jsonStr))
	req.Header.Set("Content-Type", "application/json")
	if err != nil {
		return nil, 0, err
	}
	log.Infof("Attempting to CancelSamusBenefits: [%+v]\nContext: [%+v]", req, ctx)

	cancelSamusBenefitsResponse := CancelSamusBenefitsResponse{}
	s, err := sendRequestAndParseResponse(b.entitlementsClient, ctx, req, &cancelSamusBenefitsResponse)
	if err != nil {
		return nil, s, err
	}

	log.Infof("Successfull call to CancelSamusBenefits: [%+v]\nContext: [%+v]", req, ctx)

	return &cancelSamusBenefitsResponse, 0, nil
}
