package backend

import (
	"encoding/json"

	log "github.com/sirupsen/logrus"

	"errors"

	samus_gateway "code.justin.tv/samus/gateway/clients"
	. "code.justin.tv/samus/gateway/util"
	"github.com/aws/aws-sdk-go/service/sqs"
)

/*
Sample SUCCEEDED message
{
  "Type": "Notification",
  "MessageId": "15d1e8aa-3b43-5880-8760-b1efe3868aec",
  "TopicArn": "arn:aws:sns:us-west-2:755231880819:SODSStack-beta-sodssnssodssnstopic483D1EB2-ZQ8R885ARM76",
  "Message": "
  	{
  		\"orderId\":\"amzn1.pg.order.50a7b7f3-815e-4e1a-8171-bb54efc5163d\",
  		\"orderState\":\"SUCCEEDED\",
  		\"offerId\":\"amzn1.pg.offer.72bbed82-eb0c-df5b-bfde-e9de51c3adc4\",
  		\"itemId\":\"amzn1.adg.product.6d53b913-4cc4-46d3-afb1-67d6439eda6d\",
  		\"itemTitle\":\"Songdale - FGWP Test Offer w/ Pixels\",
  		\"entitlementId\":\"amzn1.adg.product.6d53b913-4cc4-46d3-afb1-67d6439eda6d_80603332-6447-4c4b-8f30-819f1fb7154b\",\
  		"offerCrownCost\":0,
  		\"benefitOwnerCustomerId\":\"A3VA739I82MIQF\",
  		\"actingCustomerId\":\"A3VA739I82MIQF\",
  		\"nonAmazonClaimingAccountList\":[
  			{\"accountType\":\"TWITCH\",\"accountId\":\"174288998\"}
  		],
  		\"entitlementAccountList\":[
  			{\"accountType\":\"AMAZON_CONSUMER\",\"accountId\":\"A3VA739I82MIQF\"},
  			{\"accountType\":\"AMAZON\",\"accountId\":\"A3VA739I82MIQF\"},
  			{\"accountType\":\"TWITCH\",\"accountId\":\"174288998\"}
  		],
  		\"eligibilityDetails\":\"{
  			\\\"actingCustomerId\\\":\\\"A3VA739I82MIQF\\\",
  			\\\"benefitOwnerCustomerId\\\":\\\"A3VA739I82MIQF\\\",
  			\\\"customerPrimeMarketplaceId\\\":\\\"ATVPDKIKX0DER\\\",
  			\\\"offerId\\\":\\\"amzn1.pg.offer.72bbed82-eb0c-df5b-bfde-e9de51c3adc4\\\",
  			\\\"title\\\":\\\"Songdale - FGWP Test Offer w/ Pixels\\\",
  			\\\"itemId\\\":\\\"amzn1.adg.product.6d53b913-4cc4-46d3-afb1-67d6439eda6d\\\",
  			\\\"eligible\\\":true,
  			\\\"accountsToEntitle\\\":[
          {\\\"accountType\\\":\\\"AMAZON_CONSUMER\\\",\\\"accountId\\\":\\\"A3VA739I82MIQF\\\",\\\"displayName\\\":null},
          {\\\"accountType\\\":\\\"AMAZON\\\",\\\"accountId\\\":\\\"A3VA739I82MIQF\\\",\\\"displayName\\\":null},
          {\\\"accountType\\\":\\\"TWITCH\\\",\\\"accountId\\\":\\\"174288998\\\",\\\"displayName\\\":null}
  		  ],
        \\\"offerState\\\":\\\"LIVE\\\",
        \\\"eligibilityResults\\\":[
          {\\\"ruleName\\\":\\\"OFFER_WINDOW\\\",\\\"eligible\\\":true},
          {\\\"ruleName\\\":\\\"LINKED_ACCOUNT\\\",\\\"eligible\\\":true},
          {\\\"ruleName\\\":\\\"PRIME_GAMING\\\",\\\"eligible\\\":true},
          {\\\"ruleName\\\":\\\"MARKETPLACE\\\",\\\"eligible\\\":true},
          {\\\"ruleName\\\":\\\"CLAIM_LIMIT\\\",\\\"eligible\\\":true}
        ],
        \\\"priorEntitlements\\\":[],
        \\\"priorHouseholdClaimerCustomerIds\\\":[]
  		}\",
  		\"eligibilityCheckDate\":\"2021-06-17T23:11:00.576Z\",
  		\"herdWorkId\":\"amzn1.pg.order.50a7b7f3-815e-4e1a-8171-bb54efc5163d\",
  		\"attributionChannel\":\"{\\\"eventId\\\":\\\"ClaimInfo:amzn1.pg.offer.72bbed82-eb0c-df5b-bfde-e9de51c3adc4\\\",\\\"page\\\":\\\"home\\\"}\",
  		\"orderDocumentCreationDate\":\"2021-06-17T23:11:00.191Z\",
  		\"orderDocumentLastUpdateDate\":\"2021-06-17T23:11:01.575Z\",
  		\"orderDocumentCreationIdempotenceKey\":\"6b68fa5d5d358b648e05a5929d8b3856\",
  		\"versionNumber\":2
  	}",
  "Timestamp": "2021-06-17T23:11:01.967Z",
  "SignatureVersion": "1",
  "Signature": "B5U5RwG/ANmhic/ghAp1HcihMCkuLT6i4FxqT2/tT+BgsvFx3ip0RjiTzb8VV2tw/Z/vuXdYBEsNl+rmuzxM8JVgZqFzGlkEr0aiPPEzewiBhqjcbPZbs5yhg2HWW9wfrdDPHHo2k5Rwko3OjsX3WOT37skXpsdjxwRSWJq0i2hYEsElRmkN2EKejrFqCHRjleX4AmBcTElwkeb7AU8WjzCa7NTqM5r7UYkQlbMzF2bayJyOWAnhxyQL4hWEz6svygwXZ5nP3UlRVwVeG8qKAjCOqwDpIuwV+Ct+j4rIcUfyW9r8YpJC3ioCTe3VfUxiGtZkuI006Nbfq+nQCsQwxA==",
  "SigningCertURL": "https://sns.us-west-2.amazonaws.com/SimpleNotificationService-010a507c1833636cd94bdb98bd93083a.pem",
  "UnsubscribeURL": "https://sns.us-west-2.amazonaws.com/?Action=Unsubscribe&SubscriptionArn=arn:aws:sns:us-west-2:755231880819:SODSStack-beta-sodssnssodssnstopic483D1EB2-ZQ8R885ARM76:0294530f-d876-418f-aa61-5fe8f6feb454"
}
*/

const ORDER_SUCCEEDED = "SUCCEEDED"
const ORDER_FAILED = "FAILED"
const ORDER_ERRORED = "ERRORED"
const ORDER_TYPE_SUCCEEDED = "claim-success"
const ORDER_TYPE_FAILED = "claim-failed"
const TWITCH_ACCOUNT_TYPE = "TWITCH"

// EntitlementAccount is an account that is entitled to this order
type ClaimingAccount struct {
	AccountType string `json:"accountType"`
	AccountId   string `json:"accountId"`
}

// OrderSQSMessage holds the details of the order message
type OrderSQSMessage struct {
	OrderState          string            `json:"orderState"`
	OrderId             string            `json:"orderId"`
	OfferId             string            `json:"offerId"`
	ClaimingAccountList []ClaimingAccount `json:"nonAmazonClaimingAccountList"`
}

// ProcessOrderSQSNotification checks for Order status and then pushes the succeeded and failed messages to a PubSub topic
func (b *Backend) ProcessOrderSQSNotification(rawMsg *sqs.Message) error {
	var msg SQSMessage

	rawBytes := []byte(*rawMsg.Body)
	if err := json.Unmarshal(rawBytes, &msg); err != nil {
		return err
	}

	var msgData OrderSQSMessage

	msgBytes := []byte(msg.Message)
	if err := json.Unmarshal(msgBytes, &msgData); err != nil {
		return err
	}

	orderId := msgData.OrderId
	offerId := msgData.OfferId
	orderState := msgData.OrderState
	claimingAccountList := msgData.ClaimingAccountList
	err := b.HandleOrderSQSMessage(orderId, offerId, orderState, claimingAccountList)
	if err != nil {
		log.WithError(err).Error("Error processing PubSub data for orderId: ", orderId, ", offerId:", offerId, ", orderState: ", orderState)
		return err
	}

	return nil
}

// HandleOrderSQSMessage validates the SQS message, builds the PubSubEvent, and publishes the PubSubEvent to the topic
func (b *Backend) HandleOrderSQSMessage(orderId string, offerId string, orderState string, claimingAccountList []ClaimingAccount) error {
	if len(orderId) == 0 || len(offerId) == 0 || len(orderState) == 0 {
		errorMsg := "Order SQS message malformed: missing the OrderId, OfferId, or OrderState"
		log.Error(errorMsg)
		return errors.New(errorMsg)
	}

	for _, account := range claimingAccountList {
		accountType := account.AccountType
		userId := account.AccountId
		if accountType == TWITCH_ACCOUNT_TYPE && ValidateTuid(userId) == nil {
			if orderState == ORDER_SUCCEEDED {
				orderCompletionEvent := SqsToPubsubEvent(orderId, offerId, ORDER_TYPE_SUCCEEDED)
				err := b.pubsubClient.PublishOrderCompletion(orderCompletionEvent, userId)
				if err != nil {
					log.WithError(err).Error("Error while publishing a SUCCESS order to prime-gaming-offer.userId")
				}
			} else if orderState == ORDER_FAILED || orderState == ORDER_ERRORED {
				orderCompletionEvent := SqsToPubsubEvent(orderId, offerId, ORDER_TYPE_FAILED)
				err := b.pubsubClient.PublishOrderCompletion(orderCompletionEvent, userId)
				if err != nil {
					log.WithError(err).Error("Error while publishing a FAILED or ERRORED order to prime-gaming-offer.userId")
				}
			}
		}
	}
	return nil
}

func SqsToPubsubEvent(orderId string, offerId string, orderCompletionType string) samus_gateway.OrderCompletionEvent {
	return samus_gateway.OrderCompletionEvent{
		Type: orderCompletionType,
		Data: SqsToPubsubCompletionEventData(orderId, offerId),
	}
}

func SqsToPubsubCompletionEventData(orderId string, offerId string) samus_gateway.OrderCompletionEventData {
	return samus_gateway.OrderCompletionEventData{
		OrderId: orderId,
		OfferId: offerId,
	}
}
