package backend

import (
	"testing"

	"code.justin.tv/common/config"
	"code.justin.tv/samus/gateway/clients"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

const ORDER_ID = "amzn1.pg.order.50a7b7f3-815e-4e1a-8171-bb54efc5163d"
const OFFER_ID = "amzn1.pg.offer.72bbed82-eb0c-df5b-bfde-e9de51c3adc4"
const ORDER_STATE_SUCCEEDED = "SUCCEEDED"
const ORDER_STATE_FAILED = "FAILED"
const ACCOUNT_TYPE = "TWITCH"
const ACCOUNT_ID = "accountId"

func TestOrderSQSHandler(t *testing.T) {

	Convey("Order SQS Handler", t, func() {
		log.SetLevel(log.DebugLevel)

		mockPubSubClient := new(clients.PubSubClientMock)

		backend := Backend{
			stats:        config.Statsd(),
			pubsubClient: mockPubSubClient,
		}

		Convey("Publishes a SUCCEEDED message to PubSub", func() {
			mockClaimingAccounts := []ClaimingAccount{
				ClaimingAccount{
					AccountType: ACCOUNT_TYPE,
					AccountId:   ACCOUNT_ID,
				},
			}
			mockPubSubClient.On("Publish", mock.Anything).Return(nil)

			err := backend.HandleOrderSQSMessage(ORDER_ID, OFFER_ID, ORDER_STATE_SUCCEEDED, mockClaimingAccounts)
			if err != nil {
				log.Debug("Error", err)
			}
			So(err, ShouldBeNil)
		})

		Convey("Publishes a FAILED message to PubSub", func() {
			mockClaimingAccounts := []ClaimingAccount{
				ClaimingAccount{
					AccountType: ACCOUNT_TYPE,
					AccountId:   ACCOUNT_ID,
				},
			}
			mockPubSubClient.On("Publish", mock.Anything).Return(nil)

			err := backend.HandleOrderSQSMessage(ORDER_ID, OFFER_ID, ORDER_STATE_FAILED, mockClaimingAccounts)
			if err != nil {
				log.Debug("Error", err)
			}
			So(err, ShouldBeNil)
		})

		Convey("Skips messages without a TWITCH account", func() {
			mockPubSubClient.On("Publish", mock.Anything).Return(nil)

			err := backend.HandleOrderSQSMessage(ORDER_ID, OFFER_ID, ORDER_STATE_FAILED, nil)
			if err != nil {
				log.Debug("Error", err)
			}
			So(err, ShouldBeNil)
		})

		Convey("Errors if orderId is empty", func() {
			mockPubSubClient.On("Publish", mock.Anything).Return(nil)

			err := backend.HandleOrderSQSMessage("", OFFER_ID, ORDER_STATE_FAILED, nil)
			if err != nil {
				log.Debug("Error", err)
			}
			So(err, ShouldNotBeNil)
		})
	})
}
