package backend

import (
	"fmt"
	"net/http"

	"strconv"

	"code.justin.tv/samus/gateway/clients"
	"code.justin.tv/samus/gateway/dynamo"
	log "github.com/sirupsen/logrus"
	"golang.org/x/net/context"
)

// GetPrimeEntitlement returns the entitlement metadata for a users claimed offer, if they have one
func (b *Backend) GetPrimeEntitlement(ctx context.Context, userID, offerID, locale string) (*PrimeEntitlementResponse, int, error) {
	log.Debug("GetPrimeEntitlement for:", userID, " OfferID: ", offerID)

	entitlement, err := b.primeEntitlementsDao.Get(userID, offerID)
	if err != nil {
		errorMsg := fmt.Sprintf("Failed to load entitlement from Dynamo for, %v, %v", userID, err)
		log.WithError(err).Error(errorMsg)
		return nil, http.StatusInternalServerError, err
	}

	primeEntitlementResponse := PrimeEntitlementResponse{
		UserID:         userID,
		HasEntitlement: entitlement.IsEntitled,
		OfferID:        offerID,
	}

	if !entitlement.IsEntitled {
		log.Debug("User: ", userID, " is not entitled to OfferID: ", offerID)
		return &primeEntitlementResponse, http.StatusOK, nil
	}
	jsonStr := []byte{}
	resp, s, err := b.ClaimOffer(ctx, userID, offerID, locale, jsonStr)
	if err != nil {
		errorMsg := fmt.Sprintf("Failed to call claim offer for, %v, %v", userID, err)
		log.WithError(err).Error(errorMsg)
		return nil, s, err
	}

	response := &PrimeEntitlementResponse{
		UserID:             userID,
		HasEntitlement:     true,
		OfferID:            offerID,
		OfferClaimData:     resp.OfferClaimData,
		ClaimMethod:        resp.ClaimMethod,
		ClaimInstruction:   resp.ClaimInstruction,
		OfferClaimMetadata: resp.OfferClaimMetadata,
		OfferTitle:         resp.OfferTitle,
	}
	return response, http.StatusOK, nil
}

// UpdatePrimeEntitlement saves the entitlement once the user has claimed the offer
func (b *Backend) UpdatePrimeEntitlement(ctx context.Context, userID, offerID, locale string) error {
	log.Debug("Storing PrimeEntitlement for:", userID, " OfferID: ", offerID)

	primeEntitlement := &dynamo.PrimeEntitlement{
		OfferID:      dynamo.OfferID(offerID),
		IsEntitled:   true,
		TwitchUserID: dynamo.TwitchUserID(userID),
	}

	log.Infof("Attempting to UpdatePrimeEntitlement: [%+v]\nContext: [%+v]", primeEntitlement, ctx)

	dynamoErr := b.primeEntitlementsDao.Put(primeEntitlement)
	if dynamoErr != nil {
		log.Errorf("UpdatePrimeEntitlement FAILED: [%+v]\nContext: [%+v]", primeEntitlement, ctx)
		return dynamoErr
	}

	log.Infof("Successfull UpdatePrimeEntitlement: [%+v]\nContext: [%+v]", primeEntitlement, ctx)
	return dynamoErr
}

// SetPrimeEntitlement sets the entitlement state for a users claimed offer
func (b *Backend) SetPrimeEntitlement(ctx context.Context, userID string, offerID string, hasEntitlement string) (*PrimeEntitlementResponse, int, error) {
	log.Debug("SetPrimeEntitlement for:", userID, " OfferID: ", offerID, " new hasEntitlement:", hasEntitlement)

	newHasEntitlement, err := strconv.ParseBool(hasEntitlement)
	if err != nil {
		errorMsg := fmt.Sprintf("Invalid value sent for HasEntitlement, %v, %v", hasEntitlement, err)
		log.WithError(err).Error(errorMsg)
		return nil, http.StatusBadRequest, err
	}

	primeEntitlement := &dynamo.PrimeEntitlement{
		OfferID:      dynamo.OfferID(offerID),
		IsEntitled:   newHasEntitlement,
		TwitchUserID: dynamo.TwitchUserID(userID),
	}

	log.Infof("Storing PrimeEntitlement: [%+v]\nContext: [%+v]", primeEntitlement, ctx)

	dynamoErr := b.primeEntitlementsDao.Put(primeEntitlement)
	if dynamoErr != nil {
		return nil, http.StatusInternalServerError, dynamoErr
	}
	response := &PrimeEntitlementResponse{
		UserID:         userID,
		HasEntitlement: newHasEntitlement,
		OfferID:        offerID,
	}

	log.Infof("Successfully stored PrimeEntitlement: [%+v]\nContext: [%+v]", response, ctx)
	return response, http.StatusOK, nil
}

// Clears a customer's claim code so the customer can claim a new code. This API is intended for internal use only for the owning team and CS.
func (b *Backend) ClearOfferClaimCodeForUser(ctx context.Context, userID string, marketplaceID string, offerID string, csAgent string, csContactID string) (*clients.ClearOfferClaimCodeForUserResponse, int, error) {

	logger := log.WithFields(log.Fields{
		"twitchUserID":  userID,
		"marketPlaceID": marketplaceID,
		"offerID":       offerID,
		"csAgent":       csAgent,
		"csContactID":   csContactID,
	})
	logger.Info("Clearing offer claim code")

	clearOfferClaimCodeForUserRequest := clients.ClearOfferClaimCodeForUserRequest{
		TwitchUserID:  userID,
		MarketplaceID: marketplaceID,
		OfferID:       offerID,
		CsAgent:       csAgent,
		CsContactID:   csContactID,
	}

	response, s, err := b.samusTProxClient.ClearOfferClaimCodeForUser(ctx, &clearOfferClaimCodeForUserRequest)
	// swallow downstream errors
	failedResponse := &clients.ClearOfferClaimCodeForUserResponse{
		Success: false,
	}

	if err != nil {
		logger.Error("[ClearOfferClaimCodeForUser] ", "ClearOfferClaimCodeForUser failed with error: ", err.Error())
		return failedResponse, s, err
	}

	if s != http.StatusOK {
		logger.Error("[ClearOfferClaimCodeForUser] ", "ClearOfferClaimCodeForUser failed with status code: ", s)
		return failedResponse, s, err
	}

	logger.Info("[ClearOfferClaimCodeForUser] succeeded")
	return response, s, nil
}
