package backend

import (
	"testing"

	"bytes"
	"io/ioutil"
	"net/http"

	"encoding/json"

	"errors"

	"code.justin.tv/common/config"
	gatewayClient "code.justin.tv/samus/gateway/client"
	"code.justin.tv/samus/gateway/clients"
	"code.justin.tv/samus/gateway/clients/mocks"
	"code.justin.tv/samus/gateway/dynamo"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
	"golang.org/x/net/context"
)

//TestSamusEntitlementBackendAPIs Validates PrimeEntitlement backend APIs
func TestSamusEntitlementBackendAPIs(t *testing.T) {

	Convey("Samus Backend Entitlement APIs", t, func() {
		log.SetLevel(log.DebugLevel)

		samusSWSClientMock := new(mocks.HttpMock)
		samusTProxClientMock := new(clients.SamusTProxClientMock)
		dynamoMock := new(dynamo.IUserDaoMock)
		primeEntitlementDynamoMock := new(dynamo.IPrimeEntitlementDaoMock)

		entitlementsClientMock := new(mocks.HttpMock)

		b := Backend{stats: config.Statsd(),
			samusSWSClient:       samusSWSClientMock,
			samusTProxClient:     samusTProxClientMock,
			userDao:              dynamoMock,
			entitlementsClient:   entitlementsClientMock,
			primeEntitlementsDao: primeEntitlementDynamoMock,
		}

		Convey("GetPrimeEntitlement API", func() {
			Convey("When the call is successful", func() {
				primeEntitlement := dynamo.PrimeEntitlement{
					OfferID:      dynamo.OfferID(offerID),
					IsEntitled:   true,
					TwitchUserID: dynamo.TwitchUserID(userID),
				}

				primeEntitlementDynamoMock.On("Get", userID, offerID).Return(primeEntitlement, nil)

				claimOfferResponse := PrimeEntitlementResponse{
					OfferID:            offerID,
					ClaimInstruction:   "claimInstructions",
					ClaimMethod:        "claimCode",
					OfferClaimData:     "offerClaimData",
					OfferClaimMetadata: gatewayClient.OfferClaimMetadata{},
					HasEntitlement:     true,
					UserID:             userID,
				}
				buf, err := json.Marshal(claimOfferResponse)
				if err != nil {
					log.Debug(err)
				}
				mockedResp := new(http.Response)
				mockedResp.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
				mockedResp.StatusCode = http.StatusOK
				samusSWSClientMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockedResp, nil)

				resp, s, e := b.GetPrimeEntitlement(context.TODO(), userID, offerID, "en")

				So(resp.OfferID, ShouldEqual, claimOfferResponse.OfferID)
				So(resp.UserID, ShouldEqual, claimOfferResponse.UserID)
				So(resp.HasEntitlement, ShouldEqual, true)
				So(s, ShouldEqual, http.StatusOK)
				So(e, ShouldBeNil)
			})

			Convey("When the call returns false", func() {
				primeEntitlement := dynamo.PrimeEntitlement{
					OfferID:      dynamo.OfferID(offerID),
					IsEntitled:   false,
					TwitchUserID: dynamo.TwitchUserID(userID),
				}

				primeEntitlementDynamoMock.On("Get", userID, offerID).Return(primeEntitlement, nil)

				resp, s, e := b.GetPrimeEntitlement(context.TODO(), userID, offerID, "en")

				So(resp.OfferID, ShouldEqual, offerID)
				So(resp.UserID, ShouldEqual, userID)
				So(resp.HasEntitlement, ShouldEqual, false)
				So(s, ShouldEqual, http.StatusOK)
				So(e, ShouldBeNil)
			})
		})

		Convey("UpdatePrimeEntitlement API", func() {
			Convey("When the call is successful", func() {
				primeEntitlementDynamoMock.On("Put", mock.Anything).Return(nil)

				err := b.UpdatePrimeEntitlement(context.TODO(), userID, offerID, "en")
				So(err, ShouldBeNil)
			})

			Convey("When the call is a failure", func() {
				primeEntitlementDynamoMock.On("Put", mock.Anything).Return(errors.New("Failed to save entitlement"))

				err := b.UpdatePrimeEntitlement(context.TODO(), userID, offerID, "en")
				So(err, ShouldNotBeNil)
			})
		})

		Convey("SetPrimeEntitlement API", func() {
			Convey("When the call is successful", func() {
				primeEntitlement := &dynamo.PrimeEntitlement{
					OfferID:      dynamo.OfferID(offerID),
					IsEntitled:   false,
					TwitchUserID: dynamo.TwitchUserID(userID),
				}

				primeEntitlementDynamoMock.On("Put", primeEntitlement).Return(nil)

				resp, s, e := b.SetPrimeEntitlement(context.TODO(), userID, offerID, "false")

				So(resp.OfferID, ShouldEqual, offerID)
				So(resp.UserID, ShouldEqual, userID)
				So(resp.HasEntitlement, ShouldEqual, false)
				So(s, ShouldEqual, http.StatusOK)
				So(e, ShouldBeNil)
			})

			Convey("When the call is a failure", func() {
				primeEntitlementDynamoMock.On("Put", mock.Anything).Return(errors.New("Failed to save entitlement"))

				_, _, err := b.SetPrimeEntitlement(context.TODO(), userID, offerID, "false")
				So(err, ShouldNotBeNil)
			})

			Convey("When the call has bad input", func() {
				primeEntitlementDynamoMock.On("Put", mock.Anything).Return(nil)

				_, _, err := b.SetPrimeEntitlement(context.TODO(), userID, offerID, "notMyEntitlement")

				So(err, ShouldNotBeNil)
			})
		})

		Convey("ClearOfferClaimCodeForUser API", func() {
			twitchUserID := "testUserId"
			marketplaceID := "testMarketplaceID"
			offerID := "testOfferID"
			csAgent := "testCSAgent"
			csContactID := "testCSContactID"

			Convey("When the offer's cleam code was successfully cleared", func() {
				expectedResponse := clients.ClearOfferClaimCodeForUserResponse{
					Success: true,
				}
				samusTProxClientMock.On("ClearOfferClaimCodeForUser", mock.Anything, mock.Anything).Return(&expectedResponse, http.StatusOK, nil)

				resp, s, e := b.ClearOfferClaimCodeForUser(context.TODO(), twitchUserID, marketplaceID, offerID, csAgent, csContactID)

				So(resp.Success, ShouldEqual, true)
				So(s, ShouldEqual, http.StatusOK)
				So(e, ShouldBeNil)
			})

			Convey("When the offer's claim code was not successfully cleared", func() {
				expectedResponse := clients.ClearOfferClaimCodeForUserResponse{
					Success: false,
				}
				samusTProxClientMock.On("ClearOfferClaimCodeForUser", mock.Anything, mock.Anything).Return(&expectedResponse, http.StatusOK, nil)

				resp, s, e := b.ClearOfferClaimCodeForUser(context.TODO(), twitchUserID, marketplaceID, offerID, csAgent, csContactID)

				So(resp.Success, ShouldEqual, false)
				So(s, ShouldEqual, http.StatusOK)
				So(e, ShouldBeNil)
			})

			Convey("When the call failed downstream", func() {
				samusTProxClientMock.On("ClearOfferClaimCodeForUser", mock.Anything, mock.Anything).Return(nil, http.StatusForbidden, nil)

				resp, s, e := b.ClearOfferClaimCodeForUser(context.TODO(), twitchUserID, marketplaceID, offerID, csAgent, csContactID)

				So(resp.Success, ShouldEqual, false)
				So(s, ShouldEqual, http.StatusForbidden)
				So(e, ShouldBeNil)
			})

			Convey("When the call failed in Gateway", func() {
				samusTProxClientMock.On("ClearOfferClaimCodeForUser", mock.Anything, mock.Anything).Return(nil, http.StatusInternalServerError, errors.New("Big problem"))

				resp, s, e := b.ClearOfferClaimCodeForUser(context.TODO(), twitchUserID, marketplaceID, offerID, csAgent, csContactID)

				So(resp.Success, ShouldEqual, false)
				So(s, ShouldEqual, http.StatusInternalServerError)
				So(e, ShouldNotBeNil)
			})
		})
	})

}
