package backend

import (
	"net/http"
	"strings"
	"time"

	"code.justin.tv/samus/gateway/promotion_string"
	log "github.com/sirupsen/logrus"
	"golang.org/x/net/context"
)

// Whitelisted accounts (usually QA)
// Can be used to check if an account is whitelisted, i.e.:
//     if (USERID_WHITELIST[myUserId]) { ... }
var USERID_WHITELIST = map[string]bool{
	"130802556": true,
	"129381151": true,
	"130731407": true,
	"130731473": true,
	"130731730": true,
	"130731783": true,
	"130732061": true,
}

type GetDynamicStringResponse struct {
	DynamicStringMap map[string]DynamicString `json:"dynamicStringMap"`
}

type DynamicString struct {
	ID             string `json:"id"`
	Text           string `json:"text"`
	ExternalUrl    string `json:"externalUrl"`
	IsExternalLink bool   `json:"isExternalLink"`
	IsCached       bool   `json:"isCached"`
}

func (b *Backend) GetDynamicStrings(ctx context.Context, stringIdsString string, countryCode string, locale string, userId string, dateOverride string) (*GetDynamicStringResponse, int, error) {
	stringsIds := strings.Split(stringIdsString, ",")
	now := time.Now().UTC().Format(time.RFC3339)
	stringMap := make(map[string]DynamicString)

	if USERID_WHITELIST[userId] && len(dateOverride) > 0 {
		now = dateOverride
	}

	for _, item := range stringsIds {

		// If userId is not a QA whitelisted account, use cache
		if !USERID_WHITELIST[userId] {
			promoString, promoStringFound := b.promoCache.GetPromotionString(promotion_string.PromotionStringCacheKey{
				Item:        item,
				CountryCode: countryCode,
				Locale:      locale,
			})

			if promoStringFound {
				log.Info("Using cached string \"", promoString.(DynamicString).Text, "\" for item ", item)
				stringMap[item] = promoString.(DynamicString)
				continue
			}
		}

		log.Info("Getting string: ", item, " for time: ", now)

		retString, err := b.promoStringDao.GetActiveString(item, now, countryCode, locale)

		if err != nil {
			log.WithError(err).Error("Error getting active string: ", item, "with error: ", err)
			return nil, http.StatusInternalServerError, err
		}

		if retString == nil {
			retStringDefault, err := b.promoStringDao.GetDefaultString(item, locale)

			if err != nil {
				log.WithError(err).Error("Error getting default string: ", item, "with error: ", err)
				return nil, http.StatusInternalServerError, err
			} else {
				retString = retStringDefault
			}
		}

		dynamicString := DynamicString{
			ID:             item,
			Text:           retString.String,
			ExternalUrl:    retString.ExternalUrl,
			IsExternalLink: retString.IsExternalLink,
			IsCached:       false,
		}

		stringMap[item] = dynamicString

		dynamicString.IsCached = true

		// If userId is not a QA whitelisted account, use cache
		if !USERID_WHITELIST[userId] {
			b.promoCache.SetPromotionString(promotion_string.PromotionStringCacheKey{
				Item:        item,
				CountryCode: countryCode,
				Locale:      locale,
			}, dynamicString)
		}

	}
	response := &GetDynamicStringResponse{
		DynamicStringMap: stringMap,
	}
	return response, http.StatusOK, nil
}
