package backend

import (
	"testing"
	"time"

	"code.justin.tv/samus/gateway/dynamo"

	"net/http"

	"code.justin.tv/common/config"
	"code.justin.tv/samus/gateway/clients/mocks"
	"code.justin.tv/samus/gateway/promotion_string"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
	"golang.org/x/net/context"
)

const PROMO_CACHE_EXPIRATION = 1 * time.Second

//TestSamusBackendAPIs Validates all Samus backend APIs
func TestSamusBackendAPIPromotions(t *testing.T) {

	Convey("Samus Backend API Promotions", t, func() {
		log.SetLevel(log.DebugLevel)

		samusSWSClientMock := new(mocks.HttpMock)
		dynamoMock := new(dynamo.IUserDaoMock)
		primeEntitlementDynamoMock := new(dynamo.IPrimeEntitlementDaoMock)
		promoStringDaoMock := new(promotion_string.IPromotionStringDaoMock)
		promoStringCache := promotion_string.NewCacheWithExpiration(PROMO_CACHE_EXPIRATION)

		entitlementsClientMock := new(mocks.HttpMock)

		b := Backend{stats: config.Statsd(),
			samusSWSClient:       samusSWSClientMock,
			userDao:              dynamoMock,
			promoStringDao:       promoStringDaoMock,
			entitlementsClient:   entitlementsClientMock,
			primeEntitlementsDao: primeEntitlementDynamoMock,
			promoCache:           promoStringCache,
		}

		Convey("When userId is whitelisted", func() {
			stringIds := "a"
			countryCode := "US"
			locale := "en"
			userId := "130802556"
			dateOverride := "2017-10-01T00:00:00Z"

			textA := "stringA"

			externalUrl := "url"

			promoStringDaoMock.On("GetActiveString", "a", dateOverride, countryCode, locale).Return(promotion_string.GetPromotionStringResponse{ExternalUrl: externalUrl, String: textA, IsExternalLink: true}, nil)

			resp, s, e := b.GetDynamicStrings(context.TODO(), stringIds, countryCode, locale, userId, dateOverride)

			stringA := resp.DynamicStringMap["a"]

			So(s, ShouldEqual, http.StatusOK)
			So(e, ShouldBeNil)

			// Ensure a second request is not cached for whitelisted
			resp2, s2, e2 := b.GetDynamicStrings(context.TODO(), stringIds, countryCode, locale, userId, dateOverride)

			stringA = resp2.DynamicStringMap["a"]

			So(stringA.ID, ShouldEqual, "a")
			So(stringA.Text, ShouldEqual, textA)
			So(stringA.ExternalUrl, ShouldEqual, externalUrl)
			So(stringA.IsExternalLink, ShouldEqual, true)
			So(stringA.IsCached, ShouldEqual, false)

			So(s2, ShouldEqual, http.StatusOK)
			So(e2, ShouldBeNil)
		})

		Convey("When promotion string is cached", func() {
			stringIds := "a"
			countryCode := "US"
			locale := "en"
			textA := "stringA"
			externalUrl := "url"

			promoStringDaoMock.On("GetActiveString", "a", mock.Anything, countryCode, locale).Return(promotion_string.GetPromotionStringResponse{ExternalUrl: externalUrl, String: textA, IsExternalLink: true}, nil)

			resp, s, e := b.GetDynamicStrings(context.TODO(), stringIds, countryCode, locale, "123456", "")

			stringA := resp.DynamicStringMap["a"]

			So(stringA.IsCached, ShouldEqual, false)

			So(s, ShouldEqual, http.StatusOK)
			So(e, ShouldBeNil)

			// Check cached on 2nd try
			resp2, s2, e2 := b.GetDynamicStrings(context.TODO(), stringIds, countryCode, locale, "123456", "")

			stringA = resp2.DynamicStringMap["a"]

			So(stringA.IsCached, ShouldEqual, true)

			So(s2, ShouldEqual, http.StatusOK)
			So(e2, ShouldBeNil)

			// Make sure the cache expires
			time.Sleep(2 * PROMO_CACHE_EXPIRATION)

			resp3, s3, e3 := b.GetDynamicStrings(context.TODO(), stringIds, countryCode, locale, "123456", "")

			stringA = resp3.DynamicStringMap["a"]

			So(stringA.IsCached, ShouldEqual, false)

			So(s3, ShouldEqual, http.StatusOK)
			So(e3, ShouldBeNil)

		})

	})
}
