package backend

import (
	"bytes"
	"fmt"

	log "github.com/sirupsen/logrus"
	"golang.org/x/net/context"

	"strings"

	"encoding/json"
	"net/http"

	nitro "code.justin.tv/samus/nitro/rpc"

	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/samus/gateway/cache"
	client "code.justin.tv/samus/gateway/client"
	"code.justin.tv/samus/gateway/clients"
	"code.justin.tv/samus/gateway/dynamo"
	internal_models "code.justin.tv/samus/gateway/internal/models"
	"github.com/pkg/errors"
)

// const used in HTTP messages to Samus
const (
	SWSHEADER   = "x-amzn-sws-rest-service"
	SSCSHEADER  = "SamusSubscriptionCreditService"
	SOSHEADER   = "SamusOfferService"
	SORDSHEADER = "SamusOrderingService"
	SISHEADER   = "SamusInventoryService"
)

const ClientId = "TWITCH"

// StatusResponse struct for prime status
type StatusResponse struct {
	UserID      string `json:"twitchUserId"`
	TwitchPrime bool   `json:"twitchPrime"`
}

type SettingsResponse struct {
	UserID                string `json:"twitchUserId"`
	IsSubscriptionMessage bool   `json:"isSubscriptionMessage"`
}

type OfferBlacklistResponse struct {
	Blacklist map[string][]string `json:"offerBlacklist"`
}

type CurrentOffersResponse struct {
	CurrentPrimeOffers []client.OfferWithNoClaimData `json:"offerList"`
}

type PrimeEntitlementResponse struct {
	OfferClaimMetadata client.OfferClaimMetadata `json:"offerClaimCode"`
	ClaimMethod        string                    `json:"claimMethod"`
	OfferID            string                    `json:"offerId"`
	OfferTitle         string                    `json:"offerTitle"`
	OfferClaimData     string                    `json:"offerClaimData"`
	ClaimInstruction   string                    `json:"contentClaimInstruction"`
	HasEntitlement     bool                      `json:"hasEntitlement"`
	UserID             string                    `json:"twitchUserId"`
}

type PlaceOrderResponse struct {
	OrderID string `json:"orderId"`
}
type GetOrdersByCustomerResponse struct {
	OrderDocuments []client.OrderDocument `json:"orderDocuments"`
	NextToken      string                 `json:"nextToken"`
}

type ListInventoryResponse struct {
	Inventory client.Inventory `json:"inventory"`
	NextToken string           `json:"nextToken"`
}

// Status retrieves Prime status from Nitro
func (b *Backend) Status(ctx context.Context, userID string) (*StatusResponse, int, error) {
	log.Debug("Calling Nitro for Prime Status for userId: ", userID)

	hasPrime, err := b.fetchPrimeStatusFromSamus(ctx, userID)
	if err != nil {
		return nil, 0, err
	}

	statusResponse := StatusResponse{
		TwitchPrime: hasPrime,
		UserID:      userID,
	}

	return &statusResponse, 0, nil
}

func (b *Backend) fetchPrimeStatusFromSamus(ctx context.Context, userID string) (bool, error) {
	log.Debug("Fetching Prime Status for userId: ", userID)

	req := &nitro.GetPremiumStatusesRequest{
		TwitchUserID: userID,
	}

	resp, err := b.nitroClient.GetPremiumStatuses(ctx, req)
	if err != nil {
		return false, errors.Wrapf(err, "[fetchPrimeStatusFromSamus] Error calling Nitro::GetPremiumStatuses for userID: %v", userID)
	}

	return resp.HasPrime, nil
}

// Balance retrieves user's available credit balance
func (b *Backend) Balance(ctx context.Context, userID string) (*BalanceResponse, int, error) {
	log.Debug("Checking Credit Balance for userId:", userID)
	req, err := b.samusSWSClient.NewRequest("GET", fmt.Sprintf("/samus/users/%s/subscription_credit/balance", userID), nil)
	if err != nil {
		return nil, 0, err
	}
	req.Header.Set(SWSHEADER, SSCSHEADER)
	balanceResponse := BalanceResponse{}
	s, err := sendRequestAndParseResponse(b.samusSWSClient, ctx, req, &balanceResponse)
	if err != nil {
		return nil, s, err
	}
	return &balanceResponse, 0, nil
}

func (b *Backend) GetSettings(ctx context.Context, userID string) (*SettingsResponse, int, error) {
	user, err := b.userDao.GetOrCreate(dynamo.UserId(userID))
	if err != nil {
		return nil, http.StatusInternalServerError, err
	}

	return &SettingsResponse{
		UserID:                string(user.Id),
		IsSubscriptionMessage: user.IsSubscriptionMessage,
	}, http.StatusOK, nil
}

func (b *Backend) UpdateSettings(ctx context.Context, userID string, isSubscriptionMessage bool) (*SettingsResponse, int, error) {
	user, err := b.userDao.GetOrCreate(dynamo.UserId(userID))
	if err != nil {
		return nil, http.StatusInternalServerError, err
	}
	if user.IsSubscriptionMessage != isSubscriptionMessage {
		user.IsSubscriptionMessage = isSubscriptionMessage
		err := b.userDao.Put(user)
		if err != nil {
			return nil, http.StatusInternalServerError, err
		}
	}

	return &SettingsResponse{
		UserID:                string(user.Id),
		IsSubscriptionMessage: user.IsSubscriptionMessage,
	}, http.StatusOK, nil
}

func (b *Backend) GetOfferBlacklist(ctx context.Context) (*OfferBlacklistResponse, int, error) {
	// I am hardcoding this offer list as for the next list, it will be passed through to SamusOfferService and blacklisted there
	return &OfferBlacklistResponse{
		Blacklist: map[string][]string{
			"9aadfcdd-da46-8d47-07f4-ee8885482da3": {"tw"}, // poe
			"c4adac90-2917-9d86-078a-425791d5e7ca": {"af", "bo", "do", "ec", "sv", "gt", "hn", "ni", "pa", "py", "uy", "ve", "pr", "gu", "vi", "mp", "as", "im", "je",
				"gg", "ax", "al", "ad", "ao", "ai", "aq", "ag", "am", "aw", "az", "bs", "bb", "by", "bz", "bj", "bm", "bt", "ba", "bw", "bv", "io", "vg", "bn", "bf", "bi",
				"kh", "cm", "cv", "ky", "cf", "td", "cx", "cc", "km", "cg", "cd", "ck", "ci", "cw", "dj", "dm", "gq", "er", "et", "fk", "fo", "fj", "gf", "pf", "tf", "gp",
				"mq", "ga", "gm", "ge", "gh", "gi", "gl", "gd", "gn", "gw", "gy", "ht", "hm", "iq", "jm", "jo", "ke", "ki", "xk", "kg", "la", "ls", "lr", "ly", "mo", "mk",
				"mg", "mw", "mv", "ml", "mh", "mu", "yt", "fm", "md", "mc", "mn", "me", "ms", "mz", "mm", "na", "nr", "np", "nc", "ne", "ng", "nu", "nf", "pw", "ps", "pg",
				"pn", "re", "rw", "sh", "kn", "lc", "pm", "vc", "ws", "sm", "st", "sn", "rs", "sc", "sl", "sb", "so", "gs", "ss", "lk", "bl", "mf", "sr", "sj", "sz", "tw",
				"tj", "tz", "tl", "tg", "tk", "to", "tm", "tc", "tv", "ug", "um", "uz", "vu", "va", "vn", "wf", "eh", "zm", "zw"}, // nba
			"6aae2348-806c-1d40-efaf-1e094a9f6f96": {"us"}, // test offer
			// Overwatch headset
			"8cafe0ac-0382-ea7d-b9d5-c8c68849e3b4": {"af", "ax", "al", "dz", "as", "ad", "ao", "ai", "aq", "ag", "ar", "am", "aw", "au", "at", "az", "bs", "bh", "bd",
				"bb", "by", "be", "bz", "bj", "bm", "bt", "bo", "ba", "bw", "bv", "br", "vg", "io", "bn", "bg", "bf", "bi", "kh", "cm", "ca", "cv", "ky", "cf", "td", "cl",
				"cn", "hk", "mo", "cx", "cc", "co", "km", "cg", "cd", "ck", "cr", "ci", "hr", "cu", "cy", "cz", "dk", "dj", "dm", "do", "ec", "eg", "sv", "gq", "er", "ee",
				"et", "fk", "fo", "fj", "fi", "fr", "gf", "pf", "tf", "ga", "gm", "ge", "de", "gh", "gi", "gr", "gl", "gd", "gp", "gu", "gt", "gg", "gn", "gw", "gy", "ht",
				"hm", "va", "hn", "hu", "is", "in", "id", "ir", "iq", "ie", "im", "il", "it", "jm", "jp", "je", "jo", "kz", "ke", "ki", "kp", "kr", "kw", "kg", "la", "lv",
				"lb", "ls", "lr", "ly", "li", "lt", "lu", "mk", "mg", "mw", "my", "mv", "ml", "mt", "mh", "mq", "mr", "mu", "yt", "mx", "fm", "md", "mc", "mn", "me", "ms",
				"ma", "mz", "mm", "na", "nr", "np", "nl", "an", "nc", "nz", "ni", "ne", "ng", "nu", "nf", "mp", "no", "om", "pk", "pw", "ps", "pa", "pg", "py", "pe", "ph",
				"pn", "pl", "pt", "pr", "qa", "re", "ro", "ru", "rw", "bl", "sh", "kn", "lc", "mf", "pm", "vc", "ws", "sm", "st", "sa", "sn", "rs", "sc", "sl", "sg", "sk",
				"si", "sb", "so", "za", "gs", "ss", "es", "lk", "sd", "sr", "sj", "sz", "se", "ch", "sy", "tw", "tj", "tz", "th", "tl", "tg", "tk", "to", "tt", "tn", "tr",
				"tm", "tc", "tv", "ug", "ua", "ae", "gb", "um", "uy", "uz", "vu", "ve", "vn", "vi", "wf", "eh", "ye", "zm", "zw"}, // all countries but us
			// COD Merch Deal $5
			"e8b00249-32a5-5e4a-14f1-94f09ea3855e": {"af", "ax", "al", "dz", "as", "ad", "ao", "ai", "aq", "ag", "ar", "am", "aw", "au", "at", "az", "bs", "bh", "bd",
				"bb", "by", "be", "bz", "bj", "bm", "bt", "bo", "ba", "bw", "bv", "br", "vg", "io", "bn", "bg", "bf", "bi", "kh", "cm", "ca", "cv", "ky", "cf", "td", "cl",
				"cn", "hk", "mo", "cx", "cc", "co", "km", "cg", "cd", "ck", "cr", "ci", "hr", "cu", "cy", "cz", "dk", "dj", "dm", "do", "ec", "eg", "sv", "gq", "er", "ee",
				"et", "fk", "fo", "fj", "fi", "fr", "gf", "pf", "tf", "ga", "gm", "ge", "de", "gh", "gi", "gr", "gl", "gd", "gp", "gu", "gt", "gg", "gn", "gw", "gy", "ht",
				"hm", "va", "hn", "hu", "is", "in", "id", "ir", "iq", "ie", "im", "il", "it", "jm", "jp", "je", "jo", "kz", "ke", "ki", "kp", "kr", "kw", "kg", "la", "lv",
				"lb", "ls", "lr", "ly", "li", "lt", "lu", "mk", "mg", "mw", "my", "mv", "ml", "mt", "mh", "mq", "mr", "mu", "yt", "mx", "fm", "md", "mc", "mn", "me", "ms",
				"ma", "mz", "mm", "na", "nr", "np", "nl", "an", "nc", "nz", "ni", "ne", "ng", "nu", "nf", "mp", "no", "om", "pk", "pw", "ps", "pa", "pg", "py", "pe", "ph",
				"pn", "pl", "pt", "pr", "qa", "re", "ro", "ru", "rw", "bl", "sh", "kn", "lc", "mf", "pm", "vc", "ws", "sm", "st", "sa", "sn", "rs", "sc", "sl", "sg", "sk",
				"si", "sb", "so", "za", "gs", "ss", "es", "lk", "sd", "sr", "sj", "sz", "se", "ch", "sy", "tw", "tj", "tz", "th", "tl", "tg", "tk", "to", "tt", "tn", "tr",
				"tm", "tc", "tv", "ug", "ua", "ae", "gb", "um", "uy", "uz", "vu", "ve", "vn", "vi", "wf", "eh", "ye", "zm", "zw"}, // all countries but us
		},
	}, 200, nil
}

// GetCurrentOffers API
func (b *Backend) GetCurrentOffers(ctx context.Context, userID string, countryCode string, locale string, dateOverride string, clientID string) (*CurrentOffersResponse, int, error) {
	if locale == "" {
		locale = "en"
	}
	if countryCode == "" {
		countryCode = "US"
	}

	shouldUseCache := dateOverride == "" && userID == ""
	logger := log.WithFields(log.Fields{
		"userID":         userID,
		"locale":         locale,
		"countryCode":    countryCode,
		"dateOverride":   dateOverride,
		"shouldUseCache": shouldUseCache,
	})

	logger.Info("Getting Offers... ")

	if shouldUseCache {
		offersCacheKey := cache.OffersCacheKey{
			CountryCode: strings.ToLower(countryCode),
			Locale:      strings.ToLower(locale),
		}
		offers, offersFound := b.offersCache.GetOffers(offersCacheKey)

		if offersFound {
			logger.Info("Using cached offers for input ", offersCacheKey)
			cachedOffers := offers.(CurrentOffersResponse)
			return &cachedOffers, 0, nil
		}
	}

	url := fmt.Sprintf("/samus/offers/%s?locale=%s&userId=%s", countryCode, locale, userID)

	if len(dateOverride) > 0 {
		url = fmt.Sprintf("/samus/offers/%s?locale=%s&userId=%s&dateOverride=%s", countryCode, locale, userID, dateOverride)
	}

	logMessage := fmt.Sprintf("Calling API: %s", url)

	log.Info(logMessage)

	req, err := b.samusSWSClient.NewRequest("GET", url, nil)
	if err != nil {
		return nil, 0, err
	}
	req.Header.Set(SWSHEADER, SOSHEADER)
	currentOffersResponse := CurrentOffersResponse{}
	s, err := sendRequestAndParseResponse(b.samusSWSClient, ctx, req, &currentOffersResponse)
	if err != nil {
		return nil, s, err
	}

	if shouldUseCache {
		offersCacheKey := cache.OffersCacheKey{
			CountryCode: strings.ToLower(countryCode),
			Locale:      strings.ToLower(locale),
		}
		b.offersCache.SetOffers(offersCacheKey, currentOffersResponse)
	}

	return &currentOffersResponse, 0, nil
}

// GetCurrentOffers API
func (b *Backend) GetCurrentOffersForUser(ctx context.Context, userID string, countryCode string, locale string, dateOverride string, clientID string) (*client.CurrentPrimeOffersResponse, int, error) {
	if locale == "" {
		locale = "en"
	}
	if countryCode == "" {
		countryCode = "US"
	}

	logger := log.WithFields(log.Fields{
		"userID":       userID,
		"locale":       locale,
		"countryCode":  countryCode,
		"dateOverride": dateOverride,
		"clientID":     clientID,
	})
	logger.Info("Getting Offers for user in country... ")

	getCurrentOffersForUserInCountryRequest := clients.GetCurrentOffersForUserInCountryRequest{
		TUID:         userID,
		CountryCode:  countryCode,
		Locale:       locale,
		DateOverride: dateOverride,
		ClientID:     clientID,
	}
	gcofuicResponse, s, err := b.samusTProxClient.GetCurrentOffersForUserInCountry(ctx, &getCurrentOffersForUserInCountryRequest)
	if err != nil || s != http.StatusOK {
		logger.Error("[GetCurrentOffersForUserInCountry] ", "GetCurrentOffersForUserInCountry failed: ", err.Error())
		return nil, s, err
	}

	var cpoResponse client.CurrentPrimeOffersResponse
	cpoResponse.PrimeOffers = make([]client.PrimeOffer, len(gcofuicResponse.Offers))

	for i, offer := range gcofuicResponse.Offers {
		cpoResponse.PrimeOffers[i] = client.PrimeOffer{
			OfferWithClaimData: offer,
			ClaimHint:          gcofuicResponse.ClaimStatuses[offer.OfferID].ClaimHint,
		}
	}

	return &cpoResponse, s, nil
}

// GetCurrentOffersWithEligibility
func (b *Backend) GetCurrentOffersWithEligibility(ctx context.Context, userID string, countryCode string, locale string, dateOverride string, clientID string) (*client.GetCurrentOffersWithEligibilityResponse, int, error) {
	if locale == "" {
		locale = "en"
	}
	if countryCode == "" {
		countryCode = "US"
	}

	logger := log.WithFields(log.Fields{
		"userID":       userID,
		"locale":       locale,
		"countryCode":  countryCode,
		"dateOverride": dateOverride,
	})

	/** Retrieve offers from SOS GetCurrentOffersByCountry **/

	currentOffersResponse, s, err := b.getCurrentOffersWithoutClaimData(ctx, userID, countryCode, locale, dateOverride)
	if err != nil {
		logger.Error("[GetCurrentOffersWithEligibility] ", "SamusOfferService failed: ", err.Error())
		return nil, s, err
	}

	catalogOfferList := filterCatalogOffers(currentOffersResponse.CurrentPrimeOffers)

	/** If the Twitch user is logged in, try to retrieve eligibility data for each offer **/

	if len(userID) > 0 {
		// We only need to get elgibility data for FGWP and code offers, since all other offers link to external ODPs
		fgwpAndCodeOfferIds := getFGWPAndCodeOfferIds(catalogOfferList)

		batchGetEligibilityStatusResponse, s, err := b.getOfferEligibilityData(ctx, userID, fgwpAndCodeOfferIds, dateOverride)

		if batchGetEligibilityStatusResponse == nil || len(batchGetEligibilityStatusResponse.Results) == 0 || s == http.StatusBadRequest {
			logger.Info("[GetCurrentOffersWithEligibility] ", "No linked Amazon account or empty response found for TUID", userID)
		} else if err != nil || s != http.StatusOK {
			logger.Error("[GetCurrentOffersWithEligibility] ", "TwitchProxyService failed: ", err.Error())
		} else {

			/** Parse eligibility response and map eligibility data to each offer **/

			catalogOfferIDToEligibilityMap := buildOfferEligibilityMap(batchGetEligibilityStatusResponse.Results)

			currentOffersWithEligibilityResponse := buildOffersResponseWithEligibility(catalogOfferIDToEligibilityMap, catalogOfferList)

			return &currentOffersWithEligibilityResponse, 0, nil
		}
	}

	/** If the Twitch user is not logged in or TProx threw an error, return offer list without eligibility data **/

	currentOffersWithEligibilityResponse := buildOffersResponseWithoutEligibility(catalogOfferList)

	return &currentOffersWithEligibilityResponse, 0, nil
}

func (b *Backend) getCurrentOffersWithoutClaimData(ctx context.Context, userID string, countryCode string, locale string, dateOverride string) (*CurrentOffersResponse, int, error) {
	url := fmt.Sprintf("/samus/offers/%s?locale=%s&userId=%s", countryCode, locale, userID)

	if len(dateOverride) > 0 {
		url = fmt.Sprintf("/samus/offers/%s?locale=%s&userId=%s&dateOverride=%s", countryCode, locale, userID, dateOverride)
	}

	req, err := b.samusSWSClient.NewRequest("GET", url, nil)
	if err != nil {
		return nil, 0, err
	}

	req.Header.Set(SWSHEADER, SOSHEADER)
	currentOffersResponse := CurrentOffersResponse{}
	s, err := sendRequestAndParseResponse(b.samusSWSClient, ctx, req, &currentOffersResponse)

	return &currentOffersResponse, s, err
}

func (b *Backend) getOfferEligibilityData(ctx context.Context, userID string, offerIDList []string, dateOverride string) (*clients.BatchGetEligibilityStatusResponse, int, error) {
	batchGetEligibilityStatusTwitchRequest := clients.BatchGetEligibilityStatusTwitchRequest{
		TwitchUserID: userID,
		OfferIDs:     offerIDList,
		DateOverride: dateOverride,
	}

	batchGetEligibilityStatusResponse, s, err := b.samusTProxClient.BatchGetEligibilityStatus(ctx, &batchGetEligibilityStatusTwitchRequest)
	if err != nil {
		return nil, s, err
	}

	return batchGetEligibilityStatusResponse, s, err
}

func buildOffersResponseWithEligibility(catalogOfferIDToEligibilityMap map[string]clients.GetEligibilityStatusResponse, catalogOfferList []client.OfferWithNoClaimData) client.GetCurrentOffersWithEligibilityResponse {
	currentOffersWithEligibilityResponse := client.GetCurrentOffersWithEligibilityResponse{
		CurrentOffers: make([]client.OfferWithEligibility, len(catalogOfferList)),
	}

	for i, catalogOffer := range catalogOfferList {
		currentOffersWithEligibilityResponse.CurrentOffers[i] = client.OfferWithEligibility{
			OfferWithNoClaimData: catalogOffer,
			OfferEligibility:     parseOfferEligibility(catalogOfferIDToEligibilityMap[catalogOffer.CatalogOfferID]),
		}
	}

	return currentOffersWithEligibilityResponse
}

func buildOffersResponseWithoutEligibility(catalogOfferList []client.OfferWithNoClaimData) client.GetCurrentOffersWithEligibilityResponse {
	currentOffersWithEligibilityResponse := client.GetCurrentOffersWithEligibilityResponse{
		CurrentOffers: make([]client.OfferWithEligibility, len(catalogOfferList)),
	}

	for i, catalogOffer := range catalogOfferList {
		currentOffersWithEligibilityResponse.CurrentOffers[i] = client.OfferWithEligibility{
			OfferWithNoClaimData: catalogOffer,
		}
	}

	return currentOffersWithEligibilityResponse
}

// Parse offer list and return list of offers that have an entry in the offer catalog
func filterCatalogOffers(offerList []client.OfferWithNoClaimData) []client.OfferWithNoClaimData {
	catalogOfferList := make([]client.OfferWithNoClaimData, 0, len(offerList))

	for _, offer := range offerList {
		if len(offer.CatalogOfferID) > 0 {
			catalogOfferList = append(catalogOfferList, offer)
		}
	}

	return catalogOfferList
}

// Parse list of offers and return list of catalog offer IDs corresponding to the FGWP and code offers in the list
func getFGWPAndCodeOfferIds(offerList []client.OfferWithNoClaimData) []string {
	offerIdList := make([]string, 0, len(offerList))

	for _, offer := range offerList {
		if offer.ContentDeliveryMethod == client.DirectEntitlement || offer.ContentDeliveryMethod == client.ClaimCode { // DIRECT_ENTITLEMENT || CLAIM_CODE
			offerIdList = append(offerIdList, offer.CatalogOfferID)
		}
	}

	return offerIdList
}

// Parse list of eligibility responses and return a map of all responses using the corresponding catalog offer IDs as keys
func buildOfferEligibilityMap(eligibilityList []clients.GetEligibilityStatusResponse) map[string]clients.GetEligibilityStatusResponse {
	catalogOfferIDToEligibilityMap := make(map[string]clients.GetEligibilityStatusResponse, len(eligibilityList))
	for _, eligibilityResponse := range eligibilityList {
		catalogOfferIDToEligibilityMap[eligibilityResponse.OfferID] = eligibilityResponse
	}

	return catalogOfferIDToEligibilityMap
}

// Parses the eligibility response from SCES for each offer
// Maps eligiblility data for each rule defined by SCES to a matching eligibility field
func parseOfferEligibility(eligibilityResponse clients.GetEligibilityStatusResponse) client.OfferEligibility {
	eligibility := client.OfferEligibility{
		OfferState: eligibilityResponse.OfferState,
		CanClaim:   eligibilityResponse.Eligible,
	}

	for _, priorEntitlementRecord := range eligibilityResponse.PriorEntitlements {
		if priorEntitlementRecord.EntitlementStatus == clients.EntitlementEntitled {
			eligibility.IsClaimed = true
			break
		}
	}

	for _, eligibilityResult := range eligibilityResponse.EligibilityResults {
		switch eligibilityResult.RuleName {
		case clients.ClaimLimitRule: // CLAIM_LIMIT
			eligibility.ClaimLimitRule = eligibilityResult.Eligible
		case clients.PrimeGamingRule: // PRIME_GAMING
			eligibility.PrimeGamingRule = eligibilityResult.Eligible
		case clients.LinkedAccountRule: // LINKED_ACCOUNT
			eligibility.LinkedAccountRule = eligibilityResult.Eligible
		case clients.MarketplaceRule: // MARKETPLACE
			eligibility.MarketplaceRule = eligibilityResult.Eligible
		case clients.OfferWindowRule: // OFFER_WINDOW
			eligibility.OfferWindowRule = eligibilityResult.Eligible
		}
	}

	return eligibility
}

// ClaimOffer API
func (b *Backend) ClaimOffer(ctx context.Context, userID string, offerID string, locale string, jsonStr []byte) (*PrimeEntitlementResponse, int, error) {
	req, err := b.samusSWSClient.NewRequest("POST", fmt.Sprintf("samus/users/%s/offerCode/%s?locale=%s", userID, offerID, locale), bytes.NewBuffer(jsonStr))
	if err != nil {
		return nil, 0, err
	}

	log.Infof("Attempting to ClaimOffer: userID: %s, offerID: %s, locale: %s\nContext: [%+v]", userID, offerID, locale, ctx)

	req.Header.Set(SWSHEADER, SOSHEADER)
	claimOfferResponse := PrimeEntitlementResponse{}
	resp, err := b.samusSWSClient.Do(ctx, req, twitchhttp.ReqOpts{})

	// Check if there was an error on the wire call
	if err != nil {
		if resp != nil {
			closeBody(resp.Body)
		}
		return nil, http.StatusInternalServerError, err
	}

	// If call was okay but it was not a 200
	if resp.StatusCode != http.StatusOK {
		var coralError internal_models.CoralError
		err = json.NewDecoder(resp.Body).Decode(&coralError)
		if err != nil {
			log.WithError(err).Error("Error Decoding response to interface :", err)
			return nil, http.StatusInternalServerError, err
		}
		status := http.StatusBadGateway
		if resp.StatusCode == http.StatusBadRequest {
			status = http.StatusBadRequest
		}
		if resp.StatusCode == http.StatusNotFound {
			status = http.StatusNotFound
		}

		return nil, status, errors.New(coralError.ErrorCode)
	}

	err = json.NewDecoder(resp.Body).Decode(&claimOfferResponse)
	if err != nil {
		log.WithError(err).Error("Error Decoding response to interface :", err)
		return nil, http.StatusInternalServerError, err
	}

	claimOfferResponse.HasEntitlement = true
	claimOfferResponse.UserID = userID

	log.Infof("Successfull call to ClaimOffer: userID: %s, offerID: %s, locale: %s\nContext: [%+v]", userID, offerID, locale, ctx)

	return &claimOfferResponse, 0, nil
}

// PlaceOrder API
func (b *Backend) PlaceOrder(ctx context.Context, userID string, offerID string, idempotenceKey string, attributionChannel string, dateOverride string, clientID string) (*clients.PlaceOrderResponse, int, error) {

	placeOrderTwitchRequest := clients.PlaceOrderTwitchRequest{
		SelectedTwitchUserId: userID,
		OfferID:              offerID,
		IdempotenceKey:       idempotenceKey,
		AttributionChannel:   attributionChannel,
		DateOverride:         dateOverride,
	}

	placeOrderResponse, s, err := b.samusTProxClient.PlaceOrder(ctx, &placeOrderTwitchRequest)
	if err != nil {
		return nil, s, err
	}

	return placeOrderResponse, s, err
}

// GetOrdersByCustomer API
func (b *Backend) GetOrdersByCustomer(ctx context.Context, userID string, nextToken string, pageSize int, offerID string, orderID string, clientID string) (*clients.GetOrdersByCustomerResponse, int, error) {

	getOrdersByCustomerTwitchRequest := clients.GetOrdersByCustomerTwitchRequest{
		TwitchUserID: userID,
		NextToken:    nextToken,
		PageSize:     pageSize,
		OfferID:      offerID,
		OrderID:      orderID,
	}

	getOrdersByCustomerResponse, s, err := b.samusTProxClient.GetOrdersByCustomer(ctx, &getOrdersByCustomerTwitchRequest)
	if err != nil {
		return nil, s, err
	}

	return getOrdersByCustomerResponse, s, err
}

// ListInventory API
func (b *Backend) ListInventory(ctx context.Context, userID string, amazonCustomerID string, entitlementAccountType string, itemIds []string, nextToken string, maxResults int, entitlementStatusFilters []string, clientID string) (*clients.ListInventoryResponse, int, error) {

	listInventoryTwitchRequest := clients.ListInventoryTwitchRequest{
		TwitchUserID:             userID,
		AmazonCustomerID:         amazonCustomerID,
		EntitlementAccountType:   entitlementAccountType,
		ItemIds:                  itemIds,
		NextToken:                nextToken,
		MaxResults:               maxResults,
		EntitlementStatusFilters: entitlementStatusFilters,
	}

	listInventoryResponse, s, err := b.samusTProxClient.ListInventory(ctx, &listInventoryTwitchRequest)
	if err != nil {
		return nil, s, err
	}

	return listInventoryResponse, s, err
}
