package backend

import (
	"errors"
	"testing"

	"bytes"
	"io/ioutil"
	"net/http"

	"encoding/json"

	"code.justin.tv/common/config"
	"code.justin.tv/samus/gateway/cache"
	gatewayClient "code.justin.tv/samus/gateway/client"
	clients "code.justin.tv/samus/gateway/clients"
	"code.justin.tv/samus/gateway/clients/mocks"
	"code.justin.tv/samus/gateway/dynamo"
	"code.justin.tv/samus/gateway/promotion_string"
	nitro "code.justin.tv/samus/nitro/rpc"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
	"golang.org/x/net/context"
)

const userID = "11111"
const offerID = "OFFER-ID-UUID"
const catalogOfferID = "CATALOG-OFFER-ID-UUID"
const broadcasterID = "BROADCASTER-ID"
const TransactionID = "amzn1.monad3.1468959699787.1.3.Samus.1607122017dHoRxmHeTGSVZ1J/drm9xA.RA3n3IeeS8KGCR6wWuQS9Q.2"

//TestSamusBackendAPIs Validates all Samus backend APIs
func TestSamusBackendAPIs(t *testing.T) {

	Convey("Samus Backend APIs", t, func() {
		log.SetLevel(log.DebugLevel)

		samusSWSClientMock := new(mocks.HttpMock)
		samusTProxClientMock := new(clients.SamusTProxClientMock)
		dynamoMock := new(dynamo.IUserDaoMock)
		primeEntitlementDynamoMock := new(dynamo.IPrimeEntitlementDaoMock)
		promoStringDaoMock := new(promotion_string.IPromotionStringDaoMock)
		promoStringCacheMock := new(promotion_string.IPromotionStringCacheMock)
		offersCacheMock := new(cache.IOffersCacheMock)
		entitlementsClientMock := new(mocks.HttpMock)
		nitroClientMock := new(mocks.Nitro)

		promoStringCacheMock.On("GetPromotionString", mock.Anything).Return(mock.Anything, false)
		offersCacheMock.On("GetOffers", mock.Anything).Return(mock.Anything, false)
		offersCacheMock.On("SetOffers", mock.Anything, mock.Anything)

		b := Backend{stats: config.Statsd(),
			samusSWSClient:       samusSWSClientMock,
			samusTProxClient:     samusTProxClientMock,
			userDao:              dynamoMock,
			promoStringDao:       promoStringDaoMock,
			entitlementsClient:   entitlementsClientMock,
			primeEntitlementsDao: primeEntitlementDynamoMock,
			promoCache:           promoStringCacheMock,
			offersCache:          offersCacheMock,
			nitroClient:          nitroClientMock,
		}

		Convey("When Status API is successful", func() {

			premiumStatusResponse := &nitro.GetPremiumStatusesResponse{
				HasPrime: true,
			}

			req := &nitro.GetPremiumStatusesRequest{
				TwitchUserID: userID,
			}

			nitroClientMock.On("GetPremiumStatuses", context.Background(), req).Return(premiumStatusResponse, nil)

			resp, s, e := b.Status(context.Background(), userID)

			So(resp.TwitchPrime, ShouldEqual, true)
			So(resp.UserID, ShouldEqual, userID)
			So(s, ShouldEqual, 0)
			So(e, ShouldBeNil)
		})

		Convey("When fetchPrimeStatusFromSamus API is successful", func() {

			premiumStatusResponse := &nitro.GetPremiumStatusesResponse{
				HasPrime: true,
			}

			req := &nitro.GetPremiumStatusesRequest{
				TwitchUserID: userID,
			}

			nitroClientMock.On("GetPremiumStatuses", context.Background(), req).Return(premiumStatusResponse, nil)

			hasPrime, e := b.fetchPrimeStatusFromSamus(context.Background(), userID)

			So(hasPrime, ShouldEqual, true)
			So(e, ShouldBeNil)
		})

		Convey("When GetSettings API is successful", func() {
			userId := dynamo.UserId(userID)

			dynamoMock.On("GetOrCreate", userId).Return(dynamo.User{Id: userId, IsSubscriptionMessage: true}, nil)

			resp, s, e := b.GetSettings(context.TODO(), userID)

			So(resp.IsSubscriptionMessage, ShouldEqual, true)
			So(resp.UserID, ShouldEqual, userID)
			So(s, ShouldEqual, http.StatusOK)
			So(e, ShouldBeNil)
		})

		Convey("When UpdateSettings API has the same settings", func() {
			userId := dynamo.UserId(userID)

			dynamoMock.On("GetOrCreate", userId).Return(dynamo.User{Id: userId, IsSubscriptionMessage: true}, nil)

			resp, s, e := b.UpdateSettings(context.TODO(), userID, true)

			So(resp.IsSubscriptionMessage, ShouldEqual, true)
			So(resp.UserID, ShouldEqual, userID)
			So(s, ShouldEqual, http.StatusOK)
			So(e, ShouldBeNil)
		})

		Convey("When UpdateSettings API has different settings", func() {
			userId := dynamo.UserId(userID)

			user := dynamo.User{Id: userId, IsSubscriptionMessage: true}

			dynamoMock.On("GetOrCreate", userId).Return(dynamo.User{Id: userId, IsSubscriptionMessage: false}, nil)
			dynamoMock.On("Put", &user).Return(nil)

			resp, s, e := b.UpdateSettings(context.TODO(), userID, true)

			So(resp.IsSubscriptionMessage, ShouldEqual, true)
			So(resp.UserID, ShouldEqual, userID)
			So(s, ShouldEqual, http.StatusOK)
			So(e, ShouldBeNil)
		})

		Convey("When GetCurrentOffers API is successful", func() {

			currentPrimeOffers := []gatewayClient.OfferWithNoClaimData{}
			currentPrimeOffer := gatewayClient.OfferWithNoClaimData{
				ContentCategories:       []string{},
				ExternalOfferUri:        "externalOfferUri",
				PublisherName:           "publisherName",
				ApplicableGame:          "applicableGame",
				ContentDeliveryMethod:   "claimCode",
				Assets:                  []gatewayClient.OfferAsset{},
				ContentClaimInstruction: "claimIntructions",
				Priority:                1,
				OfferDescription:        "offerDescription",
				OfferTitle:              "offerTitle",
				OfferID:                 offerID,
				EndTime:                 "endTime",
				StartTime:               "startTime",
				Tags:                    gatewayClient.OfferTag{},
			}
			currentPrimeOffers = append(currentPrimeOffers, currentPrimeOffer)
			currentOffersResponse := CurrentOffersResponse{
				CurrentPrimeOffers: currentPrimeOffers,
			}
			buf, err := json.Marshal(currentOffersResponse)
			if err != nil {
				log.Debug(err)
			}
			mockedResp := new(http.Response)
			mockedResp.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
			mockedResp.StatusCode = http.StatusOK
			samusSWSClientMock.On("Do", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(mockedResp, nil)

			resp, s, e := b.GetCurrentOffers(context.TODO(), "", "US", "en", "2017-12-10T12:10:12Z", "TWITCH")

			So(len(resp.CurrentPrimeOffers), ShouldBeGreaterThan, 0)
			So(resp.CurrentPrimeOffers[0].OfferID, ShouldEqual, offerID)
			So(resp.CurrentPrimeOffers[0].OfferTitle, ShouldEqual, "offerTitle")
			So(s, ShouldEqual, 0)
			So(e, ShouldBeNil)
		})

		Convey("When GetCurrentOffersForUser API is successful", func() {

			expectedResponse := clients.GetCurrentOffersForUserInCountryResponse{
				Offers: []gatewayClient.OfferWithClaimData{
					gatewayClient.OfferWithClaimData{
						ContentCategories:       []string{},
						ExternalOfferUri:        "externalOfferUri",
						PublisherName:           "publisherName",
						ApplicableGame:          "applicableGame",
						ContentDeliveryMethod:   "claimCode",
						Assets:                  []gatewayClient.OfferAsset{},
						ContentClaimInstruction: "claimIntructions",
						Priority:                1,
						OfferDescription:        "offerDescription",
						OfferTitle:              "offerTitle",
						OfferID:                 offerID,
						EndTime:                 "endTime",
						StartTime:               "startTime",
						Tags:                    gatewayClient.OfferTag{},
						OfferClaimMetadata:      gatewayClient.OfferClaimMetadata{},
						OfferClaimData:          "claimData",
					},
				},
				ClaimStatuses: map[string]clients.ClaimStatus{
					offerID: {
						ClaimHint: gatewayClient.Available,
					},
				},
			}
			samusTProxClientMock.On("GetCurrentOffersForUserInCountry", mock.Anything, mock.Anything).Return(&expectedResponse, http.StatusOK, nil)

			resp, s, e := b.GetCurrentOffersForUser(context.TODO(), "", "US", "en", "", "TWITCH")

			So(len(resp.PrimeOffers), ShouldEqual, 1)
			So(resp.PrimeOffers[0].OfferID, ShouldEqual, offerID)
			So(resp.PrimeOffers[0].OfferTitle, ShouldEqual, "offerTitle")
			So(resp.PrimeOffers[0].ClaimHint, ShouldEqual, gatewayClient.Available)
			So(s, ShouldEqual, http.StatusOK)
			So(e, ShouldBeNil)
		})

		Convey("When GetCurrentOffersForUser API is not successful", func() {
			samusTProxClientMock.On("GetCurrentOffersForUserInCountry", mock.Anything, mock.Anything).Return(nil, http.StatusInternalServerError, errors.New("Something went wrong!"))

			resp, s, e := b.GetCurrentOffersForUser(context.TODO(), "", "US", "en", "", "TWITCH")

			So(resp, ShouldBeNil)
			So(s, ShouldEqual, http.StatusInternalServerError)
			So(e, ShouldNotBeNil)
		})

		Convey("When ClaimOffer API is successful", func() {
			claimOfferResponse := PrimeEntitlementResponse{
				OfferID:            offerID,
				ClaimInstruction:   "claimInstructions",
				ClaimMethod:        "claimCode",
				OfferClaimData:     "offerClaimData",
				OfferClaimMetadata: gatewayClient.OfferClaimMetadata{},
				HasEntitlement:     true,
				UserID:             userID,
			}
			buf, err := json.Marshal(claimOfferResponse)
			if err != nil {
				log.Debug(err)
			}

			mockedResp := new(http.Response)
			mockedResp.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
			mockedResp.StatusCode = http.StatusOK
			samusSWSClientMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockedResp, nil)

			resp, s, e := b.ClaimOffer(context.TODO(), userID, offerID, "en", []byte("{\"metadata\":\"SomeMetadata\"}"))

			So(resp.OfferID, ShouldEqual, offerID)
			So(resp.ClaimMethod, ShouldEqual, "claimCode")
			So(resp.UserID, ShouldEqual, userID)
			So(resp.HasEntitlement, ShouldEqual, true)
			So(s, ShouldEqual, 0)
			So(e, ShouldBeNil)
		})

		Convey("When DynamicString has active record", func() {
			stringIds := "a,b"
			countryCode := "US"
			locale := "en"

			textA := "stringA"
			textB := "stringB"

			externalUrl := "url"

			promoStringDaoMock.On("GetActiveString", "a", mock.Anything, countryCode, locale).Return(promotion_string.GetPromotionStringResponse{ExternalUrl: externalUrl, String: textA, IsExternalLink: true}, nil)
			promoStringDaoMock.On("GetActiveString", "b", mock.Anything, countryCode, locale).Return(promotion_string.GetPromotionStringResponse{ExternalUrl: "", String: textB, IsExternalLink: false}, nil)

			resp, s, e := b.GetDynamicStrings(context.TODO(), stringIds, countryCode, locale, "123456", "")

			stringA := resp.DynamicStringMap["a"]
			stringB := resp.DynamicStringMap["b"]

			So(stringA.ID, ShouldEqual, "a")
			So(stringA.Text, ShouldEqual, textA)
			So(stringA.ExternalUrl, ShouldEqual, externalUrl)
			So(stringA.IsExternalLink, ShouldEqual, true)

			So(stringB.ID, ShouldEqual, "b")
			So(stringB.Text, ShouldEqual, textB)
			So(stringB.ExternalUrl, ShouldEqual, "")
			So(stringB.IsExternalLink, ShouldEqual, false)

			So(s, ShouldEqual, http.StatusOK)
			So(e, ShouldBeNil)
		})

		Convey("When DynamicString has no active record", func() {
			stringIds := "a,b"
			countryCode := "US"
			locale := "en"

			textA := "stringA"
			textB := "stringB"

			externalUrl := "url"

			promoStringDaoMock.On("GetActiveString", "a", mock.Anything, countryCode, locale).Return(nil, nil)
			promoStringDaoMock.On("GetActiveString", "b", mock.Anything, countryCode, locale).Return(promotion_string.GetPromotionStringResponse{ExternalUrl: "", String: textB, IsExternalLink: false}, nil)
			promoStringDaoMock.On("GetDefaultString", "a", locale).Return(promotion_string.GetPromotionStringResponse{ExternalUrl: externalUrl, String: textA, IsExternalLink: true}, nil)

			resp, s, e := b.GetDynamicStrings(context.TODO(), stringIds, countryCode, locale, "123456", "")

			stringA := resp.DynamicStringMap["a"]
			stringB := resp.DynamicStringMap["b"]

			So(stringA.ID, ShouldEqual, "a")
			So(stringA.Text, ShouldEqual, textA)
			So(stringA.ExternalUrl, ShouldEqual, externalUrl)
			So(stringA.IsExternalLink, ShouldEqual, true)

			So(stringB.ID, ShouldEqual, "b")
			So(stringB.Text, ShouldEqual, textB)
			So(stringB.ExternalUrl, ShouldEqual, "")
			So(stringB.IsExternalLink, ShouldEqual, false)

			So(s, ShouldEqual, http.StatusOK)
			So(e, ShouldBeNil)
		})

		Convey("When nothing is found", func() {
			stringIds := "a,b"
			countryCode := "US"
			locale := "en"

			textB := "stringB"

			promoStringDaoMock.On("GetActiveString", "a", mock.Anything, countryCode, locale).Return(nil, nil)
			promoStringDaoMock.On("GetActiveString", "b", mock.Anything, countryCode, locale).Return(promotion_string.GetPromotionStringResponse{ExternalUrl: "", String: textB, IsExternalLink: false}, nil)
			promoStringDaoMock.On("GetDefaultString", "a", locale).Return(nil, errors.New("error"))

			_, s, e := b.GetDynamicStrings(context.TODO(), stringIds, countryCode, locale, "123456", "")
			So(e, ShouldNotBeNil)
			So(s, ShouldEqual, http.StatusInternalServerError)

		})

		Convey("When whitelisted", func() {
			stringIds := "a"
			countryCode := "US"
			locale := "en"
			userId := "130802556"
			dateOverride := "2017-10-01T00:00:00Z"

			textA := "stringA"

			externalUrl := "url"

			promoStringDaoMock.On("GetActiveString", "a", dateOverride, countryCode, locale).Return(promotion_string.GetPromotionStringResponse{ExternalUrl: externalUrl, String: textA, IsExternalLink: true}, nil)

			resp, s, e := b.GetDynamicStrings(context.TODO(), stringIds, countryCode, locale, userId, dateOverride)

			stringA := resp.DynamicStringMap["a"]

			So(stringA.ID, ShouldEqual, "a")
			So(stringA.Text, ShouldEqual, textA)
			So(stringA.ExternalUrl, ShouldEqual, externalUrl)
			So(stringA.IsExternalLink, ShouldEqual, true)
			So(stringA.IsCached, ShouldEqual, false)

			So(s, ShouldEqual, http.StatusOK)
			So(e, ShouldBeNil)
		})
	})
}

func TestGetCurrentOffersWithEligibility(t *testing.T) {
	Convey("Testing GetCurrentOffersWithEligibility: Setup", t, func() {
		log.SetLevel(log.DebugLevel)

		samusSWSClientMock := new(mocks.HttpMock)
		samusTProxClientMock := new(clients.SamusTProxClientMock)
		dynamoMock := new(dynamo.IUserDaoMock)
		primeEntitlementDynamoMock := new(dynamo.IPrimeEntitlementDaoMock)
		promoStringDaoMock := new(promotion_string.IPromotionStringDaoMock)
		promoStringCacheMock := new(promotion_string.IPromotionStringCacheMock)
		offersCacheMock := new(cache.IOffersCacheMock)
		entitlementsClientMock := new(mocks.HttpMock)
		nitroClientMock := new(mocks.Nitro)

		promoStringCacheMock.On("GetPromotionString", mock.Anything).Return(mock.Anything, false)
		offersCacheMock.On("GetOffers", mock.Anything).Return(mock.Anything, false)
		offersCacheMock.On("SetOffers", mock.Anything, mock.Anything)

		b := Backend{stats: config.Statsd(),
			samusSWSClient:       samusSWSClientMock,
			samusTProxClient:     samusTProxClientMock,
			userDao:              dynamoMock,
			promoStringDao:       promoStringDaoMock,
			entitlementsClient:   entitlementsClientMock,
			primeEntitlementsDao: primeEntitlementDynamoMock,
			promoCache:           promoStringCacheMock,
			offersCache:          offersCacheMock,
			nitroClient:          nitroClientMock,
		}

		currentPrimeOffer := gatewayClient.OfferWithNoClaimData{
			ContentCategories:       []string{},
			ExternalOfferUri:        "externalOfferUri",
			PublisherName:           "publisherName",
			ApplicableGame:          "applicableGame",
			ContentDeliveryMethod:   "claimCode",
			Assets:                  []gatewayClient.OfferAsset{},
			ContentClaimInstruction: "claimIntructions",
			Priority:                1,
			OfferDescription:        "offerDescription",
			OfferTitle:              "offerTitle",
			OfferID:                 offerID,
			CatalogOfferID:          catalogOfferID,
			EndTime:                 "endTime",
			StartTime:               "startTime",
			Tags:                    gatewayClient.OfferTag{},
		}

		currentOffersResponse := CurrentOffersResponse{
			CurrentPrimeOffers: []gatewayClient.OfferWithNoClaimData{
				currentPrimeOffer,
			},
		}

		buf, err := json.Marshal(currentOffersResponse)
		if err != nil {
			log.Debug(err)
		}

		currentOffersResponseJSON := new(http.Response)
		currentOffersResponseJSON.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
		currentOffersResponseJSON.StatusCode = http.StatusOK

		samusSWSClientMock.On("Do", context.TODO(), mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(currentOffersResponseJSON, nil)

		Convey("GetCurrentOffersWithEligibility: Authenticated Twitch user with a linked Amazon account", func() {
			batchGetEligibilityStatusResponse := clients.BatchGetEligibilityStatusResponse{
				Results: []clients.GetEligibilityStatusResponse{
					clients.GetEligibilityStatusResponse{
						OfferID:    currentPrimeOffer.CatalogOfferID,
						OfferState: clients.OfferLive,
						Eligible:   true,
						PriorEntitlements: []clients.EntitlementRecord{
							clients.EntitlementRecord{
								EntitlementStatus: clients.EntitlementCleared,
							},
							clients.EntitlementRecord{
								EntitlementStatus: clients.EntitlementEntitled,
							},
						},
						EligibilityResults: []clients.EligibilityResult{
							clients.EligibilityResult{
								RuleName: clients.PrimeGamingRule,
								Eligible: true,
							},
							clients.EligibilityResult{
								RuleName: clients.LinkedAccountRule,
								Eligible: true,
							},
						},
					},
				},
			}

			expectedResp := gatewayClient.GetCurrentOffersWithEligibilityResponse{
				CurrentOffers: []gatewayClient.OfferWithEligibility{
					gatewayClient.OfferWithEligibility{
						OfferWithNoClaimData: currentPrimeOffer,
						OfferEligibility: gatewayClient.OfferEligibility{
							IsClaimed:         true,
							OfferState:        clients.OfferLive,
							CanClaim:          true,
							PrimeGamingRule:   true,
							LinkedAccountRule: true,
						},
					},
				},
			}

			samusTProxClientMock.On("BatchGetEligibilityStatus", mock.Anything, mock.Anything).Return(&batchGetEligibilityStatusResponse, http.StatusOK, nil)

			resp, s, e := b.GetCurrentOffersWithEligibility(context.TODO(), "101010101", "US", "en", "2017-12-10T12:10:12Z", "TWITCH")

			So(len(resp.CurrentOffers), ShouldEqual, len(expectedResp.CurrentOffers))
			So(resp.CurrentOffers[0].OfferID, ShouldEqual, expectedResp.CurrentOffers[0].OfferID)
			So(resp.CurrentOffers[0].OfferTitle, ShouldEqual, expectedResp.CurrentOffers[0].OfferTitle)
			So(resp.CurrentOffers[0].OfferEligibility, ShouldNotBeNil)
			So(resp.CurrentOffers[0].OfferEligibility.IsClaimed, ShouldBeTrue)
			So(resp.CurrentOffers[0].OfferEligibility.OfferState, ShouldEqual, expectedResp.CurrentOffers[0].OfferEligibility.OfferState)
			So(resp.CurrentOffers[0].OfferEligibility.CanClaim, ShouldEqual, expectedResp.CurrentOffers[0].OfferEligibility.CanClaim)
			So(resp.CurrentOffers[0].OfferEligibility.PrimeGamingRule, ShouldEqual, expectedResp.CurrentOffers[0].OfferEligibility.PrimeGamingRule)
			So(resp.CurrentOffers[0].OfferEligibility.LinkedAccountRule, ShouldEqual, expectedResp.CurrentOffers[0].OfferEligibility.LinkedAccountRule)
			So(s, ShouldEqual, 0)
			So(e, ShouldBeNil)
		})

		Convey("GetCurrentOffersWithEligibility: Unauthenticated Twitch user", func() {
			expectedResp := gatewayClient.GetCurrentOffersWithEligibilityResponse{
				CurrentOffers: []gatewayClient.OfferWithEligibility{
					gatewayClient.OfferWithEligibility{
						OfferWithNoClaimData: currentPrimeOffer,
						OfferEligibility: gatewayClient.OfferEligibility{
							CanClaim: false,
						},
					},
				},
			}

			resp, s, e := b.GetCurrentOffersWithEligibility(context.TODO(), "", "US", "en", "2017-12-10T12:10:12Z", "TWITCH")

			So(len(resp.CurrentOffers), ShouldEqual, len(expectedResp.CurrentOffers))
			So(resp.CurrentOffers[0].OfferID, ShouldEqual, expectedResp.CurrentOffers[0].OfferID)
			So(resp.CurrentOffers[0].OfferTitle, ShouldEqual, expectedResp.CurrentOffers[0].OfferTitle)
			So(resp.CurrentOffers[0].OfferEligibility.CanClaim, ShouldEqual, expectedResp.CurrentOffers[0].OfferEligibility.CanClaim)
			So(s, ShouldEqual, 0)
			So(e, ShouldBeNil)
		})

		Convey("GetCurrentOffersWithEligibility: Authenticated Twitch user without a linked Amazon account", func() {
			expectedResp := gatewayClient.GetCurrentOffersWithEligibilityResponse{
				CurrentOffers: []gatewayClient.OfferWithEligibility{
					gatewayClient.OfferWithEligibility{
						OfferWithNoClaimData: currentPrimeOffer,
						OfferEligibility: gatewayClient.OfferEligibility{
							CanClaim: false,
						},
					},
				},
			}

			samusTProxClientMock.On("BatchGetEligibilityStatus", mock.Anything, mock.Anything).Return(nil, http.StatusOK, nil)

			resp, s, e := b.GetCurrentOffersWithEligibility(context.TODO(), "101010101", "US", "en", "2017-12-10T12:10:12Z", "TWITCH")

			So(len(resp.CurrentOffers), ShouldEqual, len(expectedResp.CurrentOffers))
			So(resp.CurrentOffers[0].OfferID, ShouldEqual, expectedResp.CurrentOffers[0].OfferID)
			So(resp.CurrentOffers[0].OfferTitle, ShouldEqual, expectedResp.CurrentOffers[0].OfferTitle)
			So(resp.CurrentOffers[0].OfferEligibility.CanClaim, ShouldEqual, expectedResp.CurrentOffers[0].OfferEligibility.CanClaim)
			So(s, ShouldEqual, 0)
			So(e, ShouldBeNil)
		})
	})
}
