package backend

import (
	"encoding/json"
	"fmt"
	"time"

	log "github.com/sirupsen/logrus"

	"errors"

	"strings"

	"code.justin.tv/samus/gateway/sns"
	"github.com/aws/aws-sdk-go/service/sqs"
	"golang.org/x/net/context"
)

/*
{
   "Type":"Notification",
   "MessageId":"0004961e-b6f4-5d58-980e-c615eab659e8",
   "TopicArn":"arn:aws:sns:us-west-2:536870266760:TwitchPrime-StatusChange-NA",
   "Message":"{\"twitchUserId\":\"1707235605\",\"callbackIdentifier\":\"4eab07e3-cc22-d693-add0-6227c0db1462\",\"timestamp\":\"2016-07-21T23:25:54.117Z\"}",
   "Timestamp":"2016-07-21T23:25:54.195Z",
   "SignatureVersion":"1",
   "Signature":"XG4v8ZdItg4iOHI7FYOvXTx4Xtkbz4VxIFvDBjKhSO+511mEvq5nKsSvRu+tBaF64fK9Jpnrdqj7bXFZHcOy7DKJBAjBl6MgYmcjEMzBQSkiDUDQb1Dv/IeY+DLCp/NII0NoVMPC7DCMks9mLPbjV9GXuBIN6OIMTLmKm+F98XWZgbcgRQ5vHjRZq3+ncahSozs5qrQb4XVXGRxJv9xXCwguGppNMIbsJF1jImj5QYUtsSay0wuX7ZGO3/73mKB3NFz2iiK0xAslwWTjR55td/N99oPwVqOKWgRfWRdouvWUkTx1hjceu5P37uMGokjD9hqgJP0+rGtWGU7N3fZ7Fw==",
   "SigningCertURL":"https://sns.us-west-2.amazonaws.com/SimpleNotificationService-bb750dd426d95ee9390147a5624348ee.pem",
   "UnsubscribeURL":"https://sns.us-west-2.amazonaws.com/?Action=Unsubscribe&SubscriptionArn=arn:aws:sns:us-west-2:536870266760:TwitchPrime-StatusChange-NA:e982b840-2152-4073-af82-6515928456fa"
}
*/

const LoadTestPrefix = "100000000000"

// SamusSQSMessage holds the details of Samus message
type SamusSQSMessage struct {
	TwitchUserID       string `json:"twitchUserId"`
	CallbackIdentifier string `json:"callbackIdentifier,omitempty"`
	//Timestamp          string `json:"timestamp,omitempty"`
}

// ProcessSamusSQSNotification checks for Samus status and then grants/cancels Samus benefits
func (b *Backend) ProcessSamusSQSNotification(rawMsg *sqs.Message) error {
	// TODO: Graceful shutdown
	log.Debug("Received SQSMessage")

	var msg SQSMessage

	rawBytes := []byte(*rawMsg.Body)
	if err := json.Unmarshal(rawBytes, &msg); err != nil {
		return err
	}

	var msgData SamusSQSMessage

	msgBytes := []byte(msg.Message)
	if err := json.Unmarshal(msgBytes, &msgData); err != nil {
		return err
	}

	log.Info("SamusSQSMessage is:", msgData)

	tuid := msgData.TwitchUserID
	if len(tuid) == 0 {
		errorMsg := "SQS message malformed: missing TwitchUserID"
		log.Error(errorMsg)
		return errors.New(errorMsg)
	}

	hasPrime, err := b.fetchPrimeStatusFromSamus(context.Background(), tuid)

	if err != nil {
		log.WithError(err)
		return err
	}

	// Make a HTTP call to grant or cancel
	err = b.ProcessStatusUpdates(hasPrime, tuid)
	if err != nil {
		log.WithError(err).Error("Error Updating Prime Benefits for User : ", tuid, ". Error:", err, ". This message will be retried.")
		return err
	}

	log.Info("SQSMessage processed successfully for userId : ", tuid, " and callbackId : ", msgData.CallbackIdentifier)
	return nil
}

// ProcessStatusUpdates calls Rails endpoints to grant and cancel samus benefits
func (b *Backend) ProcessStatusUpdates(PrimeStatus bool, TwitchUserID string) error {
	logger := log.WithFields(log.Fields{
		"api":    "ProcessStatusUpdates",
		"prime":  PrimeStatus,
		"userID": TwitchUserID,
	})

	if strings.HasPrefix(TwitchUserID, LoadTestPrefix) {
		logger.Debug("Dropping message - Received Load Test User: ", TwitchUserID)
		return nil
	}

	if PrimeStatus {
		logger.Info("Grant Benefits to: ", TwitchUserID)
		response, err := b.GrantPrime(context.TODO(), TwitchUserID)
		if err != nil {
			logger.Error("Nitro Client GrantPremiumStatus failed: ", err)
			return err
		}
		logger.Info("Nitro Client grant successful: ", response)
	} else {
		logger.Info("Cancel Samus benefits for:", TwitchUserID)

		response, err := b.CancelPrime(context.TODO(), TwitchUserID)
		if err != nil {
			logger.Error("Nitro Client CancelPremiumStatus failed: ", err)
			return err
		}
		logger.Info("Nitro Client cancel successful: ", response)
	}

	timestamp := time.Now().UTC().Format(time.RFC3339)
	statusEvent := &sns.StatusEvent{
		TwitchUserID: TwitchUserID,
		HasPrime:     PrimeStatus,
		Timestamp:    timestamp,
	}
	jsonBytes, err := json.Marshal(statusEvent)
	if err != nil {
		errorMsg := fmt.Sprintf("Failed to json.Marshal event message, %v, %v", statusEvent, err)
		logger.Error(errorMsg)
		return err
	}

	eventMessage := string(jsonBytes)
	// Publish SNS Message consumed externally with Prime Status
	err = b.snsPublisher.Publish(eventMessage)
	if err != nil {
		errorMsg := fmt.Sprintf("Failed to publish SNS event, %v, %v", eventMessage, err)
		logger.Error(errorMsg)
		return err
	}

	return nil
}
