package backend

import (
	"errors"
	"net/http"
	"testing"

	"bytes"
	"encoding/json"
	"io/ioutil"

	"code.justin.tv/common/config"
	"code.justin.tv/samus/gateway/clients/mocks"
	"code.justin.tv/samus/gateway/sns"
	nitro "code.justin.tv/samus/nitro/rpc"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

/*
{
   "Type":"Notification",
   "MessageId":"0004961e-b6f4-5d58-980e-c615eab659e8",
   "TopicArn":"arn:aws:sns:us-west-2:536870266760:TwitchPrime-StatusChange-NA",
   "Message":"{\"twitchUserId\":\"1707235605\",\"callbackIdentifier\":\"4eab07e3-cc22-d693-add0-6227c0db1462\",\"timestamp\":\"2016-07-21T23:25:54.117Z\"}",
   "Timestamp":"2016-07-21T23:25:54.195Z",
   "SignatureVersion":"1",
   "Signature":"XG4v8ZdItg4iOHI7FYOvXTx4Xtkbz4VxIFvDBjKhSO+511mEvq5nKsSvRu+tBaF64fK9Jpnrdqj7bXFZHcOy7DKJBAjBl6MgYmcjEMzBQSkiDUDQb1Dv/IeY+DLCp/NII0NoVMPC7DCMks9mLPbjV9GXuBIN6OIMTLmKm+F98XWZgbcgRQ5vHjRZq3+ncahSozs5qrQb4XVXGRxJv9xXCwguGppNMIbsJF1jImj5QYUtsSay0wuX7ZGO3/73mKB3NFz2iiK0xAslwWTjR55td/N99oPwVqOKWgRfWRdouvWUkTx1hjceu5P37uMGokjD9hqgJP0+rGtWGU7N3fZ7Fw==",
   "SigningCertURL":"https://sns.us-west-2.amazonaws.com/SimpleNotificationService-bb750dd426d95ee9390147a5624348ee.pem",
   "UnsubscribeURL":"https://sns.us-west-2.amazonaws.com/?Action=Unsubscribe&SubscriptionArn=arn:aws:sns:us-west-2:536870266760:TwitchPrime-StatusChange-NA:e982b840-2152-4073-af82-6515928456fa"
}
*/

const ExtSubscriptionID = 987654321
const PurchaseDateTime = "2016-07-21T23:25:54.117Z"
const TicketID = 123456789

func TestSamusSQSUpdatesWithNitro(t *testing.T) {

	Convey("SQS Updates", t, func() {
		log.SetLevel(log.DebugLevel)

		samusSWSClientMock := new(mocks.HttpMock)
		entitlementsClientMock := new(mocks.HttpMock)
		snsPublisherMock := new(sns.SnsClientMock)
		mockNitroClient := new(mocks.Nitro)

		b := Backend{
			stats:              config.Statsd(),
			samusSWSClient:     samusSWSClientMock,
			entitlementsClient: entitlementsClientMock,
			snsPublisher:       snsPublisherMock,
			nitroClient:        mockNitroClient,
		}

		Convey("To Grant Samus Benefits using Nitro", func() {

			grantResponse := &nitro.GrantPremiumStatusResponse{
				HasPrime: true,
			}
			mockNitroClient.On("GrantPremiumStatus", mock.Anything, mock.Anything).Return(grantResponse, nil)
			snsPublisherMock.On("Publish", mock.Anything).Return(nil)

			err := b.ProcessStatusUpdates(true, userID)
			if err != nil {
				log.Debug("Error", err)
			}
			So(err, ShouldBeNil)
		})

		Convey("To Cancel Samus Benefits using Nitro", func() {
			cancelResponse := &nitro.CancelPremiumStatusResponse{
				HasPrime: false,
			}

			mockNitroClient.On("CancelPremiumStatus", mock.Anything, mock.Anything).Return(cancelResponse, nil)
			snsPublisherMock.On("Publish", mock.Anything).Return(nil)

			err := b.ProcessStatusUpdates(false, userID)
			if err != nil {
				log.Debug("Error", err)
			}
			So(err, ShouldBeNil)
		})

		Convey("To Grant Samus Benefits using Nitro fails and rails succeeds", func() {

			// fail nitro for now
			mockNitroClient.On("GrantPremiumStatus", mock.Anything, mock.Anything).Return(nil, errors.New("NitroFailed"))

			mockNitroClient.On("GrantPremiumStatus", mock.Anything, mock.Anything).Return(nil, errors.New("Test Error"))
			snsPublisherMock.On("Publish", mock.Anything).Return(nil)

			err := b.ProcessStatusUpdates(true, userID)
			if err != nil {
				log.Debug("Error", err)
			}
			So(err, ShouldNotBeNil)
		})

		Convey("To Cancel Samus Benefits using Nitro fails ", func() {

			// fail nitro for now
			mockNitroClient.On("CancelPremiumStatus", mock.Anything, mock.Anything).Return(nil, errors.New("NitroFailed"))

			mockNitroClient.On("CancelPremiumStatus", mock.Anything, mock.Anything).Return(nil, errors.New("Test Error"))
			snsPublisherMock.On("Publish", mock.Anything).Return(nil)

			err := b.ProcessStatusUpdates(false, userID)
			if err != nil {
				log.Debug("Error", err)
			}
			So(err, ShouldNotBeNil)
		})

		Convey("To Grant Samus Benefits With SNS Publish Fail", func() {

			mockNitroClient.On("GrantPremiumStatus", mock.Anything, mock.Anything).Return(nil, errors.New("NitroFailed"))
			grantSamusBenefitsResponse := GrantSamusBenefitsResponse{ExtSubscriptionID: ExtSubscriptionID, PurchaseDateTime: PurchaseDateTime}
			buf, err := json.Marshal(grantSamusBenefitsResponse)
			if err != nil {
				log.Debug(err)
			}
			mockedResp := new(http.Response)
			mockedResp.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
			mockedResp.StatusCode = http.StatusOK
			entitlementsClientMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockedResp, nil)
			mockNitroClient.On("GrantPremiumStatus", mock.Anything, mock.Anything).Return(nil, errors.New("Test Error"))
			snsPublisherMock.On("Publish", mock.Anything).Return(errors.New("TestError Failed to Publish"))

			err = b.ProcessStatusUpdates(true, userID)
			if err != nil {
				log.Debug("Error", err)
			}
			So(err, ShouldNotBeNil)
		})
	})
}
