package backend

import (
	"crypto/tls"
	"crypto/x509"
	"io/ioutil"
	"strconv"

	"golang.org/x/net/context"

	"os"

	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/samus/gateway/dynamo"
	"code.justin.tv/samus/gateway/promotion_string"
	"code.justin.tv/samus/gateway/sns"
	nitro "code.justin.tv/samus/nitro/rpc"
	log "github.com/sirupsen/logrus"

	"time"

	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/samus/gateway/cache"
	"code.justin.tv/samus/gateway/clients"
	"github.com/cactus/go-statsd-client/statsd"

	client "code.justin.tv/samus/gateway/client"
)

// Caching is enabled by default unless overridden by ENV var CACHEENABLED
var cachingEnabled = true

// Backender provides interface for supported methods
type Backender interface {
	Status(ctx context.Context, userID string) (*StatusResponse, int, error)
	Balance(ctx context.Context, userID string) (*BalanceResponse, int, error)
	CanSpendPrimeCredit(ctx context.Context, userID string, productID string) (*CanSpendPrimeCreditResponse, int, error)
	SubCreditSpend(ctx context.Context, userID string, orderID string, jsonStr []byte) (*SubCreditSpendResponse, int, error)
	GrantSamusBenefits(ctx context.Context, jsonStr []byte) (*GrantSamusBenefitsResponse, int, error)
	CancelSamusBenefits(ctx context.Context, jsonStr []byte) (*CancelSamusBenefitsResponse, int, error)
	GetSettings(ctx context.Context, userID string) (*SettingsResponse, int, error)
	UpdateSettings(ctx context.Context, userID string, isSubscriptionMessage bool) (*SettingsResponse, int, error)
	GetOfferBlacklist(ctx context.Context) (*OfferBlacklistResponse, int, error)
	GetCurrentOffers(ctx context.Context, userID string, countryCode string, locale string, dateOverride string, clientID string) (*CurrentOffersResponse, int, error)
	GetCurrentOffersForUser(ctx context.Context, userID string, countryCode string, locale string, dateOverride string, clientID string) (*client.CurrentPrimeOffersResponse, int, error)
	GetCurrentOffersWithEligibility(ctx context.Context, userID string, countryCode string, locale string, dateOverride string, clientID string) (*client.GetCurrentOffersWithEligibilityResponse, int, error)
	ClaimOffer(ctx context.Context, userID string, offerID string, locale string, jsonStr []byte) (*PrimeEntitlementResponse, int, error)
	PlaceOrder(ctx context.Context, userID string, offerID string, idempotenceKey string, attributionChannel string, dateOverride string, clientID string) (*clients.PlaceOrderResponse, int, error)
	GetOrdersByCustomer(ctx context.Context, userID string, nextToken string, pageSize int, offerID string, orderID string, clientID string) (*clients.GetOrdersByCustomerResponse, int, error)
	ListInventory(ctx context.Context, userID string, amazonCustomerId string, entitlementAccountType string, itemIds []string, nextToken string, maxResults int, entitlementStatusFilters []string, clientID string) (*clients.ListInventoryResponse, int, error)
	GetPrimeEntitlement(ctx context.Context, userID, offerID, locale string) (*PrimeEntitlementResponse, int, error)
	SetPrimeEntitlement(ctx context.Context, userID, offerID, hasEntitlement string) (*PrimeEntitlementResponse, int, error)
	UpdatePrimeEntitlement(ctx context.Context, userID, offerID, locale string) error
	ClearOfferClaimCodeForUser(ctx context.Context, userID string, marketplaceID, offerID string, csAgent string, csContactID string) (*clients.ClearOfferClaimCodeForUserResponse, int, error)
	GetDynamicStrings(ctx context.Context, stringIDs string, countryCode string, locale string, userID string, dateOverride string) (*GetDynamicStringResponse, int, error)
	CreateAccountLink(ctx context.Context, accountLink GameAccountLink) (int, error)
	GetAccountLink(ctx context.Context, userID, gameID string) (*GameAccountLink, int, error)
	DeleteAccountLink(ctx context.Context, userID, gameID string) (*GameAccountLink, int, error)
	SpendSubscriptionCredit(ctx context.Context, userID string, broadcasterID string) (*SpendSubscriptionCreditResponse, error)
	UnspendSubscriptionCredit(ctx context.Context, userID string, broadcasterID string) (*UnspendSubscriptionCreditResponse, error)
	GrantPrime(ctx context.Context, userID string) (*PremiumStatusResponse, error)
	CancelPrime(ctx context.Context, userID string) (*PremiumStatusResponse, error)
}

// Backend provides samus clients
type Backend struct {
	stats                 statsd.Statter
	samusSWSClient        twitchhttp.Client
	entitlementsClient    twitchhttp.Client
	userDao               dynamo.IUserDao
	primeEntitlementsDao  dynamo.IPrimeEntitlementDao
	promoStringDao        promotion_string.IPromotionStringDao
	accountLinkDao        dynamo.IAccountLinkDao
	snsPublisher          sns.SnsPublisher
	nitroClient           nitro.Nitro
	promoCache            promotion_string.IPromotionStringCache
	offersCache           cache.IOffersCache
	samusSWSClientWrapper clients.SamusSWSClienter
	samusTProxClient      clients.SamusTProxClienter
	subscriptionsClient   clients.SubscriptionsClienter
	voyagerClient         clients.VoyagerClienter
	paymentsClient        clients.PaymentsClienter
	pubsubClient          clients.PubSub
}

// BackendConf provides configuration required for Backend
type BackendConf struct {
	Stats                      statsd.Statter
	SamusSWSHost               string
	SamusSWSCertPath           string
	RootCertPath               string
	RedisHostPort              string
	RailsHost                  string
	DynamoRegion               string
	UserDynamoTableName        string
	PromoStringDynamoTableName string
	EntitlementDynamoTableName string
	AccountLinkDynamoTableName string
	SnsRegion                  string
	PrimeStatusSnsTopicArn     string
	NitroEndpoint              string
	OfferCacheExpiration       string
	PubSubHost                 string
}

// NewBackend initializes Backend
func NewBackend(conf *BackendConf) (*Backend, error) {
	c := os.Getenv("CACHEENABLED")
	if c != "" {
		var err error
		cachingEnabled, err = strconv.ParseBool(c)
		if err != nil {
			log.WithError(err).Error(err)
		}
		log.Debug("Caching Enabled: ", cachingEnabled)
	}

	log.Info("Samus Cert Path:", conf.SamusSWSCertPath)
	clientCert, err := tls.LoadX509KeyPair(conf.SamusSWSCertPath, conf.SamusSWSCertPath)
	if err != nil {
		return nil, err
	}

	log.Info("CA Cert Path:", conf.RootCertPath)
	rootCert, err := ioutil.ReadFile(conf.RootCertPath)
	if err != nil {
		return nil, err
	}

	rootCertPool := x509.NewCertPool()
	rootCertPool.AppendCertsFromPEM(rootCert)
	tlsConfig := &tls.Config{
		RootCAs:            rootCertPool,
		Certificates:       []tls.Certificate{clientCert},
		InsecureSkipVerify: false,
	}
	tlsConfig.BuildNameToCertificate()

	samusSWSClient, err := twitchhttp.NewClient(twitchhttp.ClientConf{
		Host:            conf.SamusSWSHost,
		Stats:           conf.Stats,
		TimingXactName:  "Samus-SWS-Server-Client",
		TLSClientConfig: tlsConfig,
	})
	if err != nil {
		return nil, err
	}

	samusTProxClient, err := clients.NewSamusTProxClient()
	if err != nil {
		log.WithError(err).Error("[NewBackend] ", "NewSamusTProxClient failed: ", err)
		return nil, err
	}

	entitlementsClient, err := twitchhttp.NewClient(twitchhttp.ClientConf{
		Host:           conf.RailsHost,
		Stats:          conf.Stats,
		TimingXactName: "Rails-Client",
	})

	if err != nil {
		return nil, err
	}

	nitroConf := twitchclient.ClientConf{
		Host:           conf.NitroEndpoint,
		TimingXactName: "Nitro-Client",
		Stats:          conf.Stats,
	}

	nitroClient := nitro.NewNitroProtobufClient(conf.NitroEndpoint, twitchclient.NewHTTPClient(nitroConf))

	if err != nil {
		return nil, err
	}

	userDynamoClient := dynamo.NewClient(&dynamo.DynamoClientConfig{
		AwsRegion: conf.DynamoRegion,
		TableName: conf.UserDynamoTableName,
	})

	userDao := dynamo.NewUserDao(userDynamoClient)

	promoStringDynamoClient := promotion_string.NewClient(&promotion_string.DynamoClientConfig{
		AwsRegion: conf.DynamoRegion,
		TableName: conf.PromoStringDynamoTableName,
	})
	promoStringDao := promotion_string.NewPromotionStringDao(promoStringDynamoClient)

	entitlementDynamoClient := dynamo.NewClient(&dynamo.DynamoClientConfig{
		AwsRegion: conf.DynamoRegion,
		TableName: conf.EntitlementDynamoTableName,
	})
	primeEntitlementsDao := dynamo.NewPrimeEntitlementDao(entitlementDynamoClient)

	accountLinkDynamoClient := dynamo.NewClient(&dynamo.DynamoClientConfig{
		AwsRegion: conf.DynamoRegion,
		TableName: conf.AccountLinkDynamoTableName,
	})
	accountLinkDao := dynamo.NewAccountLinkDao(accountLinkDynamoClient)

	snsPublisher := sns.NewClient(&sns.SnsClientConfig{
		AwsRegion: conf.SnsRegion,
		TopicArn:  conf.PrimeStatusSnsTopicArn,
	})

	promoCache := promotion_string.NewCache()

	var offersCache cache.IOffersCache
	offersCacheTTL, err := time.ParseDuration(conf.OfferCacheExpiration)
	if err != nil {
		log.Warn("Could not parse duration, using default cache TTL... ", err)
		offersCache = cache.NewOffersCache()
	} else {
		offersCache = cache.NewOffersCacheWithExpiration(offersCacheTTL)
	}

	samusSWSClientTwo, err := clients.NewSamusSWSClient()
	if err != nil {
		log.WithError(err).Error("[NewBackend] ", "NewSamusSWSClient failed: ", err)
		return nil, err
	}

	subscriptionsClient, err := clients.NewSubscriptionsClient()
	if err != nil {
		log.WithError(err).Error("[NewBackend] ", "NewSubscriptionsClient failed: ", err)
		return nil, err
	}

	voyagerClient, err := clients.NewVoyagerClient()
	if err != nil {
		log.WithError(err).Error("[NewBackend] ", "NewVoyagerClient failed: ", err)
		return nil, err
	}

	paymentsClient, err := clients.NewPaymentsClient()
	if err != nil {
		log.WithError(err).Error("[NewBackend] ", "NewPaymentsClient failed", err)
		return nil, err
	}

	pubsubClient, err := clients.NewPubSub(conf.PubSubHost)
	if err != nil {
		log.WithError(err).Error("Failed to create pubsub client", err)
	}

	//TODO: Set http client timeouts
	return &Backend{
		stats:                 conf.Stats,
		samusSWSClient:        samusSWSClient,
		entitlementsClient:    entitlementsClient,
		userDao:               userDao,
		primeEntitlementsDao:  primeEntitlementsDao,
		promoStringDao:        promoStringDao,
		accountLinkDao:        accountLinkDao,
		snsPublisher:          snsPublisher,
		nitroClient:           nitroClient,
		promoCache:            promoCache,
		offersCache:           offersCache,
		samusSWSClientWrapper: samusSWSClientTwo,
		samusTProxClient:      samusTProxClient,
		subscriptionsClient:   subscriptionsClient,
		voyagerClient:         voyagerClient,
		paymentsClient:        paymentsClient,
		pubsubClient:          pubsubClient,
	}, nil
}
