package backend

import (
	"time"

	voyagertwirp "code.justin.tv/amzn/TwitchVoyagerTwirp"
	substwirp "code.justin.tv/revenue/subscriptions/twirp"

	"errors"

	"code.justin.tv/samus/gateway/clients"
	log "github.com/sirupsen/logrus"
	"golang.org/x/net/context"
)

const (
	samusPaymentProvider = "samus"
	tier1000             = "1000"
	tierCustom           = "Custom"
)

// SpendSubscriptionCredit API checks the users balance, gets the product name, creates the Payments entitlement, spends the token on Amazon SSCS
func (b *Backend) SpendSubscriptionCredit(ctx context.Context, userID string, broadcasterID string) (*SpendSubscriptionCreditResponse, error) {

	logger := log.WithFields(log.Fields{
		"userID":        userID,
		"broadcasterID": broadcasterID,
	})

	// step 1: check if has balance
	balanceResponse, _, err := b.Balance(ctx, userID)

	if err != nil {
		logger.Error("[SpendSubscriptionCredit] ", "GetBalance failed: ", err)
		return nil, err
	}

	logger.Info("Get Balance response: ", balanceResponse.CreditBalance)

	if balanceResponse.CreditBalance < 1 {
		logger.Info("[SpendSubscriptionCredit] ", "User can't spend credit.")

		return nil, errors.New("NO_BALANCE")
	}

	getProductsRequest := &substwirp.GetChannelProductsRequest{
		ChannelId: broadcasterID,
	}
	// step 2: check if channel can receive subscriptions (partnered) and get short name.
	getChannelProductsResponse, err := b.subscriptionsClient.GetChannelProducts(ctx, getProductsRequest)

	if err != nil {
		logger.Error("[SpendSubscriptionCredit] ", "subscriptionsClient.GetChannelProducts failed: ", err)
		return nil, err
	}

	logger.Info("[SpendSubscriptionCredit] Products: ", getChannelProductsResponse)

	// Check if products exist for this Channel
	if len(getChannelProductsResponse.Products) == 0 {
		logger.Error("[SpendSubscriptionCredit] ", "Broadcaster/Channel can't receive subscriptions.")
		return nil, errors.New("NO_PRODUCTS_FOR_CHANNEL")
	}

	products := getChannelProductsResponse.Products

	eligibleProducts := FilterBy(products, func(p *substwirp.Product) bool {
		// Filter down products based on no tier or tier 1000
		return p.GetTier() == tierCustom || p.GetTier() <= tier1000
	})

	// Check if products that are tier 1000 or less
	if len(eligibleProducts) == 0 {
		logger.Error("[SpendSubscriptionCredit] ", "Broadcaster/Channel has no eligible products based on tier.")
		return nil, errors.New("NO_PRODUCTS_FOR_CHANNEL")
	}

	var productID = eligibleProducts[0].GetId()

	log.Info("Checking whether channel is valid for prime spend", productID)

	if !validProductForPrimeSpend(productID) {
		logger.Error("[SpendSubscriptionCredit] ", "samusSWSClient.SpendCredit failed, called on invalid product ", productID, " by user ", userID)
		return nil, errors.New("INVALID_PRODUCT_FOR_SPEND")
	}

	// Step 2.5 Check if they have a paid subscription
	getUserChannelSubsRequest := &voyagertwirp.GetUserChannelSubscriptionWithPaymentsDataRequest{
		ChannelId: broadcasterID,
		UserId:    userID,
	}
	voyagerResponse, err := b.voyagerClient.GetUserChannelSubscriptionWithPaymentsData(ctx, getUserChannelSubsRequest)
	if err != nil {
		logger.Error("[SpendSubscriptionCredit] ", "voyagerClient.GetUserChannelSubscriptionWithPaymentsData failed: ", err)
		return nil, err
	}

	if voyagerResponse != nil && voyagerResponse.GetSubscription() != nil {
		// Fail the attempt if the user has an active paid subscription
		if voyagerResponse.GetSubscription().Subscription.Type != voyagertwirp.SubscriptionType_SUBSCRIPTION_TYPE_PRIME {
			logger.Error("[SpendSubscriptionCredit] ", "User already has a paid subscription for this channel.")
			return nil, errors.New("HAS_PAID_SUB")
		}

		getPaidUpgradesRequest := &substwirp.GetPaidUpgradesRequest{
			Keys: []*substwirp.SubKey{{
				Product: voyagerResponse.GetSubscription().Subscription.ProductId,
				Owner:   voyagerResponse.GetSubscription().Subscription.UserId,
				Origin:  voyagerResponse.GetSubscription().Subscription.OriginId,
			}},
		}
		paidUpgradesResponse, err := b.subscriptionsClient.GetPaidUpgrades(ctx, getPaidUpgradesRequest)

		if err != nil {
			logger.Error("[SpendSubscriptionCredit] ", "subscriptionsClient.GetPaidUpgrades failed: ", err)
			return nil, err
		}

		paidUpgrades := paidUpgradesResponse.GetPaidUpgrades()

		// Fail the attempt if the user has a paid upgrade (which is an inactive Twitch paid sub with a future start date).
		// Paid upgrades include Prime -> Paid, Gift -> Paid, Did Not Renew (DNR) -> Resub, Code Redemption -> Paid
		if len(paidUpgrades) != 0 {
			logger.Error("[SpendSubscriptionCredit] ", "User already has a paid subscription upgrade for this channel: ", paidUpgrades)
			return nil, errors.New("HAS_PAID_SUB")
		}
	}

	logger.Info("[SpendSubscriptionCredit] Product Found, stupidly taking 1st: ", eligibleProducts[0])
	shortName := eligibleProducts[0].ShortName
	orderID := userID + shortName + time.Now().UTC().Format(time.RFC3339)

	logger = log.WithFields(log.Fields{
		"userID":        userID,
		"broadcasterID": broadcasterID,
		"shortName":     shortName,
		"orderID":       orderID,
	})

	spendCreditRequest := &clients.SpendCreditRequest{
		UserID:    userID,
		ProductID: shortName,
		OrderID:   orderID,
		Channels:  []string{broadcasterID},
	}
	// step 3 call SamusSubscriptionCreditService
	samusSpendResponse, err := b.samusSWSClientWrapper.SpendCredit(ctx, spendCreditRequest)
	if err != nil {
		// TOO_MANY_RECENT_SPENDS needs to come from Client Downstream or logic added here
		logger.Error("[SpendSubscriptionCredit] ", "samusSWSClient.SpendCredit failed: ", err.Error())
		return nil, errors.New("TOO_MANY_RECENT_SPENDS")
	}

	paymentOptions := clients.PaymentOptions{
		OrderID: orderID,
	}
	request := clients.InternalCompletePurchaseRequest{
		PaymentProvider: samusPaymentProvider,
		UserID:          userID,
		ProductID:       shortName,
		Recurring:       false,
		Options:         paymentOptions,
		PurchaseRegion:  samusSpendResponse.PurchaseRegion,
	}
	logger.Info("[SpendSubscriptionCredit] ", "Calling paymentsClient.InternalCompletePurchase: ", request)

	// step 4: call payments service
	purchaseProfile, statusCode, err := b.paymentsClient.InternalCompletePurchase(ctx, userID, shortName, request, nil)
	if err != nil {
		logger.Error("[SpendSubscriptionCredit] ", "paymentsClient.CompletePurchase failed: ", err)
		logger.Info("[UnpendSubscriptionCredit] ", "Attempting samusSWSClient.UnspendCredit: ", request)
		unspendCreditRequest := &clients.UnspendCreditRequest{
			UserID:  userID,
			OrderID: orderID,
		}
		_, unspendErr := b.samusSWSClientWrapper.UnspendCredit(ctx, unspendCreditRequest)
		if unspendErr != nil {
			logger.Error("[UnpendSubscriptionCredit] ", "samusSWSClient.UnspendCredit failed: ", request)
		}

		// a channel blocked or banned user will receive a 409 "User already subscribed" from payments
		if statusCode == 409 {
			logger.Error("[SpendSubscriptionCredit] ", "User is blocked/banned from the channel: ", err)
			return nil, errors.New("PAYMENT_CONFLICT")
		}
		return nil, err
	}

	// step 5: If setting is enabled, create chat notification token
	settings, _, err := b.GetSettings(ctx, userID)
	if err != nil {
		logger.Error("[SpendSubscriptionCredit] ", "GetSettings failed: ", err)
		return nil, err
	}
	if settings.IsSubscriptionMessage {
		chatResp, err := b.subscriptionsClient.CreateChatNotificationToken(ctx, userID, productID)
		if err != nil {
			logger.Error("[SpendSubscriptionCredit] ", "subscriptionsClient.CreateChatNotificationToken failed: ", err)
		}
		logger.Info("[SpendSubscriptionCredit] ", "Chat notification Token Response: ", chatResp)
	}

	logger = log.WithFields(log.Fields{
		"userID":            userID,
		"broadcasterID":     broadcasterID,
		"shortName":         shortName,
		"orderID":           orderID,
		"purchaseProfileID": purchaseProfile.ID,
	})

	logger.Info("[SpendSubscriptionCredit] SUCCESS : ", samusSpendResponse)

	return &SpendSubscriptionCreditResponse{
		UserID:                    userID,
		BroadcasterID:             broadcasterID,
		SubscriptionCreditBalance: 0,
	}, nil
}

// FilterBy filters the Subs Products based on an input Predicte function
func FilterBy(products []*substwirp.Product, f func(*substwirp.Product) bool) []*substwirp.Product {
	eligibleProducts := make([]*substwirp.Product, 0)
	for _, product := range products {
		if f(product) {
			eligibleProducts = append(eligibleProducts, product)
		}
	}
	return eligibleProducts
}
