package backend

import (
	"testing"
	"time"

	"github.com/golang/protobuf/ptypes"
	"github.com/pkg/errors"

	"bytes"
	"io/ioutil"
	"net/http"

	"encoding/json"

	voyagertwirp "code.justin.tv/amzn/TwitchVoyagerTwirp"
	"code.justin.tv/common/config"
	payments "code.justin.tv/revenue/payments-service-go-client/client"
	substwirp "code.justin.tv/revenue/subscriptions/twirp"
	"code.justin.tv/samus/gateway/clients"
	"code.justin.tv/samus/gateway/clients/mocks"
	"code.justin.tv/samus/gateway/dynamo"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
	"golang.org/x/net/context"
)

func TestSpendSubscriptionCreditBackendSuccess(t *testing.T) {

	log.SetLevel(log.DebugLevel)
	samusSWSClientMock := new(mocks.HttpMock)
	samusSWSClient2Mock := new(clients.SamusSWSClientMock)
	paymentsMock := new(clients.PaymentsClientMock)
	subsMock := new(clients.SubscriptionsClientMock)
	dynamoMock := new(dynamo.IUserDaoMock)

	b := Backend{stats: config.Statsd(),
		samusSWSClient:        samusSWSClientMock,
		samusSWSClientWrapper: samusSWSClient2Mock,
		userDao:               dynamoMock,
		paymentsClient:        paymentsMock,
		subscriptionsClient:   subsMock,
	}

	Convey("When New Spend API is used on a blocked product", t, func() {

		balanceResponse := BalanceResponse{CreditBalance: 1}
		buf, err := json.Marshal(balanceResponse)
		if err != nil {
			log.Debug(err)
		}
		mockedResp := new(http.Response)
		mockedResp.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
		mockedResp.StatusCode = http.StatusOK
		samusSWSClientMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockedResp, nil)

		product1 := substwirp.Product{
			ShortName: "TEST_SHORTNAME_BLOCKED",
			Price:     499,
			Id:        "689376",
			Tier:      "1000",
		}
		subsResp := substwirp.GetChannelProductsResponse{
			Products: []*substwirp.Product{&product1},
		}
		subsMock.On("GetChannelProducts", mock.Anything, mock.Anything).Return(&subsResp, nil)

		profile := payments.PurchaseProfile{
			ID: 123,
		}
		paymentsMock.On("InternalCompletePurchase", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(&profile, 200, nil)

		spendCreditResponse := clients.SpendCreditResponse{CreditBalance: 0, UserID: userID}
		buf, err = json.Marshal(spendCreditResponse)
		if err != nil {
			log.Debug(err)
		}
		mockedResp = new(http.Response)
		mockedResp.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
		mockedResp.StatusCode = http.StatusOK
		samusSWSClient2Mock.On("SpendCredit", mock.Anything, mock.Anything).Return(&spendCreditResponse, nil)

		resp, e := b.SpendSubscriptionCredit(context.TODO(), userID, broadcasterID)

		So(e, ShouldNotBeNil)
		So(resp, ShouldBeNil)
	})

}

func TestSpendSubscriptionCreditBackendTierTooHigh(t *testing.T) {

	log.SetLevel(log.DebugLevel)

	samusSWSClientMock := new(mocks.HttpMock)
	samusSWSClient2Mock := new(clients.SamusSWSClientMock)
	paymentsMock := new(clients.PaymentsClientMock)
	subsMock := new(clients.SubscriptionsClientMock)

	b := Backend{stats: config.Statsd(),
		samusSWSClient:        samusSWSClientMock,
		samusSWSClientWrapper: samusSWSClient2Mock,
		paymentsClient:        paymentsMock,
		subscriptionsClient:   subsMock,
	}

	Convey("When product has invalid tier", t, func() {

		balanceResponse := BalanceResponse{CreditBalance: 1}
		buf, err := json.Marshal(balanceResponse)
		if err != nil {
			log.Debug(err)
		}
		mockedResp := new(http.Response)
		mockedResp.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
		mockedResp.StatusCode = http.StatusOK
		samusSWSClientMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockedResp, nil)

		product1 := substwirp.Product{
			ShortName: "TEST_SHORTNAME_TIER_WRONG",
			Tier:      "2000",
		}
		product2 := substwirp.Product{
			ShortName: "TEST_SHORTNAME_TIER_WRONG_2",
			Tier:      "4000",
		}
		subsResp := substwirp.GetChannelProductsResponse{
			Products: []*substwirp.Product{&product1, &product2},
		}
		subsMock.On("GetChannelProducts", mock.Anything, mock.Anything).Return(&subsResp, nil)

		resp, e := b.SpendSubscriptionCredit(context.TODO(), userID, broadcasterID)

		So(resp, ShouldBeNil)
		So(e, ShouldNotBeNil)
		So(e.Error(), ShouldEqual, "NO_PRODUCTS_FOR_CHANNEL")
	})

}

func TestSpendSubscriptionCreditBackendChatTokenIsTrue(t *testing.T) {

	log.SetLevel(log.DebugLevel)

	samusSWSClientMock := new(mocks.HttpMock)
	samusSWSClient2Mock := new(clients.SamusSWSClientMock)
	paymentsMock := new(clients.PaymentsClientMock)
	subsMock := new(clients.SubscriptionsClientMock)
	voyagerMock := new(clients.VoyagerClientMock)
	dynamoMock := new(dynamo.IUserDaoMock)

	b := Backend{stats: config.Statsd(),
		samusSWSClient:        samusSWSClientMock,
		samusSWSClientWrapper: samusSWSClient2Mock,
		userDao:               dynamoMock,
		paymentsClient:        paymentsMock,
		subscriptionsClient:   subsMock,
		voyagerClient:         voyagerMock,
	}

	Convey("When IsSubscriptionMessage: true", t, func() {
		profile := payments.PurchaseProfile{
			ID: 123,
		}
		balanceResponse := BalanceResponse{CreditBalance: 1}

		buf, err := json.Marshal(balanceResponse)
		if err != nil {
			log.Debug(err)
		}
		mockedResp := new(http.Response)
		mockedResp.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
		mockedResp.StatusCode = http.StatusOK
		samusSWSClientMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockedResp, nil)

		dynamoMock.On("GetOrCreate", mock.Anything, mock.Anything).Return(dynamo.User{IsSubscriptionMessage: true}, nil)
		paymentsMock.On("InternalCompletePurchase", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(&profile, 200, nil)

		subsMock.On("CreateChatNotificationToken", mock.Anything, mock.Anything, mock.Anything).Return(&substwirp.CreateChatNotificationTokenResponse{Message: "message"}, nil)
		product1 := substwirp.Product{
			ShortName: "TEST_SHORTNAME_TIER_WRONG",
			Price:     999,
			Tier:      "2000",
		}
		product2 := substwirp.Product{
			ShortName: "TEST_SHORTNAME_GOOD",
			Price:     499,
			Tier:      "1000",
		}
		subsResp := substwirp.GetChannelProductsResponse{
			Products: []*substwirp.Product{&product1, &product2},
		}
		subsMock.On("GetChannelProducts", mock.Anything, mock.Anything).Return(&subsResp, nil)
		voyagerResponse := voyagertwirp.GetUserChannelSubscriptionWithPaymentsDataResponse{
			Subscription: nil,
		}
		voyagerMock.On("GetUserChannelSubscriptionWithPaymentsData", mock.Anything, mock.Anything).Return(&voyagerResponse, nil)

		paidUpgradesResponse := &substwirp.GetPaidUpgradesResponse{
			PaidUpgrades: make([]*substwirp.PaidUpgrade, 0),
		}

		subsMock.On("GetPaidUpgrades", mock.Anything, mock.Anything).Return(paidUpgradesResponse, nil)

		spendCreditResponse := clients.SpendCreditResponse{CreditBalance: 0, UserID: userID}
		buf, err = json.Marshal(spendCreditResponse)
		if err != nil {
			log.Debug(err)
		}
		mockedResp = new(http.Response)
		mockedResp.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
		mockedResp.StatusCode = http.StatusOK
		samusSWSClient2Mock.On("SpendCredit", mock.Anything, mock.Anything).Return(&spendCreditResponse, nil)

		resp, e := b.SpendSubscriptionCredit(context.TODO(), userID, broadcasterID)

		So(e, ShouldBeNil)
		So(resp.UserID, ShouldEqual, userID)
		So(resp.BroadcasterID, ShouldEqual, broadcasterID)
		So(resp.SubscriptionCreditBalance, ShouldEqual, 0)
	})
}

func TestSpendSubscriptionCreditCustomProduct(t *testing.T) {

	log.SetLevel(log.DebugLevel)

	samusSWSClientMock := new(mocks.HttpMock)
	samusSWSClient2Mock := new(clients.SamusSWSClientMock)
	paymentsMock := new(clients.PaymentsClientMock)
	subsMock := new(clients.SubscriptionsClientMock)
	voyagerMock := new(clients.VoyagerClientMock)
	dynamoMock := new(dynamo.IUserDaoMock)

	b := Backend{stats: config.Statsd(),
		samusSWSClient:        samusSWSClientMock,
		samusSWSClientWrapper: samusSWSClient2Mock,
		userDao:               dynamoMock,
		paymentsClient:        paymentsMock,
		subscriptionsClient:   subsMock,
		voyagerClient:         voyagerMock,
	}

	Convey("When used on custom tier product", t, func() {
		profile := payments.PurchaseProfile{
			ID: 123,
		}
		balanceResponse := BalanceResponse{CreditBalance: 1}

		buf, err := json.Marshal(balanceResponse)
		if err != nil {
			log.Debug(err)
		}
		mockedResp := new(http.Response)
		mockedResp.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
		mockedResp.StatusCode = http.StatusOK
		samusSWSClientMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockedResp, nil)

		dynamoMock.On("GetOrCreate", mock.Anything, mock.Anything).Return(dynamo.User{IsSubscriptionMessage: true}, nil)
		paymentsMock.On("InternalCompletePurchase", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(&profile, 200, nil)

		subsMock.On("CreateChatNotificationToken", mock.Anything, mock.Anything, mock.Anything).Return(&substwirp.CreateChatNotificationTokenResponse{Message: "message"}, nil)
		product1 := substwirp.Product{
			ShortName: "TEST_SHORTNAME_TIER_WRONG",
			Price:     999,
			Tier:      "2000",
		}
		product2 := substwirp.Product{
			ShortName: "TEST_SHORTNAME_GOOD",
			Price:     499,
			Tier:      "Custom",
		}
		subsResp := substwirp.GetChannelProductsResponse{
			Products: []*substwirp.Product{&product1, &product2},
		}
		subsMock.On("GetChannelProducts", mock.Anything, mock.Anything).Return(&subsResp, nil)
		voyagerResponse := voyagertwirp.GetUserChannelSubscriptionWithPaymentsDataResponse{
			Subscription: nil,
		}
		voyagerMock.On("GetUserChannelSubscriptionWithPaymentsData", mock.Anything, mock.Anything).Return(&voyagerResponse, nil)

		paidUpgradesResponse := &substwirp.GetPaidUpgradesResponse{
			PaidUpgrades: make([]*substwirp.PaidUpgrade, 0),
		}

		subsMock.On("GetPaidUpgrades", mock.Anything, mock.Anything).Return(paidUpgradesResponse, nil)

		spendCreditResponse := clients.SpendCreditResponse{CreditBalance: 0, UserID: userID}
		buf, err = json.Marshal(spendCreditResponse)
		if err != nil {
			log.Debug(err)
		}
		mockedResp = new(http.Response)
		mockedResp.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
		mockedResp.StatusCode = http.StatusOK
		samusSWSClient2Mock.On("SpendCredit", mock.Anything, mock.Anything).Return(&spendCreditResponse, nil)

		resp, e := b.SpendSubscriptionCredit(context.TODO(), userID, broadcasterID)

		So(e, ShouldBeNil)
		So(resp.UserID, ShouldEqual, userID)
		So(resp.BroadcasterID, ShouldEqual, broadcasterID)
		So(resp.SubscriptionCreditBalance, ShouldEqual, 0)
	})
}

func TestSpendSubscriptionCreditBackendChatTokenIsFalse(t *testing.T) {

	log.SetLevel(log.DebugLevel)

	samusSWSClientMock := new(mocks.HttpMock)
	samusSWSClient2Mock := new(clients.SamusSWSClientMock)
	paymentsMock := new(clients.PaymentsClientMock)
	subsMock := new(clients.SubscriptionsClientMock)
	voyagerMock := new(clients.VoyagerClientMock)
	dynamoMock := new(dynamo.IUserDaoMock)

	b := Backend{stats: config.Statsd(),
		samusSWSClient:        samusSWSClientMock,
		samusSWSClientWrapper: samusSWSClient2Mock,
		userDao:               dynamoMock,
		paymentsClient:        paymentsMock,
		subscriptionsClient:   subsMock,
		voyagerClient:         voyagerMock,
	}

	Convey("When IsSubscriptionMessage: false", t, func() {
		profile := payments.PurchaseProfile{
			ID: 123,
		}
		balanceResponse := BalanceResponse{CreditBalance: 1}

		buf, err := json.Marshal(balanceResponse)
		if err != nil {
			log.Debug(err)
		}
		mockedResp := new(http.Response)
		mockedResp.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
		mockedResp.StatusCode = http.StatusOK
		samusSWSClientMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockedResp, nil)

		dynamoMock.On("GetOrCreate", mock.Anything, mock.Anything).Return(dynamo.User{IsSubscriptionMessage: false}, nil)
		paymentsMock.On("InternalCompletePurchase", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(&profile, 200, nil)
		product1 := substwirp.Product{
			ShortName: "TEST_SHORTNAME_TIER_WRONG",
			Price:     999,
			Tier:      "2000",
		}
		product2 := substwirp.Product{
			ShortName: "TEST_SHORTNAME_GOOD",
			Price:     499,
			Tier:      "1000",
		}
		subsResp := substwirp.GetChannelProductsResponse{
			Products: []*substwirp.Product{&product1, &product2},
		}
		subsMock.On("GetChannelProducts", mock.Anything, mock.Anything).Return(&subsResp, nil)
		voyagerResponse := voyagertwirp.GetUserChannelSubscriptionWithPaymentsDataResponse{
			Subscription: nil,
		}
		voyagerMock.On("GetUserChannelSubscriptionWithPaymentsData", mock.Anything, mock.Anything).Return(&voyagerResponse, nil)

		paidUpgradesResponse := &substwirp.GetPaidUpgradesResponse{
			PaidUpgrades: make([]*substwirp.PaidUpgrade, 0),
		}

		subsMock.On("GetPaidUpgrades", mock.Anything, mock.Anything).Return(paidUpgradesResponse, nil)

		spendCreditResponse := clients.SpendCreditResponse{CreditBalance: 0, UserID: userID}
		buf, err = json.Marshal(spendCreditResponse)
		if err != nil {
			log.Debug(err)
		}
		mockedResp = new(http.Response)
		mockedResp.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
		mockedResp.StatusCode = http.StatusOK
		samusSWSClient2Mock.On("SpendCredit", mock.Anything, mock.Anything).Return(&spendCreditResponse, nil)

		resp, e := b.SpendSubscriptionCredit(context.TODO(), userID, broadcasterID)

		So(e, ShouldBeNil)
		So(resp.UserID, ShouldEqual, userID)
		So(resp.BroadcasterID, ShouldEqual, broadcasterID)
		So(resp.SubscriptionCreditBalance, ShouldEqual, 0)
	})
}

func TestSpendSubscriptionCreditBackendWhenActivePaidSubscription(t *testing.T) {

	log.SetLevel(log.DebugLevel)

	samusSWSClientMock := new(mocks.HttpMock)
	samusSWSClient2Mock := new(clients.SamusSWSClientMock)
	paymentsMock := new(clients.PaymentsClientMock)
	subsMock := new(clients.SubscriptionsClientMock)
	voyagerMock := new(clients.VoyagerClientMock)
	dynamoMock := new(dynamo.IUserDaoMock)

	b := Backend{stats: config.Statsd(),
		samusSWSClient:        samusSWSClientMock,
		samusSWSClientWrapper: samusSWSClient2Mock,
		userDao:               dynamoMock,
		paymentsClient:        paymentsMock,
		subscriptionsClient:   subsMock,
		voyagerClient:         voyagerMock,
	}

	Convey("When failed because has paid sub", t, func() {
		profile := payments.PurchaseProfile{
			ID: 123,
		}
		balanceResponse := BalanceResponse{CreditBalance: 1}

		buf, err := json.Marshal(balanceResponse)
		if err != nil {
			log.Debug(err)
		}
		mockedResp := new(http.Response)
		mockedResp.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
		mockedResp.StatusCode = http.StatusOK
		samusSWSClientMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockedResp, nil)

		dynamoMock.On("GetOrCreate", mock.Anything, mock.Anything).Return(dynamo.User{IsSubscriptionMessage: false}, nil)
		paymentsMock.On("InternalCompletePurchase", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(&profile, 200, nil)
		product1 := substwirp.Product{
			ShortName: "TEST_SHORTNAME_TIER_WRONG",
			Price:     999,
			Tier:      "2000",
		}
		product2 := substwirp.Product{
			ShortName: "TEST_SHORTNAME_GOOD",
			Price:     499,
			Tier:      "1000",
		}
		subsResp := substwirp.GetChannelProductsResponse{
			Products: []*substwirp.Product{&product1, &product2},
		}
		subsMock.On("GetChannelProducts", mock.Anything, mock.Anything).Return(&subsResp, nil)

		voyagerResponse := voyagertwirp.GetUserChannelSubscriptionWithPaymentsDataResponse{
			Subscription: &voyagertwirp.SubscriptionWithPaymentsData{
				Subscription: &voyagertwirp.Subscription{
					UserId: userID,
					Type:   0,
				},
			},
		}
		voyagerMock.On("GetUserChannelSubscriptionWithPaymentsData", mock.Anything, mock.Anything).Return(&voyagerResponse, nil)

		resp, e := b.SpendSubscriptionCredit(context.TODO(), userID, broadcasterID)

		So(resp, ShouldBeNil)
		So(e, ShouldNotBeNil)
		So(e.Error(), ShouldEqual, "HAS_PAID_SUB")
	})
}

func TestSpendSubscriptionCreditBackendWhenPaidUpgrades(t *testing.T) {

	log.SetLevel(log.DebugLevel)
	samusSWSClientMock := new(mocks.HttpMock)
	samusSWSClient2Mock := new(clients.SamusSWSClientMock)
	paymentsMock := new(clients.PaymentsClientMock)
	subsMock := new(clients.SubscriptionsClientMock)
	voyagerMock := new(clients.VoyagerClientMock)
	dynamoMock := new(dynamo.IUserDaoMock)

	b := Backend{stats: config.Statsd(),
		samusSWSClient:        samusSWSClientMock,
		samusSWSClientWrapper: samusSWSClient2Mock,
		userDao:               dynamoMock,
		paymentsClient:        paymentsMock,
		subscriptionsClient:   subsMock,
		voyagerClient:         voyagerMock,
	}

	Convey("When failed because has paid sub", t, func() {
		profile := payments.PurchaseProfile{
			ID: 123,
		}
		balanceResponse := BalanceResponse{CreditBalance: 1}

		buf, err := json.Marshal(balanceResponse)
		if err != nil {
			log.Debug(err)
		}
		mockedResp := new(http.Response)
		mockedResp.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
		mockedResp.StatusCode = http.StatusOK
		samusSWSClientMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockedResp, nil)

		dynamoMock.On("GetOrCreate", mock.Anything, mock.Anything).Return(dynamo.User{IsSubscriptionMessage: false}, nil)
		paymentsMock.On("InternalCompletePurchase", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(&profile, 200, nil)
		product1 := substwirp.Product{
			ShortName: "TEST_SHORTNAME_TIER_WRONG",
			Price:     999,
			Tier:      "2000",
		}
		product2 := substwirp.Product{
			ShortName: "TEST_SHORTNAME_GOOD",
			Price:     499,
			Tier:      "1000",
		}
		subsResp := substwirp.GetChannelProductsResponse{
			Products: []*substwirp.Product{&product1, &product2},
		}
		subsMock.On("GetChannelProducts", mock.Anything, mock.Anything).Return(&subsResp, nil)

		voyagerResponse := voyagertwirp.GetUserChannelSubscriptionWithPaymentsDataResponse{
			Subscription: &voyagertwirp.SubscriptionWithPaymentsData{
				Subscription: &voyagertwirp.Subscription{
					UserId: userID,
					Type:   0,
				},
			},
		}
		voyagerMock.On("GetUserChannelSubscriptionWithPaymentsData", mock.Anything, mock.Anything).Return(&voyagerResponse, nil)

		start := time.Now().Add(time.Hour)
		startTimeStamp, err := ptypes.TimestampProto(start)
		So(err, ShouldBeNil)

		baseKey := substwirp.SubKey{
			Product: "Rubber duckies",
			Owner:   "Ducky lord",
			Origin:  "Quack quack",
		}

		upgradeKey := substwirp.SubKey{
			Product: "Bubble bath",
			Owner:   "Bubble lord",
			Origin:  "Blub blub",
		}

		paidUpgrade := substwirp.PaidUpgrade{
			BaseKey:       &baseKey,
			UpgradeKey:    &upgradeKey,
			EntitlementId: "1",
			StartDate:     startTimeStamp,
			Tier:          "1000",
			Price:         499,
		}

		paidUpgradesResponse := &substwirp.GetPaidUpgradesResponse{
			PaidUpgrades: []*substwirp.PaidUpgrade{&paidUpgrade},
		}

		subsMock.On("GetPaidUpgrades", mock.Anything, mock.Anything).Return(&paidUpgradesResponse, nil)

		resp, e := b.SpendSubscriptionCredit(context.TODO(), userID, broadcasterID)

		So(resp, ShouldBeNil)
		So(e, ShouldNotBeNil)
		So(e.Error(), ShouldEqual, "HAS_PAID_SUB")
	})
}

func TestSpendSubscriptionCreditBackendWhenUserIsBlocked(t *testing.T) {

	log.SetLevel(log.DebugLevel)
	samusSWSClientMock := new(mocks.HttpMock)
	samusSWSClient2Mock := new(clients.SamusSWSClientMock)
	paymentsMock := new(clients.PaymentsClientMock)
	subsMock := new(clients.SubscriptionsClientMock)
	dynamoMock := new(dynamo.IUserDaoMock)

	b := Backend{stats: config.Statsd(),
		samusSWSClient:        samusSWSClientMock,
		samusSWSClientWrapper: samusSWSClient2Mock,
		userDao:               dynamoMock,
		paymentsClient:        paymentsMock,
		subscriptionsClient:   subsMock,
	}

	Convey("When Payments API fails with a conflict", t, func() {

		balanceResponse := BalanceResponse{CreditBalance: 1}
		buf, err := json.Marshal(balanceResponse)
		if err != nil {
			log.Debug(err)
		}
		mockedResp := new(http.Response)
		mockedResp.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
		mockedResp.StatusCode = http.StatusOK
		samusSWSClientMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockedResp, nil)

		product1 := substwirp.Product{
			ShortName: "TEST_SHORTNAME_BLOCKED",
			Price:     499,
			Id:        "689376",
			Tier:      "1000",
		}
		subsResp := substwirp.GetChannelProductsResponse{
			Products: []*substwirp.Product{&product1},
		}
		subsMock.On("GetChannelProducts", mock.Anything, mock.Anything).Return(&subsResp, nil)

		paymentsMock.On("InternalCompletePurchase", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(nil, 409, errors.New("Conflict"))

		resp, e := b.SpendSubscriptionCredit(context.TODO(), userID, broadcasterID)

		So(e, ShouldNotBeNil)
		So(resp, ShouldBeNil)
	})

}
