package backend

import (
	"errors"
	"time"

	substwirp "code.justin.tv/revenue/subscriptions/twirp"
	"code.justin.tv/samus/gateway/clients"
	log "github.com/sirupsen/logrus"
	"golang.org/x/net/context"
)

// UnspendSubscriptionCredit reverses spent credit
func (b *Backend) UnspendSubscriptionCredit(ctx context.Context, userID, broadcasterID string) (*UnspendSubscriptionCreditResponse, error) {
	logger := log.WithFields(log.Fields{
		"userID":        userID,
		"broadcasterID": broadcasterID,
	})

	getProductsRequest := &substwirp.GetChannelProductsRequest{
		ChannelId: broadcasterID,
	}
	// step 2: check if channel can receive subscriptions (partnered) and get short name.
	getChannelProductsResponse, err := b.subscriptionsClient.GetChannelProducts(ctx, getProductsRequest)

	if err != nil {
		logger.WithError(err).Error("[UnspendSubscriptionCredit] ", "subscriptionsClient.GetChannelProducts failed: ", err)
		return nil, err
	}

	logger.Infof("[UnspendSubscriptionCredit] Products: [%+v]\nContext: [%+v]", getChannelProductsResponse, ctx)

	// Check if products exist for this Channel
	if len(getChannelProductsResponse.Products) == 0 {
		logger.Error("[UnspendSubscriptionCredit] ", "Broadcaster/Channel can't receive subscriptions.")
		return nil, errors.New("NO_PRODUCTS_FOR_CHANNEL")
	}

	products := getChannelProductsResponse.Products

	eligibleProducts := FilterBy(products, func(p *substwirp.Product) bool {
		// Filter down products based on no tier or tier 1000
		return p.GetTier() == tierCustom || p.GetTier() <= tier1000
	})

	// Check if products that are tier 1000 or less
	if len(eligibleProducts) == 0 {
		logger.Error("[UnspendSubscriptionCredit] ", "Broadcaster/Channel has no eligible products based on tier.")
		return nil, errors.New("NO_PRODUCTS_FOR_CHANNEL")
	}

	shortName := eligibleProducts[0].ShortName
	orderID := userID + shortName + time.Now().UTC().Format(time.RFC3339)

	logger = log.WithFields(log.Fields{
		"userID":        userID,
		"broadcasterID": broadcasterID,
		"shortName":     shortName,
		"orderID":       orderID,
	})

	unspendCreditRequest := &clients.UnspendCreditRequest{
		UserID:  userID,
		OrderID: orderID,
	}

	samusUnspendResponse, err := b.samusSWSClientWrapper.UnspendCredit(ctx, unspendCreditRequest)
	if err != nil {
		logger.WithError(err).Error("[UnspendSubscriptionCredit] ", "samusSWSClient.UnspendCredit failed: ", err.Error())
		return nil, errors.New("COULD NOT UNSPEND CREDIT")
	}

	logger.Infof("[UnspendSubscriptionCredit] SUCCESS : [%+v]\nContext: [%+v]", samusUnspendResponse, ctx)
	return &UnspendSubscriptionCreditResponse{
		samusUnspendResponse.TransactionID,
	}, nil
}
