package backend

import (
	"bytes"
	"fmt"
	"io"
	"net/http"

	"code.justin.tv/common/twitchhttp"
	"golang.org/x/net/context"

	"encoding/json"

	log "github.com/sirupsen/logrus"
)

func closeBody(body io.ReadCloser) {
	err := body.Close()
	if err != nil {
		log.WithError(err)
	}
}

// requireOK is used to wrap Do and check for a http.StatusOK
func requireOK(resp *http.Response, e error) (*http.Response, int, error) {
	//log.Debug("Validating Response: ", resp)
	log.Debug("Validating error: ", e)

	// Check if there was an error on the wire call
	if e != nil {
		if resp != nil {
			closeBody(resp.Body)
		}
		return nil, http.StatusInternalServerError, e
	}

	// If call was okay but it was not a 200
	if resp.StatusCode != http.StatusOK {
		var buf bytes.Buffer
		status := http.StatusBadGateway
		if _, err := io.Copy(&buf, resp.Body); err != nil {
			return nil, status, err
		}
		defer closeBody(resp.Body)

		if resp.StatusCode == http.StatusBadRequest {
			status = http.StatusBadRequest
		}
		if resp.StatusCode == http.StatusNotFound {
			status = http.StatusNotFound
		}
		return nil, status, fmt.Errorf("%s", buf.Bytes())

	}
	return resp, http.StatusOK, nil
}

func sendRequestAndParseResponse(c twitchhttp.Client, ctx context.Context, req *http.Request, v interface{}) (int, error) {

	resp, s, err := requireOK(c.Do(ctx, req, twitchhttp.ReqOpts{}))
	if err != nil {
		return s, err
	}

	defer closeBody(resp.Body)

	//log.Debug("Raw Response: ", resp)

	err = json.NewDecoder(resp.Body).Decode(&v)
	if err != nil {
		log.WithError(err).Error("Error Decoding response to interface :", err)
		return http.StatusInternalServerError, err
	}

	//log.Debug("Decorated interface :", v)

	return http.StatusOK, nil
}

func validProductForPrimeSpend(productID string) bool {
	var blockedProductMap = map[string]bool{
		"689376":  true, // OWL 2018
		"1189177": true, // OWL 2019
	}

	return !blockedProductMap[productID]
}
