package cache

import (
	"fmt"
	"strings"
	"time"

	"github.com/patrickmn/go-cache"
)

const (
	DefaultExpiration      time.Duration = 5 * time.Minute // 5 minutes
	DefaultCleanupInterval time.Duration = 0
)

type OffersCacheKey struct {
	CountryCode string
	Locale      string
}

var ExpirationDuration time.Duration

type OffersCache struct {
	cache *cache.Cache
}

type IOffersCache interface {
	GetOffers(key OffersCacheKey) (interface{}, bool)
	SetOffers(key OffersCacheKey, value interface{})
}

func (offersCache *OffersCache) GetOffers(key OffersCacheKey) (interface{}, bool) {
	keyAsString := getOffersCacheKeyAsString(key)
	offers, found := offersCache.cache.Get(keyAsString)
	return offers, found
}

func (offersCache *OffersCache) SetOffers(key OffersCacheKey, value interface{}) {
	keyAsString := getOffersCacheKeyAsString(key)
	offersCache.cache.Set(keyAsString, value, ExpirationDuration)
}

func getOffersCacheKeyAsString(key OffersCacheKey) string {
	return fmt.Sprintf("%s-%s", key.CountryCode, key.Locale)
}

func getOffersCacheKeyFromString(key string) OffersCacheKey {
	offerCacheKeyItems := strings.Split(key, "-")
	return OffersCacheKey{
		CountryCode: offerCacheKeyItems[0],
		Locale:      offerCacheKeyItems[1],
	}
}

func setExpirationDuration(duration time.Duration) {
	ExpirationDuration = duration
}

func NewOffersCache() IOffersCache {
	offersCache := &OffersCache{}
	setExpirationDuration(DefaultExpiration)
	cache := cache.New(ExpirationDuration, DefaultCleanupInterval)
	offersCache.cache = cache
	return offersCache
}

func NewOffersCacheWithExpiration(duration time.Duration) IOffersCache {
	offersCache := &OffersCache{}
	setExpirationDuration(duration)
	cache := cache.New(duration, DefaultCleanupInterval)
	offersCache.cache = cache
	return offersCache
}
