package samus_gateway

import (
	"encoding/json"
	"net/url"
	"strconv"
	"strings"

	"bytes"
	"net/http"

	"code.justin.tv/foundation/twitchclient"
	. "code.justin.tv/samus/gateway/util"
	"golang.org/x/net/context"
)

const (
	defaultStatSampleRate = 0.1
	defaultTimingXactName = "samus-gateway"
)

type Client interface {
	GetPrimeStatus(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (*PrimeStatusResponse, error)

	GetBalance(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (*GetBalanceResponse, error)
	CanSpendPrimeCredit(ctx context.Context, userID string, productID string, reqOpts *twitchclient.ReqOpts) (*CanSpendPrimeCreditResponse, error)
	SpendSubscriptionCredit(ctx context.Context, userID string, broadcasterID string, reqOpts *twitchclient.ReqOpts) (*SpendSubscriptionCreditResponse, *ErrorResponse, error)

	UnspendSubscriptionCredit(ctx context.Context, userID string, broadcasterID string, reqOpts *twitchclient.ReqOpts) (*ErrorResponse, error)

	ClaimPrimeOffer(ctx context.Context, userID string, offerID string, locale string, jsonStr []byte, reqOptsIn *twitchclient.ReqOpts) (*PrimeOfferEntitlementResponse, *ErrorResponse, error)
	PlaceOrder(ctx context.Context, userID string, offerID string, idempotenceKey string, attributionChannel string, dateOverride string, reqOpts *twitchclient.ReqOpts) (*PlaceOrderResponse, *ErrorResponse, error)
	GetOrdersByCustomer(ctx context.Context, userID string, nextToken string, pageSize int, offerID string, orderID string, reqOpts *twitchclient.ReqOpts) (*GetOrdersByCustomerResponse, *ErrorResponse, error)
	ListInventory(ctx context.Context, userID string, amazonCustomerID string, entitlementAccountType string, itemIds []string, nextToken string, maxResults int, entitlementStatusFilters []string, reqOpts *twitchclient.ReqOpts) (*ListInventoryResponse, *ErrorResponse, error)
	ClearOfferClaimCodeForUser(ctx context.Context, userID string, offerID string, reqOptsIn *twitchclient.ReqOpts) (*ClearOfferClaimCodeForUserResponse, *ErrorResponse, error)
	GetCurrentPrimeOffers(ctx context.Context, userID string, countryCode string, locale string, dateOverride string, reqOpts *twitchclient.ReqOpts) (*CurrentPrimeOffersResponse, error)
	GetCurrentPrimeOffersForUser(ctx context.Context, userID string, countryCode string, locale string, dateOverride string, reqOpts *twitchclient.ReqOpts) (*CurrentPrimeOffersResponse, error)
	GetCurrentOffersWithEligibility(ctx context.Context, userID string, countryCode string, locale string, dateOverride string, reqOpts *twitchclient.ReqOpts) (*GetCurrentOffersWithEligibilityResponse, error)
	GetPrimeOfferEntitlement(ctx context.Context, userID string, offerID string, locale string, reqOptsIn *twitchclient.ReqOpts) (*PrimeOfferEntitlementResponse, error)

	GetDynamicStrings(ctx context.Context, userID string, stringIds []string, countryCode string, locale string, dateOverride string, reqOptsIn *twitchclient.ReqOpts) (*GetDynamicStringResponse, error)

	CreateAccountLink(ctx context.Context, userID string, gameName string, gameAccount GameAccount, reqOptsIn *twitchclient.ReqOpts) (*CreateAccountLinkResponse, int, error)
	GetAccountLink(ctx context.Context, userID string, gameName string, reqOptsIn *twitchclient.ReqOpts) (*GetAccountLinkResponse, error)
	DeleteAccountLink(ctx context.Context, userID string, gameName string, reqOptsIn *twitchclient.ReqOpts) (*DeleteAccountLinkResponse, error)

	GetPrimeSettings(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (*GetPrimeSettingsResponse, error)
	SetPrimeSettings(ctx context.Context, userID string, primeSettings PrimeSettings, reqOpts *twitchclient.ReqOpts) (*SetPrimeSettingsResponse, error)
}

type client struct {
	twitchclient.Client
}

func NewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}
	twitchClient, err := twitchclient.NewClient(conf)
	return &client{twitchClient}, err
}

func (c *client) GetPrimeStatus(ctx context.Context, userID string, reqOptsIn *twitchclient.ReqOpts) (*PrimeStatusResponse, error) {
	var primeStatusResponse PrimeStatusResponse

	url := url.URL{Path: "/api/users/" + userID + "/samus/status"}
	req, err := c.NewRequest("GET", url.String(), nil)
	if err != nil {
		return nil, err
	}
	combinedReqOpts := twitchclient.MergeReqOpts(reqOptsIn, twitchclient.ReqOpts{
		StatName:       "service.samus-gateway.get_prime_status",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if http.StatusOK != resp.StatusCode {
		return nil, &twitchclient.Error{
			Message:    "Could not GetPrimeStatus",
			StatusCode: resp.StatusCode,
		}
	}

	defer safeClose(resp, &err)

	err = json.NewDecoder(resp.Body).Decode(&primeStatusResponse)
	if err != nil {
		return nil, err
	}

	return &primeStatusResponse, nil
}

func (c *client) GetBalance(ctx context.Context, userID string, reqOptsIn *twitchclient.ReqOpts) (*GetBalanceResponse, error) {
	var getBalanceResponse GetBalanceResponse

	url := url.URL{Path: "/api/users/" + userID + "/samus/balance"}
	req, err := c.NewRequest("GET", url.String(), nil)
	if err != nil {
		return nil, err
	}
	combinedReqOpts := twitchclient.MergeReqOpts(reqOptsIn, twitchclient.ReqOpts{
		StatName:       "service.samus-gateway.get_balance",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if http.StatusOK != resp.StatusCode {
		return nil, &twitchclient.Error{
			Message:    "Could not GetBalance",
			StatusCode: resp.StatusCode,
		}
	}

	defer safeClose(resp, &err)

	err = json.NewDecoder(resp.Body).Decode(&getBalanceResponse)
	if err != nil {
		return nil, err
	}

	return &getBalanceResponse, nil
}

func (c *client) GetDynamicStrings(ctx context.Context, userID string, stringIds []string, countryCode string, locale string, dateOverride string, reqOptsIn *twitchclient.ReqOpts) (*GetDynamicStringResponse, error) {

	var getDynamicStringsResponse GetDynamicStringResponse

	q := url.Values{}
	q.Add("userId", userID)
	q.Add("stringIds", strings.Join(stringIds, ","))
	q.Add("countryCode", countryCode)
	q.Add("locale", locale)
	q.Add("dateOverride", dateOverride)
	u := &url.URL{Path: "/api/string", RawQuery: q.Encode()}
	req, err := c.NewRequest("GET", u.String(), nil)
	if err != nil {
		return nil, err
	}
	combinedReqOpts := twitchclient.MergeReqOpts(reqOptsIn, twitchclient.ReqOpts{
		StatName:       "service.samus-gateway.get_dynamic_strings",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if http.StatusOK != resp.StatusCode {
		return nil, &twitchclient.Error{
			Message:    "Could not GetDynamicStrings",
			StatusCode: resp.StatusCode,
		}
	}

	defer safeClose(resp, &err)

	err = json.NewDecoder(resp.Body).Decode(&getDynamicStringsResponse)
	if err != nil {
		return nil, err
	}

	return &getDynamicStringsResponse, nil
}

// CanSpendPrimeCredit checks whether the user can spend their prime credit on a given channel
func (c *client) CanSpendPrimeCredit(ctx context.Context, userID string, productID string, reqOptsIn *twitchclient.ReqOpts) (*CanSpendPrimeCreditResponse, error) {
	var canSpendPrimeCreditResponse CanSpendPrimeCreditResponse

	q := url.Values{}
	q.Add("productId", productID)
	url := url.URL{Path: "/api/users/" + userID + "/samus/credit", RawQuery: q.Encode()}

	req, err := c.NewRequest("GET", url.String(), nil)
	if err != nil {
		return nil, err
	}
	combinedReqOpts := twitchclient.MergeReqOpts(reqOptsIn, twitchclient.ReqOpts{
		StatName:       "service.samus-gateway.can_spend_prime_credit",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if http.StatusOK != resp.StatusCode {
		return nil, &twitchclient.Error{
			Message:    "Could not retrieve CanSpendPrimeCredit",
			StatusCode: resp.StatusCode,
		}
	}

	defer safeClose(resp, &err)

	err = json.NewDecoder(resp.Body).Decode(&canSpendPrimeCreditResponse)
	if err != nil {
		return nil, err
	}

	return &canSpendPrimeCreditResponse, nil
}

func (c *client) SpendSubscriptionCredit(ctx context.Context, userID string, broadcasterID string, reqOptsIn *twitchclient.ReqOpts) (*SpendSubscriptionCreditResponse, *ErrorResponse, error) {
	var spendSubscriptionCreditResponse SpendSubscriptionCreditResponse

	if validationErr := ValidateTuids(userID, broadcasterID); validationErr != nil {
		return nil, nil, validationErr
	}

	apiURL := url.URL{Path: "/api/users/" + userID + "/subscription_credit/spend/" + broadcasterID}

	req, err := c.NewRequest("PUT", apiURL.String(), nil)
	if err != nil {
		return nil, nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOptsIn, twitchclient.ReqOpts{
		StatName:       "service.samus-gateway.spend_subscription_credit",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, nil, err
	}

	if resp.StatusCode != http.StatusOK {
		var spendSubscriptionCreditError ErrorResponse
		err = json.NewDecoder(resp.Body).Decode(&spendSubscriptionCreditError)
		if err != nil {
			return nil, nil, err
		}
		return nil, &spendSubscriptionCreditError, nil
	}

	defer safeClose(resp, &err)

	err = json.NewDecoder(resp.Body).Decode(&spendSubscriptionCreditResponse)
	if err != nil {
		return nil, nil, err
	}

	return &spendSubscriptionCreditResponse, nil, nil
}

func (c *client) UnspendSubscriptionCredit(ctx context.Context, userID string, broadcasterID string, reqOptsIn *twitchclient.ReqOpts) (*ErrorResponse, error) {
	if validationErr := ValidateTuids(userID, broadcasterID); validationErr != nil {
		return nil, validationErr
	}

	apiURL := url.URL{Path: "/api/users/" + userID + "/subscription_credit/spend/" + broadcasterID}

	req, err := c.NewRequest("DELETE", apiURL.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOptsIn, twitchclient.ReqOpts{
		StatName:       "service.samus-gateway.unspend_subscription_credit",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		var unspendSubscriptionCreditError ErrorResponse
		err = json.NewDecoder(resp.Body).Decode(&unspendSubscriptionCreditError)
		if err != nil {
			return nil, err
		}
		return &unspendSubscriptionCreditError, nil
	}

	defer safeClose(resp, &err)
	var unspendSubscriptionCreditResponse UnspendSubscriptionCreditResponse
	err = json.NewDecoder(resp.Body).Decode(&unspendSubscriptionCreditResponse)
	if err != nil {
		return nil, err
	}

	return nil, nil
}

// ClaimPrimeOffer entitles the user to a given prime offer and returns localized response data
func (c *client) ClaimPrimeOffer(ctx context.Context, userID string, offerID string, locale string, jsonStr []byte, reqOptsIn *twitchclient.ReqOpts) (*PrimeOfferEntitlementResponse, *ErrorResponse, error) {
	var primeOfferEntitlementResponse PrimeOfferEntitlementResponse

	if validationErr := ValidateOffer(offerID); validationErr != nil {
		return nil, nil, validationErr
	}

	q := url.Values{}
	q.Add("locale", locale)
	url := url.URL{Path: "/api/users/" + userID + "/prime/claim/offer/" + offerID, RawQuery: q.Encode()}

	req, err := c.NewRequest("POST", url.String(), bytes.NewReader(jsonStr))
	if err != nil {
		return nil, nil, err
	}
	combinedReqOpts := twitchclient.MergeReqOpts(reqOptsIn, twitchclient.ReqOpts{
		StatName:       "service.samus-gateway.claim_prime_offer",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, nil, err
	}

	if http.StatusOK != resp.StatusCode {
		var claimOfferError ErrorResponse
		err = json.NewDecoder(resp.Body).Decode(&claimOfferError)
		if err != nil {
			return nil, nil, err
		}
		return nil, &claimOfferError, &twitchclient.Error{
			Message:    "Could not ClaimPrimeOffer",
			StatusCode: resp.StatusCode,
		}
	}

	defer safeClose(resp, &err)

	err = json.NewDecoder(resp.Body).Decode(&primeOfferEntitlementResponse)
	if err != nil {
		return nil, nil, err
	}

	return &primeOfferEntitlementResponse, nil, nil
}

// PlaceOrder places an order for a given prime (catalog) offer and returns respective orderId
func (c *client) PlaceOrder(ctx context.Context, userID string, offerID string, idempotenceKey string, attributionChannel string, dateOverride string, reqOptsIn *twitchclient.ReqOpts) (*PlaceOrderResponse, *ErrorResponse, error) {
	var placeOrderResponse PlaceOrderResponse

	url := url.URL{Path: "/api/users/" + userID + "/prime/place/order/" + offerID}

	body, err := json.Marshal(PlaceOrderRequestBody{
		IdempotenceKey:     idempotenceKey,
		AttributionChannel: attributionChannel,
		DateOverride:       dateOverride,
	})
	if err != nil {
		return nil, nil, err
	}
	req, err := c.NewRequest("POST", url.String(), bytes.NewReader(body))
	if err != nil {
		return nil, nil, err
	}
	combinedReqOpts := twitchclient.MergeReqOpts(reqOptsIn, twitchclient.ReqOpts{
		StatName:       "service.samus-gateway.place_order",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, nil, err
	}

	if http.StatusOK != resp.StatusCode {
		var placeOrderError ErrorResponse
		err = json.NewDecoder(resp.Body).Decode(&placeOrderError)
		if err != nil {
			return nil, nil, err
		}
		return nil, &placeOrderError, &twitchclient.Error{
			Message:    "Could not PlaceOrder",
			StatusCode: resp.StatusCode,
		}
	}

	defer safeClose(resp, &err)

	err = json.NewDecoder(resp.Body).Decode(&placeOrderResponse)
	if err != nil {
		return nil, nil, err
	}

	return &placeOrderResponse, nil, nil
}

// GetOrdersByCustomer returns order details for a given customer

func (c *client) GetOrdersByCustomer(ctx context.Context, userID string, nextToken string, pageSize int, offerID string, orderID string, reqOptsIn *twitchclient.ReqOpts) (*GetOrdersByCustomerResponse, *ErrorResponse, error) {
	var getOrdersByCustomerResponse GetOrdersByCustomerResponse

	q := url.Values{}
	q.Add("nextToken", nextToken)
	q.Add("pageSize", strconv.Itoa(pageSize))
	q.Add("offerId", offerID)
	q.Add("orderId", orderID)

	url := url.URL{Path: "/api/users/" + userID + "/prime/orders", RawQuery: q.Encode()}

	req, err := c.NewRequest("GET", url.String(), nil)
	if err != nil {
		return nil, nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOptsIn, twitchclient.ReqOpts{
		StatName:       "service.samus-gateway.get_orders_by_customer",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, nil, err
	}

	if http.StatusOK != resp.StatusCode {
		var getOrdersByCustomerError ErrorResponse
		err = json.NewDecoder(resp.Body).Decode(&getOrdersByCustomerError)
		if err != nil {
			return nil, nil, err
		}
		return nil, nil, &twitchclient.Error{
			Message:    "Could not GetOrdersByCustomer",
			StatusCode: resp.StatusCode,
		}
	}

	defer safeClose(resp, &err)

	err = json.NewDecoder(resp.Body).Decode(&getOrdersByCustomerResponse)
	if err != nil {
		return nil, nil, err
	}

	return &getOrdersByCustomerResponse, nil, nil
}

// ListInventory returns inventory details (code offers) for a given customer
func (c *client) ListInventory(ctx context.Context, userID string, amazonCustomerID string, entitlementAccountType string, itemIds []string, nextToken string, maxResults int, entitlementStatusFilters []string, reqOptsIn *twitchclient.ReqOpts) (*ListInventoryResponse, *ErrorResponse, error) {
	var listInventoryResponse ListInventoryResponse

	q := url.Values{}
	q.Add("amazonCustomerId", amazonCustomerID)
	q.Add("entitlementAccountType", entitlementAccountType)
	q.Add("itemIds", strings.Join(itemIds, ","))
	q.Add("nextToken", nextToken)
	q.Add("maxResults", strconv.Itoa(maxResults))
	q.Add("entitlementStatusFilters", strings.Join(entitlementStatusFilters, ","))

	url := url.URL{Path: "/api/users/" + userID + "/prime/inventory", RawQuery: q.Encode()}

	req, err := c.NewRequest("GET", url.String(), nil)
	if err != nil {
		return nil, nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOptsIn, twitchclient.ReqOpts{
		StatName:       "service.samus-gateway.list_inventory",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, nil, err
	}

	if http.StatusOK != resp.StatusCode {
		var listInventoryError ErrorResponse
		err = json.NewDecoder(resp.Body).Decode(&listInventoryError)
		if err != nil {
			return nil, nil, err
		}
		return nil, nil, &twitchclient.Error{
			Message:    "Could not get ListInventory",
			StatusCode: resp.StatusCode,
		}
	}

	defer safeClose(resp, &err)

	err = json.NewDecoder(resp.Body).Decode(&listInventoryResponse)
	if err != nil {
		return nil, nil, err
	}

	return &listInventoryResponse, nil, nil
}

// GetCurrentPrimeOffers gets active prime offers based on the identified country and returns localized response data
func (c *client) GetCurrentPrimeOffers(ctx context.Context, userID string, countryCode string, locale string, dateOverride string, reqOptsIn *twitchclient.ReqOpts) (*CurrentPrimeOffersResponse, error) {
	var currentPrimeOffersResponse CurrentPrimeOffersResponse

	q := url.Values{}
	q.Add("locale", locale)
	q.Add("userId", userID)
	q.Add("countryCode", countryCode)
	q.Add("dateOverride", dateOverride)
	url := url.URL{Path: "/api/prime/offers", RawQuery: q.Encode()}

	req, err := c.NewRequest("GET", url.String(), nil)
	if err != nil {
		return nil, err
	}
	combinedReqOpts := twitchclient.MergeReqOpts(reqOptsIn, twitchclient.ReqOpts{
		StatName:       "service.samus-gateway.get_current_prime_offers",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if http.StatusOK != resp.StatusCode {
		return nil, &twitchclient.Error{
			Message:    "Could not GetCurrentPrimeOffers",
			StatusCode: resp.StatusCode,
		}
	}

	defer safeClose(resp, &err)

	err = json.NewDecoder(resp.Body).Decode(&currentPrimeOffersResponse)
	if err != nil {
		return nil, err
	}

	return &currentPrimeOffersResponse, nil
}

// GetCurrentPrimeOffersForUser gets active prime offers based on the identified country and user and returns localized response data
func (c *client) GetCurrentPrimeOffersForUser(ctx context.Context, userID string, countryCode string, locale string, dateOverride string, reqOptsIn *twitchclient.ReqOpts) (*CurrentPrimeOffersResponse, error) {
	var currentPrimeOffersResponse CurrentPrimeOffersResponse

	queryParams := url.Values{}
	queryParams.Add("locale", locale)
	queryParams.Add("userId", userID)
	queryParams.Add("countryCode", countryCode)
	queryParams.Add("dateOverride", dateOverride)
	url := url.URL{Path: "/api/prime/offersv2", RawQuery: queryParams.Encode()}

	req, err := c.NewRequest("GET", url.String(), nil)
	if err != nil {
		return nil, err
	}
	combinedReqOpts := twitchclient.MergeReqOpts(reqOptsIn, twitchclient.ReqOpts{
		StatName:       "service.samus-gateway.get_current_prime_offers_for_user",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if http.StatusOK != resp.StatusCode {
		return nil, &twitchclient.Error{
			Message:    "Could not GetCurrentPrimeOffersForUser",
			StatusCode: resp.StatusCode,
		}
	}

	defer safeClose(resp, &err)

	err = json.NewDecoder(resp.Body).Decode(&currentPrimeOffersResponse)
	if err != nil {
		return nil, err
	}

	return &currentPrimeOffersResponse, nil
}

func (c *client) GetCurrentOffersWithEligibility(ctx context.Context, userID string, countryCode string, locale string, dateOverride string, reqOptsIn *twitchclient.ReqOpts) (*GetCurrentOffersWithEligibilityResponse, error) {
	if userID != "" {
		if err := ValidateTuid(userID); err != nil {
			return nil, err
		}
	}

	var getCurrentOffersWithEligibilityResponse GetCurrentOffersWithEligibilityResponse

	queryParams := url.Values{}
	queryParams.Add("locale", locale)
	queryParams.Add("userId", userID)
	queryParams.Add("countryCode", countryCode)
	queryParams.Add("dateOverride", dateOverride)
	url := url.URL{Path: "/api/prime/offers/eligibility", RawQuery: queryParams.Encode()}

	req, err := c.NewRequest("GET", url.String(), nil)
	if err != nil {
		return nil, err
	}
	combinedReqOpts := twitchclient.MergeReqOpts(reqOptsIn, twitchclient.ReqOpts{
		StatName:       "service.samus-gateway.get_current_offers_with_eligibility",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if http.StatusOK != resp.StatusCode {
		return nil, &twitchclient.Error{
			Message:    "Could not GetCurrentOffersWithEligibility",
			StatusCode: resp.StatusCode,
		}
	}

	defer safeClose(resp, &err)

	err = json.NewDecoder(resp.Body).Decode(&getCurrentOffersWithEligibilityResponse)
	if err != nil {
		return nil, err
	}

	return &getCurrentOffersWithEligibilityResponse, nil
}

// GetPrimeOfferEntitlement gets the users claim and entitlement info from an previously claimed prime offer and returns localized response data
func (c *client) GetPrimeOfferEntitlement(ctx context.Context, userID, offerID, locale string, reqOptsIn *twitchclient.ReqOpts) (*PrimeOfferEntitlementResponse, error) {
	var primeOfferEntitlementResponse PrimeOfferEntitlementResponse

	q := url.Values{}
	q.Add("locale", locale)
	url := url.URL{Path: "/api/users/" + userID + "/prime/claim/offer/" + offerID, RawQuery: q.Encode()}

	req, err := c.NewRequest("GET", url.String(), nil)
	if err != nil {
		return nil, err
	}
	combinedReqOpts := twitchclient.MergeReqOpts(reqOptsIn, twitchclient.ReqOpts{
		StatName:       "service.samus-gateway.get_prime_offer_entitlement",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if http.StatusOK != resp.StatusCode {
		return nil, &twitchclient.Error{
			Message:    "Could not GetPrimeOfferEntitlement",
			StatusCode: resp.StatusCode,
		}
	}

	defer safeClose(resp, &err)

	err = json.NewDecoder(resp.Body).Decode(&primeOfferEntitlementResponse)
	if err != nil {
		return nil, err
	}

	return &primeOfferEntitlementResponse, nil
}

// Creates an account link between the user and the specified game account
func (c *client) CreateAccountLink(ctx context.Context, userID string, gameName string, gameAccount GameAccount, reqOptsIn *twitchclient.ReqOpts) (*CreateAccountLinkResponse, int, error) {
	var createAccountLinkResponse CreateAccountLinkResponse

	url := url.URL{Path: "/api/users/" + userID + "/link/" + gameName}

	body, err := json.Marshal(CreateAccountLinkRequestBody{
		GameAccount: gameAccount,
	})
	if err != nil {
		return nil, http.StatusBadRequest, err
	}

	req, err := c.NewRequest("POST", url.String(), bytes.NewReader(body))
	if err != nil {
		return nil, http.StatusBadRequest, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOptsIn, twitchclient.ReqOpts{
		StatName:       "service.samus-gateway.create_account_link",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, http.StatusInternalServerError, err
	}

	if http.StatusConflict == resp.StatusCode {
		return nil, resp.StatusCode, nil
	}

	if http.StatusOK != resp.StatusCode {
		return nil, resp.StatusCode, &twitchclient.Error{
			Message:    "Could not create account link",
			StatusCode: resp.StatusCode,
		}
	}

	defer safeClose(resp, &err)

	err = json.NewDecoder(resp.Body).Decode(&createAccountLinkResponse)
	if err != nil {
		return nil, http.StatusInternalServerError, err
	}

	return &createAccountLinkResponse, resp.StatusCode, err
}

// Gets an existing account link, or nil if none exists
func (c *client) GetAccountLink(ctx context.Context, userID string, gameName string, reqOptsIn *twitchclient.ReqOpts) (*GetAccountLinkResponse, error) {
	var getAccountLinkResponse GetAccountLinkResponse

	url := url.URL{Path: "/api/users/" + userID + "/link/" + gameName}
	req, err := c.NewRequest("GET", url.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOptsIn, twitchclient.ReqOpts{
		StatName:       "service.samus-gateway.get_account_link",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if http.StatusNotFound == resp.StatusCode {
		return nil, &twitchclient.Error{
			Message:    "no account link found",
			StatusCode: resp.StatusCode,
		}
	} else if http.StatusOK != resp.StatusCode {
		return nil, &twitchclient.Error{
			Message:    "could not get account link",
			StatusCode: resp.StatusCode,
		}
	}

	defer safeClose(resp, &err)

	err = json.NewDecoder(resp.Body).Decode(&getAccountLinkResponse)
	if err != nil {
		return nil, err
	}

	return &getAccountLinkResponse, err
}

// Deletes an account link, returning the deleted link if one existed
func (c *client) DeleteAccountLink(ctx context.Context, userID string, gameName string, reqOptsIn *twitchclient.ReqOpts) (*DeleteAccountLinkResponse, error) {
	var deleteAccountLinkResponse DeleteAccountLinkResponse

	url := url.URL{Path: "/api/users/" + userID + "/link/" + gameName}
	req, err := c.NewRequest("DELETE", url.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOptsIn, twitchclient.ReqOpts{
		StatName:       "service.samus-gateway.delete_account_link",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if http.StatusOK != resp.StatusCode {
		return nil, &twitchclient.Error{
			Message:    "could not delete account link",
			StatusCode: resp.StatusCode,
		}
	}

	defer safeClose(resp, &err)

	err = json.NewDecoder(resp.Body).Decode(&deleteAccountLinkResponse)
	if err != nil {
		return nil, err
	}

	return &deleteAccountLinkResponse, err
}

func (c *client) GetPrimeSettings(ctx context.Context, userID string, reqOptsIn *twitchclient.ReqOpts) (*GetPrimeSettingsResponse, error) {
	var getPrimeSettingsResponse GetPrimeSettingsResponse

	url := url.URL{Path: "api/users/" + userID + "/prime/settings"}
	req, err := c.NewRequest("GET", url.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOptsIn, twitchclient.ReqOpts{
		StatName:       "service.samus-gateway.get_prime_user_settings",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, &twitchclient.Error{
			Message:    "Could not get Prime user settings",
			StatusCode: resp.StatusCode,
		}
	}

	defer safeClose(resp, &err)

	err = json.NewDecoder(resp.Body).Decode(&getPrimeSettingsResponse)
	if err != nil {
		return nil, err
	}

	return &getPrimeSettingsResponse, nil
}

func (c *client) SetPrimeSettings(ctx context.Context, userID string, primeSettings PrimeSettings, reqOptsIn *twitchclient.ReqOpts) (*SetPrimeSettingsResponse, error) {
	var setPrimeSettingsResponse SetPrimeSettingsResponse

	url := url.URL{Path: "api/users/" + userID + "/prime/settings"}

	body, err := json.Marshal(SetPrimeSettingsRequestBody{
		IsSubscriptionMessage: primeSettings.IsSubscriptionMessage,
	})
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest("POST", url.String(), bytes.NewReader(body))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOptsIn, twitchclient.ReqOpts{
		StatName:       "service.samus-gateway.set_prime_user_settings",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, &twitchclient.Error{
			Message:    "Could not set Prime user settings",
			StatusCode: resp.StatusCode,
		}
	}

	defer safeClose(resp, &err)

	err = json.NewDecoder(resp.Body).Decode(&setPrimeSettingsResponse)
	if err != nil {
		return nil, err
	}

	return &setPrimeSettingsResponse, nil
}

// ClaimPrimeOffer entitles the user to a given prime offer and returns localized response data
func (c *client) ClearOfferClaimCodeForUser(ctx context.Context, userID string, offerID string, reqOptsIn *twitchclient.ReqOpts) (*ClearOfferClaimCodeForUserResponse, *ErrorResponse, error) {
	var clearOfferClaimCodeForUserResponse ClearOfferClaimCodeForUserResponse

	if validationErr := ValidateOffer(offerID); validationErr != nil {
		return nil, nil, validationErr
	}

	q := url.Values{}
	url := url.URL{Path: "/api/users/" + userID + "/prime/unclaim/offer/" + offerID, RawQuery: q.Encode()}

	req, err := c.NewRequest("POST", url.String(), nil)
	if err != nil {
		return nil, nil, err
	}
	combinedReqOpts := twitchclient.MergeReqOpts(reqOptsIn, twitchclient.ReqOpts{
		StatName:       "service.samus-gateway.clear_offer_claim_code_for_user",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, nil, err
	}

	if http.StatusOK != resp.StatusCode {
		return nil, nil, &twitchclient.Error{
			Message:    "Could not Clear Offer Claim Code for user",
			StatusCode: resp.StatusCode,
		}
	}

	defer safeClose(resp, &err)

	err = json.NewDecoder(resp.Body).Decode(&clearOfferClaimCodeForUserResponse)
	if err != nil {
		return nil, nil, err
	}

	return &clearOfferClaimCodeForUserResponse, nil, nil
}

func safeClose(resp *http.Response, err *error) {
	if resp != nil {
		if cerr := resp.Body.Close(); cerr != nil && *err == nil {
			*err = cerr
		}
	}
}
