package samus_gateway

import (
	"bytes"
	"context"
	"encoding/json"
	"io"
	"io/ioutil"
	"net/http"
	"strings"
	"testing"

	"code.justin.tv/foundation/twitchclient"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
)

type MockTwitchClient struct{}

func (c *MockTwitchClient) Do(ctx context.Context, req *http.Request, opts twitchclient.ReqOpts) (*http.Response, error) {
	switch {
	case strings.Contains(req.URL.String(), "/subscription_credit/spend"):
		return mockSpendSubscriptionCreditAPI(ctx, req, opts), nil
	case strings.Contains(req.URL.String(), "/prime/claim/offer/"):
		return mockClaimOfferAPI(ctx, req, opts), nil
	case strings.Contains(req.URL.String(), "/prime/unclaim/offer"):
		return mockClearOfferClaimCode(ctx, req, opts), nil
	case strings.Contains(req.URL.String(), "/offers/eligibility"):
		return mockGetCurrentOffersWithEligibilityAPI(ctx, req, opts), nil
	default:
		return nil, nil
	}
}

func (c *MockTwitchClient) DoNoContent(ctx context.Context, req *http.Request, opts twitchclient.ReqOpts) (*http.Response, error) {
	return c.Do(ctx, req, opts)
}

func (c *MockTwitchClient) DoJSON(ctx context.Context, json interface{}, req *http.Request, opts twitchclient.ReqOpts) (*http.Response, error) {
	return c.Do(ctx, req, opts)
}

func (c *MockTwitchClient) NewRequest(method string, path string, body io.Reader) (*http.Request, error) {
	return http.NewRequest(method, path, body)
}

const (
	invalidUserId = "../12345"
	validUserId   = "12345"

	invalidBroadcasterId = "../54321"
	validBroadcasterId   = "54321"
)

func TestGetCurrentOffersWithEligibility(t *testing.T) {
	Convey("GetCurrentOffersWithEligibility", t, func() {
		log.SetLevel(log.DebugLevel)
		mockTwitchClient := new(MockTwitchClient)

		gatewayClient := &client{mockTwitchClient}
		So(gatewayClient, ShouldNotBeNil)

		Convey("Returns an error when userId is not numeric", func() {
			mockUserId := invalidUserId
			actualResp, e := gatewayClient.GetCurrentOffersWithEligibility(context.TODO(), mockUserId, "", "", "", nil)

			So(e, ShouldNotBeNil)
			So(actualResp, ShouldBeNil)
		})

		Convey("Does not return an error when userId is numeric", func() {
			mockUserId := validUserId
			actualResp, e := gatewayClient.GetCurrentOffersWithEligibility(context.TODO(), mockUserId, "", "", "", nil)

			So(e, ShouldBeNil)
			So(actualResp, ShouldNotBeNil)
			So(actualResp.CurrentOffers, ShouldNotBeEmpty)
		})
	})
}

func TestSpendSubscriptionCredit(t *testing.T) {
	Convey("SpendSubscriptionCredit", t, func() {
		log.SetLevel(log.DebugLevel)
		mockTwitchClient := new(MockTwitchClient)

		gatewayClient := &client{mockTwitchClient}
		So(gatewayClient, ShouldNotBeNil)

		Convey("Returns an error when broadcasterId is not numeric", func() {
			mockUserId := validUserId
			mockBroadcasterId := invalidBroadcasterId
			actualSpendCreditResponse, _, e := gatewayClient.SpendSubscriptionCredit(context.TODO(), mockUserId, mockBroadcasterId, nil)

			So(e, ShouldNotBeNil)
			So(actualSpendCreditResponse, ShouldBeNil)
		})

		Convey("Returns an error when userId is not numeric", func() {
			mockUserId := invalidUserId
			mockBroadcasterId := validBroadcasterId
			actualSpendCreditResponse, _, e := gatewayClient.SpendSubscriptionCredit(context.TODO(), mockUserId, mockBroadcasterId, nil)

			So(e, ShouldNotBeNil)
			So(actualSpendCreditResponse, ShouldBeNil)
		})

		Convey("Returns an error when both userId and broadcasterId are not numeric", func() {
			mockUserId := invalidUserId
			mockBroadcasterId := validBroadcasterId
			actualSpendCreditResponse, _, e := gatewayClient.SpendSubscriptionCredit(context.TODO(), mockUserId, mockBroadcasterId, nil)

			So(e, ShouldNotBeNil)
			So(actualSpendCreditResponse, ShouldBeNil)
		})

		Convey("Returns no error when userId and broadcasterId are numeric", func() {
			mockUserId := validUserId
			mockBroadcasterId := validBroadcasterId
			actualSpendCreditResponse, _, e := gatewayClient.SpendSubscriptionCredit(context.TODO(), mockUserId, mockBroadcasterId, nil)

			So(e, ShouldBeNil)
			So(actualSpendCreditResponse.UserID, ShouldEqual, mockUserId)
			So(actualSpendCreditResponse.BroadcasterID, ShouldEqual, mockBroadcasterId)
		})
	})
}

func TestClaimPrimeOffer(t *testing.T) {
	Convey("ClaimPrimeOffer", t, func() {
		log.SetLevel(log.DebugLevel)
		mockUserId := validUserId
		mockLocale := "en_US"
		mockTwitchClient := new(MockTwitchClient)

		gatewayClient := &client{mockTwitchClient}
		So(gatewayClient, ShouldNotBeNil)

		Convey("Returns an error if offerId is of invalid length", func() {
			mockOfferId := "78b414d4-b2f5-83e65-debf-dd34764182cb"
			actualClaimPrimeOfferResponse, _, e := gatewayClient.ClaimPrimeOffer(context.TODO(), mockUserId, mockOfferId, mockLocale, nil, nil)

			So(e, ShouldNotBeNil)
			So(actualClaimPrimeOfferResponse, ShouldBeNil)
		})

		Convey("Returns an error if offerId contains unexpected characters", func() {
			mockOfferId := "2ed07990-971-4a../-a30e-b5f628cfd707"
			actualClaimPrimeOfferResponse, _, e := gatewayClient.ClaimPrimeOffer(context.TODO(), mockUserId, mockOfferId, mockLocale, nil, nil)

			So(e, ShouldNotBeNil)
			So(actualClaimPrimeOfferResponse, ShouldBeNil)
		})

		Convey("Returns an error if offerId contains invalid alphanumeric characters", func() {
			mockOfferId := "84b5bb57-edd7-5eg0-eee4-8d0e3d2b393a"
			actualClaimPrimeOfferResponse, _, e := gatewayClient.ClaimPrimeOffer(context.TODO(), mockUserId, mockOfferId, mockLocale, nil, nil)

			So(e, ShouldNotBeNil)
			So(actualClaimPrimeOfferResponse, ShouldBeNil)
		})

		Convey("Returns an error of offerId contains incorrect number of tuples", func() {
			mockOfferId := "b0b5da90-3d36-0a32-e4792a2ebd9a"
			actualClaimPrimeOfferResponse, _, e := gatewayClient.ClaimPrimeOffer(context.TODO(), mockUserId, mockOfferId, mockLocale, nil, nil)

			So(e, ShouldNotBeNil)
			So(actualClaimPrimeOfferResponse, ShouldBeNil)
		})

		Convey("Returns no error with a valid offerId", func() {
			mockOfferId := "f2b3b001-d78c-c0f8-1c41-70940bca1faf"
			actualClaimPrimeOfferResponse, _, e := gatewayClient.ClaimPrimeOffer(context.TODO(), mockUserId, mockOfferId, mockLocale, nil, nil)

			So(e, ShouldBeNil)
			So(actualClaimPrimeOfferResponse, ShouldNotBeNil)
			So(actualClaimPrimeOfferResponse.OfferID, ShouldEqual, mockOfferId)
			So(actualClaimPrimeOfferResponse.UserID, ShouldEqual, mockUserId)
		})
	})
}

func TestClearOfferClaimCodeForUser(t *testing.T) {
	// TODO mock and verify called ValidateOffer function instead of testing implementation https://jira.agscollab.com/browse/TPCX-2179
	Convey("ClearOfferClaimCodeForUser", t, func() {
		log.SetLevel(log.DebugLevel)
		mockUserId := validUserId
		mockTwitchClient := new(MockTwitchClient)

		gatewayClient := &client{mockTwitchClient}
		So(gatewayClient, ShouldNotBeNil)

		Convey("Returns an error if offerId is of invalid length", func() {
			mockOfferId := "78b414d4-b2f5-83e65-debf-dd34764182cb"
			actualClearOfferClaimCodeResponse, _, e := gatewayClient.ClearOfferClaimCodeForUser(context.TODO(), mockUserId, mockOfferId, nil)

			So(e, ShouldNotBeNil)
			So(actualClearOfferClaimCodeResponse, ShouldBeNil)
		})

		Convey("Returns an error if offerId contains unexpected characters", func() {
			mockOfferId := "2ed07990-971-4a../-a30e-b5f628cfd707"
			actualClearOfferClaimCodeResponse, _, e := gatewayClient.ClearOfferClaimCodeForUser(context.TODO(), mockUserId, mockOfferId, nil)

			So(e, ShouldNotBeNil)
			So(actualClearOfferClaimCodeResponse, ShouldBeNil)
		})

		Convey("Returns an error if offerId contains invalid alphanumeric characters", func() {
			mockOfferId := "84b5bb57-edd7-5eg0-eee4-8d0e3d2b393a"
			actualClearOfferClaimCodeResponse, _, e := gatewayClient.ClearOfferClaimCodeForUser(context.TODO(), mockUserId, mockOfferId, nil)

			So(e, ShouldNotBeNil)
			So(actualClearOfferClaimCodeResponse, ShouldBeNil)
		})

		Convey("Returns an error of offerId contains incorrect number of tuples", func() {
			mockOfferId := "b0b5da90-3d36-0a32-e4792a2ebd9a"
			actualClearOfferClaimCodeResponse, _, e := gatewayClient.ClearOfferClaimCodeForUser(context.TODO(), mockUserId, mockOfferId, nil)

			So(e, ShouldNotBeNil)
			So(actualClearOfferClaimCodeResponse, ShouldBeNil)
		})

		Convey("Returns no error with a valid offerId", func() {
			mockOfferId := "f2b3b001-d78c-c0f8-1c41-70940bca1faf"
			actualClearOfferClaimCodeResponse, _, e := gatewayClient.ClearOfferClaimCodeForUser(context.TODO(), mockUserId, mockOfferId, nil)

			So(e, ShouldBeNil)
			So(actualClearOfferClaimCodeResponse, ShouldNotBeNil)
			So(actualClearOfferClaimCodeResponse.Success, ShouldBeTrue)
		})
	})
}

func mockGetCurrentOffersWithEligibilityAPI(ctx context.Context, req *http.Request, opts twitchclient.ReqOpts) *http.Response {
	currentPrimeOffer := OfferWithNoClaimData{
		ContentCategories:       []string{},
		ExternalOfferUri:        "externalOfferUri",
		PublisherName:           "publisherName",
		ApplicableGame:          "applicableGame",
		ContentDeliveryMethod:   "claimCode",
		Assets:                  []OfferAsset{},
		ContentClaimInstruction: "claimIntructions",
		Priority:                1,
		OfferDescription:        "offerDescription",
		OfferTitle:              "offerTitle",
		OfferID:                 "offerID",
		EndTime:                 "endTime",
		StartTime:               "startTime",
		Tags:                    OfferTag{},
	}

	getCurrentOffersWithEligibilityResponse := GetCurrentOffersWithEligibilityResponse{
		CurrentOffers: []OfferWithEligibility{
			OfferWithEligibility{
				OfferWithNoClaimData: currentPrimeOffer,
				OfferEligibility: OfferEligibility{
					OfferState:        OfferLive,
					CanClaim:          true,
					PrimeGamingRule:   true,
					LinkedAccountRule: true,
				},
			},
		},
	}

	buf, e := json.Marshal(getCurrentOffersWithEligibilityResponse)
	So(e, ShouldBeNil)

	return &http.Response{
		StatusCode: http.StatusOK,
		Request:    req,
		Body:       ioutil.NopCloser(bytes.NewBufferString(string(buf[:]))),
	}
}

// Mocking the API that is invoked by the samus_gateway client
func mockSpendSubscriptionCreditAPI(ctx context.Context, req *http.Request, opts twitchclient.ReqOpts) *http.Response {
	apiParams := strings.Split(req.URL.String(), "?")
	apiURL := strings.Split(apiParams[0], "/")

	spendSubscriptionCreditResponse := SpendSubscriptionCreditResponse{
		UserID:        apiURL[3],
		BroadcasterID: apiURL[6],
	}

	buf, e := json.Marshal(spendSubscriptionCreditResponse)
	So(e, ShouldBeNil)

	return &http.Response{
		StatusCode: http.StatusOK,
		Request:    req,
		Body:       ioutil.NopCloser(bytes.NewBufferString(string(buf[:]))),
	}
}

func mockClaimOfferAPI(ctx context.Context, req *http.Request, opts twitchclient.ReqOpts) *http.Response {
	apiParams := strings.Split(req.URL.String(), "?")
	apiURL := strings.Split(apiParams[0], "/")

	primeOfferEntitlementResponse := PrimeOfferEntitlementResponse{
		HasEntitlement: true,
		UserID:         apiURL[3],
		OfferID:        apiURL[7],
		OfferTitle:     "MOCK_OFFER_TITLE",
	}

	buf, e := json.Marshal(primeOfferEntitlementResponse)
	So(e, ShouldBeNil)

	return &http.Response{
		StatusCode: http.StatusOK,
		Request:    req,
		Body:       ioutil.NopCloser(bytes.NewBufferString(string(buf[:]))),
	}
}

func mockClearOfferClaimCode(ctx context.Context, req *http.Request, opts twitchclient.ReqOpts) *http.Response {
	clearOfferClaimCodeResponse := ClearOfferClaimCodeForUserResponse{
		Success: true,
	}

	buf, e := json.Marshal(clearOfferClaimCodeResponse)
	So(e, ShouldBeNil)

	return &http.Response{
		StatusCode: http.StatusOK,
		Request:    req,
		Body:       ioutil.NopCloser(bytes.NewBufferString(string(buf[:]))),
	}
}
