package clients

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"

	twitchConfig "code.justin.tv/common/config"
	"code.justin.tv/foundation/twitchclient"
	payments "code.justin.tv/revenue/payments-service-go-client/client"
	"code.justin.tv/samus/gateway/settings"
	log "github.com/sirupsen/logrus"
	"golang.org/x/net/context"
)

const (
	PaymentsEndpoint      = "PaymentsEndpoint"
	defaultTimingXactName = "payments-service-rails"
	defaultStatSampleRate = 0.1
)

// Development Environment config
var DevelopmentPaymentsConfig = map[string]string{
	PaymentsEndpoint: "http://payments-service-stage.us-west-2.elasticbeanstalk.com",
}

// Staging Environment config
var StagingPaymentsConfig = map[string]string{
	PaymentsEndpoint: "http://payments-service-stage.us-west-2.elasticbeanstalk.com",
}

// Production Environment config
var ProductionPaymentsConfig = map[string]string{
	PaymentsEndpoint: "http://payments-service-prod.us-west-2.elasticbeanstalk.com",
}

// Redefine payments.Client functions so we can mock.
type PaymentsClienter interface {
	InternalCompletePurchase(ctx context.Context, userID string, productShortName string, request InternalCompletePurchaseRequest, reqOpts *twitchclient.ReqOpts) (*payments.PurchaseProfile, int, error)
	CancelPurchase(ctx context.Context, userID string, productShortName string, request payments.CancelPurchaseRequest, reqOpts *twitchclient.ReqOpts) error
}

// PaymentsClient - embeds payments.Client.
type PaymentsClient struct {
	tc twitchclient.Client
	pc payments.Client
}

// NewTwitchPaymentsClientWrapper creates a new client for use in making service calls.
func NewTwitchPaymentsClientWrapper(conf twitchclient.ClientConf) (PaymentsClienter, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}
	twitchClient, err := twitchclient.NewClient(conf)
	if err != nil {
		return nil, err
	}
	paymentsClient, err := payments.NewClient(conf)
	if err != nil {
		return nil, err
	}
	return &PaymentsClient{
		tc: twitchClient,
		pc: paymentsClient,
	}, err
}

// Creates a Payments client.
func NewPaymentsClient() (PaymentsClienter, error) {

	conf := settings.GetConfiguration(DevelopmentPaymentsConfig, StagingPaymentsConfig, ProductionPaymentsConfig)

	c, err := NewTwitchPaymentsClientWrapper(twitchclient.ClientConf{
		Host:           conf[PaymentsEndpoint],
		TimingXactName: "Payments-Client",
		Stats:          twitchConfig.Statsd(),
	})
	if err != nil {
		log.WithError(err).Error("[NewPaymentsClient] payments.NewClient failed: ", err)
		return nil, err
	}

	return c, nil
}

// PaymentOptions for the purchase request
type PaymentOptions struct {
	OrderID string `json:"order_id"`
}

// InternalCompletePurchaseRequest contains inputs to the InternalCompletePurchase API on payments
type InternalCompletePurchaseRequest struct {
	Options         PaymentOptions `json:"payment_options"`
	PaymentProvider string         `json:"payment_provider"`
	Recurring       bool           `json:"recurring"`
	UserID          string         `json:"user_id"`
	ProductID       string         `json:"product_id"`
	PurchaseRegion  string         `json:"purchase_region"`
}

// InternalCompletePurchase API client wrapper call to create channel subscriptions
func (c *PaymentsClient) InternalCompletePurchase(ctx context.Context, userID string, productShortName string, request InternalCompletePurchaseRequest, reqOpts *twitchclient.ReqOpts) (*payments.PurchaseProfile, int, error) {
	jsonBody, err := json.Marshal(request)
	if err != nil {
		return nil, http.StatusInternalServerError, err
	}

	apiURL := fmt.Sprintf("/internal/users/%s/products/%s/purchase/complete", userID, productShortName)

	req, err := c.tc.NewRequest("POST", apiURL, bytes.NewBuffer(jsonBody))
	if err != nil {
		return nil, http.StatusInternalServerError, err
	}
	req.Header.Set("Content-Type", "application/json")

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.payments_service.complete_purchase",
		StatSampleRate: defaultStatSampleRate,
	})

	var purchaseProfile payments.PurchaseProfile
	resp, err := c.tc.DoJSON(ctx, &purchaseProfile, req, combinedReqOpts)

	defer safeClose(resp, &err)

	if err != nil {
		return nil, resp.StatusCode, err
	}

	return &purchaseProfile, resp.StatusCode, nil
}

// CancelPurchase Wrapper around cancel API
func (c *PaymentsClient) CancelPurchase(ctx context.Context, userID string, productShortName string, request payments.CancelPurchaseRequest, reqOpts *twitchclient.ReqOpts) error {
	return c.pc.CancelPurchase(ctx, userID, productShortName, request, reqOpts)
}
