package clients

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"

	pubclient "code.justin.tv/chat/pubsub-go-pubclient/client"
	"code.justin.tv/foundation/twitchclient"
	"github.com/afex/hystrix-go/hystrix"
)

type OrderCompletionEvent struct {
	Type string                   `json:"type"`
	Data OrderCompletionEventData `json:"data"`
}

// OrderCompletionEvent is the message published to the prime-gaming-offer PubSub topic in Twilight
type OrderCompletionEventData struct {
	OrderId string `json:"orderId"`
	OfferId string `json:"offerId"`
}

func init() {
	hystrix.Configure(map[string]hystrix.CommandConfig{
		"PubSub.Publish": {
			Timeout:               500,
			MaxConcurrentRequests: 1000,
		},
	})
}

//go:generate mockery --name PubSub
type PubSub interface {
	PublishOrderCompletion(event OrderCompletionEvent, userId string) error
}

type PubSubClient struct {
	pubClient pubclient.PubClient
}

func NewPubSub(host string) (PubSub, error) {
	pubsub, err := pubclient.NewPubClient(twitchclient.ClientConf{
		Host: host,
	})
	return &PubSubClient{pubsub}, err
}

func (client *PubSubClient) PublishOrderCompletion(event OrderCompletionEvent, userId string) error {
	return client.publish([]string{fmt.Sprintf("prime-gaming-offer.%s", userId)}, event)
}

func (client *PubSubClient) publish(topic []string, data interface{}) error {
	marshalled, err := json.Marshal(data)
	if err != nil {
		return err
	}

	return hystrix.Do("PubSub.Publish", func() (e error) {
		defer func() {
			if client := recover(); client != nil {
				e = errors.New(fmt.Sprintf("circuit panic %v", client))
			}
		}()
		pubErr := client.pubClient.Publish(context.Background(), topic, string(marshalled), nil)
		return pubErr
	}, nil)
}
