package clients

import (
	"bytes"
	"crypto/tls"
	"crypto/x509"
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"net/http"

	twitchConfig "code.justin.tv/common/config"
	"code.justin.tv/common/twitchhttp"
	internal_models "code.justin.tv/samus/gateway/internal/models"
	"code.justin.tv/samus/gateway/settings"
	log "github.com/sirupsen/logrus"
	"golang.org/x/net/context"
)

const (
	SamusSWSHost     = "SamusSWSHost"
	SamusSWSCertPath = "SamusSWSCertPath"
	RootCertPath     = "RootCertPath"

	SWS_HEADER  = "x-amzn-sws-rest-service"
	SSCS_HEADER = "SamusSubscriptionCreditService"
)

// Development Environment config
var DevelopmentSamusSWSConfig = map[string]string{
	SamusSWSHost:     "https://sws-na.amazonpmi.com",
	SamusSWSCertPath: "/var/app/sws_cert_key_chain",
	RootCertPath:     "certificates/amazon-ca-certs.pem",
}

// Staging Environment config
var StagingSamusSWSConfig = map[string]string{
	SamusSWSHost:     "https://sws-na.amazonpmi.com",
	SamusSWSCertPath: "/var/app/sws_cert_key_chain",
	RootCertPath:     "/var/app/amazon-ca-g3.pem",
}

// Production Environment config
var ProductionSamusSWSConfig = map[string]string{
	SamusSWSHost:     "https://sws-na.amazon.com",
	SamusSWSCertPath: "/var/app/sws_cert_key_chain",
	RootCertPath:     "/var/app/amazon-ca-g3.pem",
}

// Define samusSWSClient functions so we can mock.
type SamusSWSClienter interface {
	SpendCredit(ctx context.Context, spendCreditRequest *SpendCreditRequest) (*SpendCreditResponse, error)
	UnspendCredit(ctx context.Context, unspendCreditRequest *UnspendCreditRequest) (*UnspendCreditResponse, error)
}

// SamusSWSClient - embeds twitchhttp.Client.
type SamusSWSClient struct {
	twitchhttp.Client
}

// Creates a new SamusSWSClient and returns a SamusSWSClienter
func NewSamusSWSClient() (SamusSWSClienter, error) {

	conf := settings.GetConfiguration(DevelopmentSamusSWSConfig, StagingSamusSWSConfig, ProductionSamusSWSConfig)

	SamusSWSCertPath := conf[SamusSWSCertPath]
	clientCert, err := tls.LoadX509KeyPair(SamusSWSCertPath, SamusSWSCertPath)
	if err != nil {
		log.WithError(err).Error("[NewSamusSWSClient] tls.LoadX509KeyPair failed: ", err)
		return nil, err
	}

	rootCertPath := conf[RootCertPath]
	rootCert, err := ioutil.ReadFile(rootCertPath)
	if err != nil {
		log.WithError(err).Error("[NewSamusSWSClient] ioutil.ReadFile failed: ", err)
		return nil, err
	}

	rootCertPool := x509.NewCertPool()
	rootCertPool.AppendCertsFromPEM(rootCert)
	tlsConfig := &tls.Config{
		RootCAs:            rootCertPool,
		Certificates:       []tls.Certificate{clientCert},
		InsecureSkipVerify: false,
	}
	tlsConfig.BuildNameToCertificate()

	c, err := twitchhttp.NewClient(twitchhttp.ClientConf{
		Host:            conf[SamusSWSHost],
		TimingXactName:  "Samus-SWS-Client",
		TLSClientConfig: tlsConfig,
		Stats:           twitchConfig.Statsd(),
	})
	if err != nil {
		log.Debug("[NewSamusSWSClient] twitchhttp.NewClient failed ", err)
		return nil, err
	}

	return &SamusSWSClient{c}, nil
}

/******************************/
/*** [SSCS] SpendCredit ***/
/******************************/

type SpendCreditRequest struct {
	UserID        string   `json:"twitchUserId"`
	OrderID       string   `json:"orderId"`
	TransactionID string   `json:"transactionId"`
	ProductID     string   `json:"productId"`
	Channels      []string `json:"channels"`
}

type SpendCreditResponse struct {
	UserID         string `json:"twitchUserId"`
	TransactionID  string `json:"transactionId"`
	CreditBalance  int    `json:"subscriptionCreditBalance"`
	PurchaseRegion string `json:"purchaseRegion"`
}

func (sc *SamusSWSClient) SpendCredit(ctx context.Context, spendCreditRequest *SpendCreditRequest) (*SpendCreditResponse, error) {
	logger := log.WithFields(log.Fields{
		"UserID":        spendCreditRequest.UserID,
		"OrderID":       spendCreditRequest.OrderID,
		"TransactionID": spendCreditRequest.TransactionID,
		"ProductID":     spendCreditRequest.ProductID,
		"Channels":      spendCreditRequest.Channels,
	})

	path := fmt.Sprintf("/samus/users/%s/subscription_credit/spend/order/%s", spendCreditRequest.UserID, spendCreditRequest.OrderID)

	reqMsgBody, err := json.Marshal(spendCreditRequest)
	if err != nil {
		logger.Error("[SpendCredit]", "json.Marshal failed: ", err.Error())
		return nil, err
	}

	req, err := sc.NewRequest(
		"PUT",
		path,
		bytes.NewBuffer(reqMsgBody),
	)
	if err != nil {
		logger.Error("[SpendCredit]", "NewRequest failed: ", err.Error())
		return nil, err
	}

	req.Header.Set(SWS_HEADER, SSCS_HEADER)
	spendCreditResponse := SpendCreditResponse{}

	resp, err := sc.Do(ctx, req, twitchhttp.ReqOpts{})
	if err != nil {
		logger.Error("[SpendCredit]", "requestCall failed: ", err.Error())
		return nil, err
	}

	defer safeClose(resp, &err)

	// If call was okay but it was not a 200
	if resp.StatusCode != http.StatusOK {
		var coralError internal_models.CoralError
		err = json.NewDecoder(resp.Body).Decode(&coralError)
		if err != nil {
			logger.Error("[SpendCredit]", "Error Decoding response to interface :", err.Error())
			return nil, err
		}
		return nil, errors.New(coralError.ErrorCode)
	}

	err = json.NewDecoder(resp.Body).Decode(&spendCreditResponse)
	if err != nil {
		return nil, err
	}

	return &spendCreditResponse, nil
}

/******************************/
/*** [SSCS] UnspendCredit ***/
/******************************/
type UnspendCreditRequest struct {
	UserID        string `json:"twitchUserId"`
	OrderID       string `json:"orderId"`
	TransactionID string `json:"transactionId"`
}

type UnspendCreditResponse struct {
	TransactionID string `json:"transactionId"`
}

func (sc *SamusSWSClient) UnspendCredit(ctx context.Context, unspendCreditRequest *UnspendCreditRequest) (*UnspendCreditResponse, error) {
	logger := log.WithFields(log.Fields{
		"UserID":        unspendCreditRequest.UserID,
		"OrderID":       unspendCreditRequest.OrderID,
		"TransactionID": unspendCreditRequest.TransactionID,
	})

	path := fmt.Sprintf("/samus/users/%s/subscription_credit/spend/order/%s", unspendCreditRequest.UserID, unspendCreditRequest.OrderID)

	req, err := sc.NewRequest(
		"DELETE",
		path,
		nil,
	)
	if err != nil {
		logger.Error("[UnspendCredit]", "NewRequest failed: ", err.Error())
		return nil, err
	}

	req.Header.Set(SWS_HEADER, SSCS_HEADER)
	unspendCreditResponse := UnspendCreditResponse{}

	resp, err := sc.Do(ctx, req, twitchhttp.ReqOpts{})
	if err != nil {
		logger.Error("[UnspendCredit]", "requestCall failed: ", err.Error())
		return nil, err
	}

	defer safeClose(resp, &err)

	// If call was okay but it was not a 200
	if resp.StatusCode != http.StatusOK {
		var coralError internal_models.CoralError
		err = json.NewDecoder(resp.Body).Decode(&coralError)
		if err != nil {
			logger.Error("[UnspendCredit]", "Error Decoding response to interface :", err.Error())
			return nil, err
		}
		return nil, errors.New(coralError.ErrorCode)
	}

	err = json.NewDecoder(resp.Body).Decode(&unspendCreditResponse)
	if err != nil {
		return nil, err
	}

	return &unspendCreditResponse, nil
}
