package clients

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"net/http"
	"time"

	twitchConfig "code.justin.tv/common/config"
	"code.justin.tv/common/twitchhttp"
	gatewayClient "code.justin.tv/samus/gateway/client"
	internal_models "code.justin.tv/samus/gateway/internal/models"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	v4 "github.com/aws/aws-sdk-go/aws/signer/v4"
	"github.com/aws/aws-sdk-go/service/sts"
	uuid "github.com/satori/go.uuid"
	log "github.com/sirupsen/logrus"
	"golang.org/x/net/context"
)

const (
	TPROX_ACCESS_ARN         = "arn:aws:iam::461497548915:role/SamusGatewayAccessRole"
	TPROX_SIGV4_SERVICE_NAME = "TwitchProxyService"
	TPROX_SIGV4_AWS_REGION   = "us-east-1"
	TPROX_INPUT_SERVICE_NAME = "com.amazonaws.twitchproxyservice#TwitchProxyService"
	TPROX_CLIENT_ID          = "SamusGateway"
	TPROX_HOST               = "https://twitch-proxy.amazon.com/samus/"
	TPROX_CLIENT_NAME        = "Samus-TProx-Client"

	GET_CURRENT_OFFERS_FOR_USER_IN_COUNTRY_OPERATION_NAME = "com.amazonaws.twitchproxyservice#GetCurrentOffersForUserInCountry"
	CLEAR_OFFER_CLAIM_CODE_FOR_USER_OPERATION_NAME        = "com.amazonaws.twitchproxyservice#ClearOfferClaimCodeForCustomer"
	GET_ELIGIBILITY_STATUS_OPERATION_NAME                 = "com.amazonaws.twitchproxyservice#GetEligibilityStatus"
	BATCH_GET_ELIGIBILITY_STATUS_OPERATION_NAME           = "com.amazonaws.twitchproxyservice#BatchGetEligibilityStatus"
	PLACE_ORDER_OPERATION_NAME                            = "com.amazonaws.twitchproxyservice#PlaceOrder"
	GET_ORDERS_BY_CUSTOMER_OPERATION_NAME                 = "com.amazonaws.twitchproxyservice#GetOrdersByCustomer"
	LIST_INVENTORY_OPERATION_NAME                         = "com.amazonaws.twitchproxyservice#ListInventory"

	GET_CURRENT_OFFERS_FOR_USER_IN_COUNTRY_TARGET_NAME = "com.amazonaws.twitchproxyservice.TwitchProxyService.GetCurrentOffersForUserInCountry"
	CLEAR_OFFER_CLAIM_CODE_FOR_USER_TARGET_NAME        = "com.amazonaws.twitchproxyservice.TwitchProxyService.ClearOfferClaimCodeForCustomer"
	GET_ELIGIBILITY_STATUS_TARGET_NAME                 = "com.amazonaws.twitchproxyservice.TwitchProxyService.GetEligibilityStatus"
	BATCH_GET_ELIGIBILITY_STATUS_TARGET_NAME           = "com.amazonaws.twitchproxyservice.TwitchProxyService.BatchGetEligibilityStatus"
	PLACE_ORDER_TARGET_NAME                            = "com.amazonaws.twitchproxyservice.TwitchProxyService.PlaceOrder"
	GET_ORDERS_BY_CUSTOMER_TARGET_NAME                 = "com.amazonaws.twitchproxyservice.TwitchProxyService.GetOrdersByCustomer"
	LIST_INVENTORY_TARGET_NAME                         = "com.amazonaws.twitchproxyservice.TwitchProxyService.ListInventory"
)

var TPROX_OPERATION_TO_TARGET_NAMES = map[string]string{
	GET_CURRENT_OFFERS_FOR_USER_IN_COUNTRY_OPERATION_NAME: GET_CURRENT_OFFERS_FOR_USER_IN_COUNTRY_TARGET_NAME,
	CLEAR_OFFER_CLAIM_CODE_FOR_USER_OPERATION_NAME:        CLEAR_OFFER_CLAIM_CODE_FOR_USER_TARGET_NAME,
	GET_ELIGIBILITY_STATUS_OPERATION_NAME:                 GET_ELIGIBILITY_STATUS_TARGET_NAME,
	BATCH_GET_ELIGIBILITY_STATUS_OPERATION_NAME:           BATCH_GET_ELIGIBILITY_STATUS_TARGET_NAME,
	PLACE_ORDER_OPERATION_NAME:                            PLACE_ORDER_TARGET_NAME,
	GET_ORDERS_BY_CUSTOMER_OPERATION_NAME:                 GET_ORDERS_BY_CUSTOMER_TARGET_NAME,
	LIST_INVENTORY_OPERATION_NAME:                         LIST_INVENTORY_TARGET_NAME,
}

// Define samusTProxClient functions so we can mock.
type SamusTProxClienter interface {
	GetCurrentOffersForUserInCountry(ctx context.Context, getCurrentOffersForUserInCountryRequest *GetCurrentOffersForUserInCountryRequest) (*GetCurrentOffersForUserInCountryResponse, int, error)
	ClearOfferClaimCodeForUser(ctx context.Context, ClearOfferClaimCodeForUserRequest *ClearOfferClaimCodeForUserRequest) (*ClearOfferClaimCodeForUserResponse, int, error)
	GetEligibilityStatus(ctx context.Context, getEligibilityStatusTwitchRequest *GetEligibilityStatusTwitchRequest) (*GetEligibilityStatusResponse, int, error)
	BatchGetEligibilityStatus(ctx context.Context, batchGetEligibilityStatusTwitchRequest *BatchGetEligibilityStatusTwitchRequest) (*BatchGetEligibilityStatusResponse, int, error)
	PlaceOrder(ctx context.Context, placeOrderTwitchRequest *PlaceOrderTwitchRequest) (*PlaceOrderResponse, int, error)
	GetOrdersByCustomer(ctx context.Context, getOrdersByCustomerTwitchRequest *GetOrdersByCustomerTwitchRequest) (*GetOrdersByCustomerResponse, int, error)
	ListInventory(ctx context.Context, listInventoryTwitchRequest *ListInventoryTwitchRequest) (*ListInventoryResponse, int, error)
}

// SigV4Signer - TProx SigV4 signer operations
type SigV4Signer interface {
	Sign(r *http.Request, body io.ReadSeeker, service, region string, signTime time.Time) (http.Header, error)
}

// SamusTProxClient - Embeds the HTTP client and SigV4 auth signer
type SamusTProxClient struct {
	tproxSigV4Signer SigV4Signer
	twitchhttp.Client
}

// NewSamusTProxClient - Creates a new SamusTProxClient and returns a SamusTProxClienter
func NewSamusTProxClient() (SamusTProxClienter, error) {
	c, err := twitchhttp.NewClient(twitchhttp.ClientConf{
		Host:           TPROX_HOST,
		TimingXactName: TPROX_CLIENT_NAME,
		Stats:          twitchConfig.Statsd(),
	})
	if err != nil {
		log.Debug("[NewSamusTProxClient] twitchhttp.NewClient failed ", err)
		return nil, err
	}

	tproxSigV4Signer := createTProxSigV4Signer()

	return &SamusTProxClient{tproxSigV4Signer, c}, nil
}

// createTProxSigV4Signer - Creates an AWS SigV4 signer using credentials retrieved from TProx
func createTProxSigV4Signer() *v4.Signer {
	awsConfig := &aws.Config{
		Region:   aws.String(TPROX_SIGV4_AWS_REGION),
		Endpoint: aws.String(fmt.Sprintf("https://sts.%s.amazonaws.com", TPROX_SIGV4_AWS_REGION)),
	}

	stsClient := sts.New(session.New(awsConfig))

	arp := &stscreds.AssumeRoleProvider{
		Duration:     900 * time.Second,
		ExpiryWindow: 10 * time.Second,
		RoleARN:      TPROX_ACCESS_ARN,
		Client:       stsClient,
	}

	credentials := credentials.NewCredentials(arp)

	return v4.NewSigner(credentials)
}

// serializeToReader - Serializes a JSON object into a IO reader for HTTP requests
func serializeToReader(v interface{}) (io.ReadSeeker, error) {
	payload, err := serialize(v)
	if err != nil {
		return nil, err
	}

	return bytes.NewReader(payload), nil
}

// serialize - Serializes a JSON object into a byte array
func serialize(v interface{}) ([]byte, error) {
	b, err := json.Marshal(v)
	if err != nil {
		return nil, err
	}

	return b, nil
}

type TProxInput struct {
	Operation *string      `json:"Operation"`
	Service   *string      `json:"Service"`
	Input     *interface{} `json:"Input"`
}

// createTProxRequest - Creates a SigV4-signed HTTP request using the provided arguments. The request is in Coral format so that TProx can consume it.
func (sc *SamusTProxClient) createTProxRequest(callMethod string, input interface{}, operationName string, reqID string) (*http.Request, error) {

	serviceName := TPROX_INPUT_SERVICE_NAME // can't create a pointer to a const

	tproxInput := TProxInput{
		Operation: &operationName,
		Service:   &serviceName,
		Input:     &input,
	}

	payload, err := serializeToReader(tproxInput)

	req, err := sc.NewRequest(callMethod, "", payload)
	if err != nil {
		log.WithError(err).Error("[createTProxRequest] NewRequest failed: ", err)
		return nil, err
	}

	req.Header.Set("Content-Type", "application/json; charset=UTF-8")
	req.Header.Set("X-Requested-With", "XMLHttpRequest")
	req.Header.Set("x-amzn-RequestId", reqID)
	req.Header.Set("x-amz-client-id", TPROX_CLIENT_ID)
	req.Header.Set("X-Amz-Target", TPROX_OPERATION_TO_TARGET_NAMES[operationName])

	_, err = sc.tproxSigV4Signer.Sign(req, payload, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, time.Now())

	if err != nil {
		log.WithError(err).Error("[createTProxRequest] Signing request with SigV4 failed: ", err)
		return nil, err
	}

	return req, nil
}

/**********************************************/
/*** [SOS] GetCurrentOffersForUserInCountry ***/
/**********************************************/

type GetCurrentOffersForUserInCountryRequest struct {
	TUID         string `json:"twitchUserId"`
	CountryCode  string `json:"countryCode"`
	Locale       string `json:"locale"`
	DateOverride string `json:"dateOverride"`
	ClientID     string `json:"clientID"`
}

type TProxGetCurrentOffersForUserInCountryInput struct {
	Request   *GetCurrentOffersForUserInCountryRequest `json:"GetCurrentOffersForUserInCountryRequest"`
	ClientID  *string                                  `json:"clientID"`
	RequestID *string                                  `json:"requestID"`
}

type TProxGetCurrentOffersForUserInCountryResponse struct {
	Output GetCurrentOffersForUserInCountryResponse `json:"Output"`
}

type GetCurrentOffersForUserInCountryResponse struct {
	Offers        []gatewayClient.OfferWithClaimData `json:"offerList"`
	ClaimStatuses map[string]ClaimStatus             `json:"claimStatuses"`
}

type ClaimStatus struct {
	ClaimHint gatewayClient.ClaimHint `json:"claimHint"`
}

// GetCurrentOffersForUserInCountry - TProx Coral operation, proxies the SOS GetCurrentOffersForUserInCountry Coral operation.
// Retrieves currently available offers for the requested user.
func (sc *SamusTProxClient) GetCurrentOffersForUserInCountry(ctx context.Context, getCurrentOffersForUserInCountryRequest *GetCurrentOffersForUserInCountryRequest) (*GetCurrentOffersForUserInCountryResponse, int, error) {
	reqID := uuid.NewV4().String()
	fields := log.Fields{
		"tuid":         getCurrentOffersForUserInCountryRequest.TUID,
		"countryCode":  getCurrentOffersForUserInCountryRequest.CountryCode,
		"locale":       getCurrentOffersForUserInCountryRequest.Locale,
		"dateOverride": getCurrentOffersForUserInCountryRequest.DateOverride,
		"clientID":     getCurrentOffersForUserInCountryRequest.ClientID,
		"requestID":    reqID,
	}

	clientID := TPROX_CLIENT_ID // can't create a pointer to a const
	requestInput := TProxGetCurrentOffersForUserInCountryInput{
		Request:   getCurrentOffersForUserInCountryRequest,
		ClientID:  &clientID,
		RequestID: &reqID,
	}

	req, err := sc.createTProxRequest(
		"POST",
		requestInput,
		GET_CURRENT_OFFERS_FOR_USER_IN_COUNTRY_OPERATION_NAME,
		reqID,
	)
	if err != nil || req == nil {
		log.WithError(err).WithFields(fields).Error("[GetCurrentOffersForUserInCountry] createTProxRequest failed: ", err)
		return nil, http.StatusInternalServerError, err
	}

	resp, err := sc.Do(ctx, req, twitchhttp.ReqOpts{})
	defer safeClose(resp, &err)

	if err != nil {
		log.WithError(err).WithFields(fields).Error("[GetCurrentOffersForUserInCountry] requestCall failed: ", err)
		return nil, http.StatusInternalServerError, err
	}

	// Error if call was okay but it was not a 200
	// (ex: TProx correctly forwards the call to SOS, but SOS returns a 500)
	if resp.StatusCode != http.StatusOK {
		var coralError internal_models.CoralError
		err = json.NewDecoder(resp.Body).Decode(&coralError)
		if err != nil {
			log.WithError(err).WithFields(fields).Error("[GetCurrentOffersForUserInCountry] Error decoding non-200 response into interface :", err)
			return nil, resp.StatusCode, err
		}
		log.WithError(err).WithFields(fields).Error("[GetCurrentOffersForUserInCountry] response was not a 200:", err)
		return nil, resp.StatusCode, errors.New(coralError.ErrorCode)
	}

	var response TProxGetCurrentOffersForUserInCountryResponse
	err = json.NewDecoder(resp.Body).Decode(&response)
	if err != nil {
		log.WithError(err).WithFields(fields).Error("[GetCurrentOffersForUserInCountry] Error decoding response into interface :", err)
		return nil, resp.StatusCode, err
	}

	return &response.Output, resp.StatusCode, nil
}

/**********************************************/
/*** [SOS] ClearOfferClaimCodeForUser ***/
/**********************************************/

type ClearOfferClaimCodeForUserRequest struct {
	TwitchUserID  string `json:"twitchUserId"`
	MarketplaceID string `json:"marketplaceId,omitempty"`
	OfferID       string `json:"offerId"`
	CsAgent       string `json:"csAgent"`
	CsContactID   string `json:"csContactId"`
}

type TProxClearOfferClaimCodeForUserInput struct {
	Request   *ClearOfferClaimCodeForUserRequest `json:"ClearOfferClaimCodeForCustomerRequest"`
	ClientID  *string                            `json:"clientID"`
	RequestID *string                            `json:"requestID"`
}

type TProxClearOfferClaimCodeForUserResponse struct {
	Output ClearOfferClaimCodeForUserResponse `json:"Output"`
}

type ClearOfferClaimCodeForUserResponse struct {
	Success bool `json:"success"`
}

// ClearOfferClaimCodeForCustomer - TProx Coral operation, proxies the SOS ClearOfferClaimCodeForCustomer Coral operation.
// Clears the requested user's claim of the requested offer.
func (sc *SamusTProxClient) ClearOfferClaimCodeForUser(ctx context.Context, clearOfferClaimCodeForUserRequest *ClearOfferClaimCodeForUserRequest) (*ClearOfferClaimCodeForUserResponse, int, error) {
	reqID := uuid.NewV4().String()
	fields := log.Fields{
		"twitchUserID":  clearOfferClaimCodeForUserRequest.TwitchUserID,
		"marketPlaceID": clearOfferClaimCodeForUserRequest.MarketplaceID,
		"offerID":       clearOfferClaimCodeForUserRequest.OfferID,
		"csAgent":       clearOfferClaimCodeForUserRequest.CsAgent,
		"csContactID":   clearOfferClaimCodeForUserRequest.CsContactID,
		"requestID":     reqID,
	}

	clientID := TPROX_CLIENT_ID // can't create a pointer to a const
	requestInput := TProxClearOfferClaimCodeForUserInput{
		Request:   clearOfferClaimCodeForUserRequest,
		ClientID:  &clientID,
		RequestID: &reqID,
	}

	req, err := sc.createTProxRequest(
		"POST",
		requestInput,
		CLEAR_OFFER_CLAIM_CODE_FOR_USER_OPERATION_NAME,
		reqID,
	)
	if err != nil || req == nil {
		log.WithError(err).WithFields(fields).Error("[ClearOfferClaimCodeForUser] createTProxRequest failed: ", err)
		return nil, http.StatusInternalServerError, err
	}

	resp, err := sc.Do(ctx, req, twitchhttp.ReqOpts{})
	defer safeClose(resp, &err)

	if err != nil {
		log.WithError(err).WithFields(fields).Error("[ClearOfferClaimCodeForUser] requestCall failed: ", err)
		return nil, http.StatusInternalServerError, err
	}

	// Error if call was okay but it was not a 200
	// (ex: TProx correctly forwards the call to SOS, but SOS returns a 500)
	if resp.StatusCode != http.StatusOK {
		var coralError internal_models.CoralError
		err = json.NewDecoder(resp.Body).Decode(&coralError)
		if err != nil {
			log.WithError(err).WithFields(fields).Error("[ClearOfferClaimCodeForUser] Error decoding non-200 response into interface :", err)
			return nil, resp.StatusCode, err
		}
		log.WithError(err).WithFields(fields).Error("[ClearOfferClaimCodeForUser] response was not a 200:", err)
		return nil, resp.StatusCode, errors.New(coralError.ErrorCode)
	}

	var response TProxClearOfferClaimCodeForUserResponse
	err = json.NewDecoder(resp.Body).Decode(&response)
	if err != nil {
		log.WithError(err).WithFields(fields).Error("[ClearOfferClaimCodeForUser] Error decoding response into interface :", err)
		return nil, resp.StatusCode, err
	}

	log.Info("[ClearOfferClaimCodeForUser] TPROX client call succeeded, status code: ", resp.StatusCode)
	return &response.Output, resp.StatusCode, nil
}

/**********************************************/
/*** [SCES] GetEligibilityStatus ***/
/**********************************************/

// EntitlementAccount - An account entitled to an offer
type EntitlementAccount struct {
	AccountType string `json:"accountType"`
	AccountID   string `json:"accountId"`
	DisplayName string `json:"displayName"`
}

// Eligibility Rules
const (
	ClaimLimitRule    = "CLAIM_LIMIT"
	OfferWindowRule   = "OFFER_WINDOW"
	PrimeGamingRule   = "PRIME_GAMING"
	LinkedAccountRule = "LINKED_ACCOUNT"
	MarketplaceRule   = "MARKETPLACE"
)

// EligibilityResult - Result of an eligibility check for an offer and customer
type EligibilityResult struct {
	RuleName string `json:"ruleName"` // Restricted to one of: CLAIM_LIMIT, OFFER_WINDOW, PRIME_GAMING, LINKED_ACCOUNT, MARKETPLACE
	Eligible bool   `json:"eligible"`
}

// Entitlement Statuses
const (
	EntitlementCleared  = "CLEARED"
	EntitlementEntitled = "ENTITLED"
	EntitlementBlocked  = "BLOCKED"
	EntitlementRevoked  = "REVOKED"
)

// EntitlementRecord - Record of a previous entitlement for an offer and customer
type EntitlementRecord struct {
	EntitlementID     string               `json:"entitlementId"`
	EntitlementStatus string               `json:"entitlementStatus"`
	EntitledDate      string               `json:"entitledDate"`
	EntitledAccounts  []EntitlementAccount `json:"entitledAccounts"`
}

// GetEligibilityStatusTwitchRequest - Coral request model for the SCES GetEligibilityStatus operation through Twitch.
// Requires TwitchUserID and OfferID
type GetEligibilityStatusTwitchRequest struct {
	TwitchUserID string `json:"twitchUserId"`
	OfferID      string `json:"offerId"`
	OrderID      string `json:"orderId"`
	DateOverride string `json:"dateOverride"`
}

// Offer States
const (
	OfferExpired = "EXPIRED"
	OfferLive    = "LIVE"
	OfferFuture  = "FUTURE"
)

// GetEligibilityStatusResponse - Coral response model for the SCES GetEligibilityStatus operation
type GetEligibilityStatusResponse struct {
	Type                   string               `json:"__type"`
	Message                string               `json:"message"`
	ActingCustomerID       string               `json:"actingCustomerId"`
	BenefitOwnerCustomerID string               `json:"benefitOwnerCustomerId"`
	OfferID                string               `json:"offerId"`
	Title                  string               `json:"title"`
	ItemID                 string               `json:"itemId"`
	Eligible               bool                 `json:"eligible"`
	AccountsToEntitle      []EntitlementAccount `json:"accountsToEntitle"`
	OfferState             string               `json:"offerState"` // Restricted to one of: EXPIRED, LIVE, FUTURE
	EligibilityResults     []EligibilityResult  `json:"eligibilityResults"`
	PriorEntitlements      []EntitlementRecord  `json:"priorEntitlements"`
}

// GetEligibilityStatusTProxRequest - Coral request model for the TProx GetEligibilityStatus operation
type GetEligibilityStatusTProxRequest struct {
	Request   *GetEligibilityStatusTwitchRequest `json:"GetEligibilityStatusTwitchRequest"`
	ClientID  *string                            `json:"clientID"`
	RequestID *string                            `json:"requestID"`
}

// GetEligibilityStatusTProxResponse - Coral response model for the TProx GetEligibilityStatus operation
type GetEligibilityStatusTProxResponse struct {
	Output GetEligibilityStatusResponse `json:"Output"`
}

// GetEligibilityStatus - TProx Coral operation, proxies the SCES GetEligibilityStatus Coral operation.
// Retrieves the requested user's eligibility for the requested offer.
func (sc *SamusTProxClient) GetEligibilityStatus(ctx context.Context, getEligibilityStatusTwitchRequest *GetEligibilityStatusTwitchRequest) (*GetEligibilityStatusResponse, int, error) {
	reqID := uuid.NewV4().String()
	fields := log.Fields{
		"twitchUserId": getEligibilityStatusTwitchRequest.TwitchUserID,
		"offerId":      getEligibilityStatusTwitchRequest.OfferID,
		"orderId":      getEligibilityStatusTwitchRequest.OrderID,
		"dateOverride": getEligibilityStatusTwitchRequest.DateOverride,
		"requestID":    reqID,
	}

	clientID := TPROX_CLIENT_ID // can't create a pointer to a const
	requestInput := GetEligibilityStatusTProxRequest{
		Request:   getEligibilityStatusTwitchRequest,
		ClientID:  &clientID,
		RequestID: &reqID,
	}

	req, err := sc.createTProxRequest(
		"POST",
		requestInput,
		GET_ELIGIBILITY_STATUS_OPERATION_NAME,
		reqID,
	)
	if err != nil || req == nil {
		log.WithError(err).WithFields(fields).Error("[GetEligibilityStatus] createTProxRequest failed: ", err)
		return nil, http.StatusInternalServerError, err
	}

	resp, err := sc.Do(ctx, req, twitchhttp.ReqOpts{})
	defer safeClose(resp, &err)

	if err != nil {
		log.WithError(err).WithFields(fields).Error("[GetEligibilityStatus] requestCall failed: ", err)
		return nil, http.StatusInternalServerError, err
	}

	// Error if call was okay but it was not a 200
	// (ex: TProx correctly forwards the call to SCES, but SCES returns a 500)
	if resp.StatusCode != http.StatusOK {
		var coralError internal_models.CoralError
		err = json.NewDecoder(resp.Body).Decode(&coralError)
		if err != nil {
			log.WithError(err).WithFields(fields).Error("[GetEligibilityStatus] Error decoding non-200 response into interface: ", err)
			return nil, resp.StatusCode, err
		}
		log.WithError(err).WithFields(fields).Error("[GetEligibilityStatus] response was not a 200: ", err)
		return nil, resp.StatusCode, errors.New(coralError.ErrorCode)
	}

	var response GetEligibilityStatusTProxResponse
	err = json.NewDecoder(resp.Body).Decode(&response)
	if err != nil {
		log.WithError(err).WithFields(fields).Error("[GetEligibilityStatus] Error decoding response into interface: ", err)
		return nil, resp.StatusCode, err
	}

	log.Info("[GetEligibilityStatus] TPROX client call succeeded, status code: ", resp.StatusCode)
	return &response.Output, resp.StatusCode, nil
}

/**********************************************/
/*** [SCES] BatchGetEligibilityStatus ***/
/**********************************************/

// BatchGetEligibilityStatusTwitchRequest - Coral request model for the SCES BatchGetEligibilityStatus operation through Twitch.
// Requires TwitchUserID and OfferIDs
type BatchGetEligibilityStatusTwitchRequest struct {
	TwitchUserID string   `json:"twitchUserId"`
	OfferIDs     []string `json:"offerIds"`
	DateOverride string   `json:"dateOverride"`
}

// BatchGetEligibilityStatusResponse - Coral response model for the SCES BatchGetEligibilityStatus operation
type BatchGetEligibilityStatusResponse struct {
	Type    string                         `json:"__type"`
	Message string                         `json:"message"`
	Results []GetEligibilityStatusResponse `json:"results"`
}

// BatchGetEligibilityStatusTProxRequest - Coral request model for the TProx BatchGetEligibilityStatus operation
type BatchGetEligibilityStatusTProxRequest struct {
	Request   *BatchGetEligibilityStatusTwitchRequest `json:"BatchGetEligibilityStatusTwitchRequest"`
	ClientID  *string                                 `json:"clientID"`
	RequestID *string                                 `json:"requestID"`
}

// BatchGetEligibilityStatusTProxResponse - Coral response model for the TProx BatchGetEligibilityStatus operation
type BatchGetEligibilityStatusTProxResponse struct {
	Output BatchGetEligibilityStatusResponse `json:"Output"`
}

// BatchGetEligibilityStatus - TProx Coral operation, proxies the SCES BatchGetEligibilityStatus Coral operation.
// Retrieves the requested user's eligibility for the requested offers.
func (sc *SamusTProxClient) BatchGetEligibilityStatus(ctx context.Context, batchGetEligibilityStatusTwitchRequest *BatchGetEligibilityStatusTwitchRequest) (*BatchGetEligibilityStatusResponse, int, error) {
	reqID := uuid.NewV4().String()
	fields := log.Fields{
		"twitchUserId": batchGetEligibilityStatusTwitchRequest.TwitchUserID,
		"offerIDs":     batchGetEligibilityStatusTwitchRequest.OfferIDs,
		"dateOverride": batchGetEligibilityStatusTwitchRequest.DateOverride,
		"requestID":    reqID,
	}

	clientID := TPROX_CLIENT_ID // can't create a pointer to a const
	requestInput := BatchGetEligibilityStatusTProxRequest{
		Request:   batchGetEligibilityStatusTwitchRequest,
		ClientID:  &clientID,
		RequestID: &reqID,
	}

	req, err := sc.createTProxRequest(
		"POST",
		requestInput,
		BATCH_GET_ELIGIBILITY_STATUS_OPERATION_NAME,
		reqID,
	)
	if err != nil || req == nil {
		log.WithError(err).WithFields(fields).Error("[BatchGetEligibilityStatus] createTProxRequest failed: ", err)
		return nil, http.StatusInternalServerError, err
	}

	resp, err := sc.Do(ctx, req, twitchhttp.ReqOpts{})
	defer safeClose(resp, &err)

	if err != nil {
		log.WithError(err).WithFields(fields).Error("[BatchGetEligibilityStatus] requestCall failed: ", err)
		return nil, http.StatusInternalServerError, err
	}

	// Error if call was okay but it was not a 200
	// (ex: TProx correctly forwards the call to SCES, but SCES returns a 500)
	if resp.StatusCode != http.StatusOK {
		var coralError internal_models.CoralError
		err = json.NewDecoder(resp.Body).Decode(&coralError)
		if err != nil {
			log.WithError(err).WithFields(fields).Error("[BatchGetEligibilityStatus] Error decoding non-200 response into interface: ", err)
			return nil, resp.StatusCode, err
		}
		log.WithError(err).WithFields(fields).Error("[BatchGetEligibilityStatus] response was not a 200: ", err)
		return nil, resp.StatusCode, errors.New(coralError.ErrorCode)
	}

	var response BatchGetEligibilityStatusTProxResponse
	err = json.NewDecoder(resp.Body).Decode(&response)
	if err != nil {
		log.WithError(err).WithFields(fields).Error("[BatchGetEligibilityStatus] Error decoding response into interface: ", err)
		return nil, resp.StatusCode, err
	}

	log.Info("[BatchGetEligibilityStatus] TPROX client call succeeded, status code: ", resp.StatusCode)
	return &response.Output, resp.StatusCode, nil
}

/**********************************************/
/*** [SORDS] PlaceOrder ***/
/**********************************************/

// PlaceOrderTwitchRequest - Coral request model for the SORDS PlaceOrder operation through Twitch.
// Requires TwitchUserID, OfferID, IdempotenceKey, AttributionChannel
type PlaceOrderTwitchRequest struct {
	SelectedTwitchUserId string `json:"selectedTwitchUserId"`
	OfferID              string `json:"offerId"`
	IdempotenceKey       string `json:"idempotenceKey"`
	AttributionChannel   string `json:"attributionChannel"`
	DateOverride         string `json:"dateOverride"`
}

// PlaceOrderResponse - Coral response model for the SORDS PlaceOrder operation
type PlaceOrderResponse struct {
	Type    string `json:"__type"`
	Message string `json:"message"`
	OrderId string `json:"orderId"`
}

// PlaceOrderTProxRequest - Coral request model for the TProx PlaceOrder operation
type PlaceOrderTProxRequest struct {
	Request   *PlaceOrderTwitchRequest `json:"PlaceOrderTwitchRequest"`
	ClientID  *string                  `json:"clientID"`
	RequestID *string                  `json:"requestID"`
}

// PlaceOrderStatusTProxResponse - Coral response model for the TProx PlaceOrder operation
type PlaceOrderTProxResponse struct {
	Output PlaceOrderResponse `json:"Output"`
}

// PlaceOrder - TProx Coral operation, proxies the SORDS PlaceOrder Coral operation.
// Retrieves the requested user's eligibility for the requested offers.
func (sc *SamusTProxClient) PlaceOrder(ctx context.Context, placeOrderTwitchRequest *PlaceOrderTwitchRequest) (*PlaceOrderResponse, int, error) {
	reqID := uuid.NewV4().String()
	fields := log.Fields{
		"selectedTwitchUserId": placeOrderTwitchRequest.SelectedTwitchUserId,
		"offerId":              placeOrderTwitchRequest.OfferID,
		"idempotenceKey":       placeOrderTwitchRequest.IdempotenceKey,
		"attributionChannel":   placeOrderTwitchRequest.AttributionChannel,
		"requestID":            reqID,
	}

	clientID := TPROX_CLIENT_ID // can't create a pointer to a const
	requestInput := PlaceOrderTProxRequest{
		Request:   placeOrderTwitchRequest,
		ClientID:  &clientID,
		RequestID: &reqID,
	}

	req, err := sc.createTProxRequest(
		"POST",
		requestInput,
		PLACE_ORDER_OPERATION_NAME,
		reqID,
	)
	if err != nil || req == nil {
		log.WithError(err).WithFields(fields).Error("[PlaceOrder] createTProxRequest failed: ", err)
		return nil, http.StatusInternalServerError, err
	}

	resp, err := sc.Do(ctx, req, twitchhttp.ReqOpts{})
	defer safeClose(resp, &err)

	if err != nil {
		log.WithError(err).WithFields(fields).Error("[PlaceOrder] requestCall failed: ", err)
		return nil, http.StatusInternalServerError, err
	}

	// Error if call was okay but it was not a 200
	// (ex: TProx correctly forwards the call to SORDS, but SORDS returns a 500)
	if resp.StatusCode != http.StatusOK {
		var coralError internal_models.CoralError
		err = json.NewDecoder(resp.Body).Decode(&coralError)
		if err != nil {
			log.WithError(err).WithFields(fields).Error("[PlaceOrder] Error decoding non-200 response into interface: ", err)
			return nil, resp.StatusCode, err
		}
		log.WithError(err).WithFields(fields).Error("[PlaceOrder] response was not a 200: ", err)
		return nil, resp.StatusCode, errors.New(coralError.ErrorCode)
	}

	var response PlaceOrderTProxResponse
	err = json.NewDecoder(resp.Body).Decode(&response)
	if err != nil {
		log.WithError(err).WithFields(fields).Error("[PlaceOrder] Error decoding response into interface: ", err)
		return nil, resp.StatusCode, err
	}

	log.Info("[PlaceOrder] TPROX client call succeeded, status code: ", resp.StatusCode)
	return &response.Output, resp.StatusCode, nil
}

/**********************************************/
/*** [SORDS] GetOrdersByCustomer ***/
/**********************************************/

// GetOrdersByCustomerTwitchRequest - Coral request model for the SORDS GetOrdersByCustomer operation through Twitch.
// Requires UserID
type GetOrdersByCustomerTwitchRequest struct {
	TwitchUserID string `json:"twitchUserId"`
	NextToken    string `json:"nextToken"`
	PageSize     int    `json:"pageSize"`
	OfferID      string `json:"offerId"`
	OrderID      string `json:"orderId"`
}

// GetOrdersByCustomerResponse - Coral response model for the SORDS GetOrdersByCustomer operation
type GetOrdersByCustomerResponse struct {
	Type           string                        `json:"__type"`
	Message        string                        `json:"message"`
	OrderDocuments []gatewayClient.OrderDocument `json:"orderDocuments"`
}

// GetOrdersByCustomerTProxRequest - Coral request model for the TProx GetOrdersByCustomer operation
type GetOrdersByCustomerTProxRequest struct {
	Request   *GetOrdersByCustomerTwitchRequest `json:"GetOrdersByCustomerTwitchRequest"`
	ClientID  *string                           `json:"clientID"`
	RequestID *string                           `json:"requestID"`
}

// GetOrdersByCustomerStatusTProxResponse - Coral response model for the TProx GetOrdersByCustomer operation
type GetOrdersByCustomerTProxResponse struct {
	Output GetOrdersByCustomerResponse `json:"Output"`
}

// GetOrdersByCustomer - TProx Coral operation, proxies the SORDS GetOrdersByCustomer Coral operation.
// Retrieves the requested user's eligibility for the requested offers.
func (sc *SamusTProxClient) GetOrdersByCustomer(ctx context.Context, getOrdersByCustomerTwitchRequest *GetOrdersByCustomerTwitchRequest) (*GetOrdersByCustomerResponse, int, error) {
	reqID := uuid.NewV4().String()
	fields := log.Fields{
		"twitchUserId": getOrdersByCustomerTwitchRequest.TwitchUserID,
		"nextToken":    getOrdersByCustomerTwitchRequest.NextToken,
		"pageSize":     getOrdersByCustomerTwitchRequest.PageSize,
		"offerId":      getOrdersByCustomerTwitchRequest.OfferID,
		"orderId":      getOrdersByCustomerTwitchRequest.OrderID,
		"requestID":    reqID,
	}

	clientID := TPROX_CLIENT_ID // can't create a pointer to a const
	requestInput := GetOrdersByCustomerTProxRequest{
		Request:   getOrdersByCustomerTwitchRequest,
		ClientID:  &clientID,
		RequestID: &reqID,
	}

	req, err := sc.createTProxRequest(
		"POST",
		requestInput,
		GET_ORDERS_BY_CUSTOMER_OPERATION_NAME,
		reqID,
	)
	if err != nil || req == nil {
		log.WithError(err).WithFields(fields).Error("[GetOrdersByCustomer] createTProxRequest failed: ", err)
		return nil, http.StatusInternalServerError, err
	}

	resp, err := sc.Do(ctx, req, twitchhttp.ReqOpts{})
	defer safeClose(resp, &err)

	if err != nil {
		log.WithError(err).WithFields(fields).Error("[GetOrdersByCustomer] requestCall failed: ", err)
		return nil, http.StatusInternalServerError, err
	}

	// Error if call was okay but it was not a 200
	// (ex: TProx correctly forwards the call to SORDS, but SORDS returns a 500)
	if resp.StatusCode != http.StatusOK {
		var coralError internal_models.CoralError
		err = json.NewDecoder(resp.Body).Decode(&coralError)
		if err != nil {
			log.WithError(err).WithFields(fields).Error("[GetOrdersByCustomer] Error decoding non-200 response into interface: ", err)
			return nil, resp.StatusCode, err
		}
		log.WithError(err).WithFields(fields).Error("[GetOrdersByCustomer] response was not a 200: ", err)
		return nil, resp.StatusCode, errors.New(coralError.ErrorCode)
	}

	var response GetOrdersByCustomerTProxResponse
	err = json.NewDecoder(resp.Body).Decode(&response)
	if err != nil {
		log.WithError(err).WithFields(fields).Error("[GetOrdersByCustomer] Error decoding response into interface: ", err)
		return nil, resp.StatusCode, err
	}

	log.Info("[GetOrdersByCustomer] TPROX client call succeeded, status code: ", resp.StatusCode)
	log.Info("GetOrdersByCustomer response: ", &response.Output)
	return &response.Output, resp.StatusCode, nil
}

/**********************************************/
/*** [SIS] ListInventory ***/
/**********************************************/

// ListInventoryTwitchRequest - Coral request model for the SIS ListInventory operation through Twitch.
// Requires UserID
type ListInventoryTwitchRequest struct {
	TwitchUserID             string   `json:"twitchUserId"`
	AmazonCustomerID         string   `json:"amazonCustomerId"`
	EntitlementAccountType   string   `json:"entitlementAccountType"`
	NextToken                string   `json:"nextToken"`
	MaxResults               int      `json:"maxResults"`
	ItemIds                  []string `json:"itemIds"`
	EntitlementStatusFilters []string `json:"entitlementStatusFilters"`
}

// ListInventoryResponse - Coral response model for the SIS ListInventory operation
type ListInventoryResponse struct {
	Type      string                  `json:"__type"`
	Message   string                  `json:"message"`
	Inventory gatewayClient.Inventory `json:"inventory"`
}

// ListInventoryTProxRequest - Coral request model for the TProx ListInventory operation
type ListInventoryTProxRequest struct {
	Request   *ListInventoryTwitchRequest `json:"ListInventoryTwitchRequest"`
	ClientID  *string                     `json:"clientID"`
	RequestID *string                     `json:"requestID"`
}

// ListInventoryTProxResponse - Coral response model for the TProx ListInventory operation
type ListInventoryTProxResponse struct {
	Output ListInventoryResponse `json:"Output"`
}

// ListInventory - TProx Coral operation, proxies the SIS ListInventory Coral operation.
// Retrieves the requested user's inventory (code offers included)
func (sc *SamusTProxClient) ListInventory(ctx context.Context, listInventoryTwitchRequest *ListInventoryTwitchRequest) (*ListInventoryResponse, int, error) {
	reqID := uuid.NewV4().String()
	fields := log.Fields{
		"twitchUserId":             listInventoryTwitchRequest.TwitchUserID,
		"amazonCustomerID":         listInventoryTwitchRequest.AmazonCustomerID,
		"entitlementAccountType":   listInventoryTwitchRequest.EntitlementAccountType,
		"nextToken":                listInventoryTwitchRequest.NextToken,
		"maxResults":               listInventoryTwitchRequest.MaxResults,
		"itemIds":                  listInventoryTwitchRequest.ItemIds,
		"entitlementStatusFilters": listInventoryTwitchRequest.EntitlementStatusFilters,
		"requestID":                reqID,
	}

	clientID := TPROX_CLIENT_ID // can't create a pointer to a const
	requestInput := ListInventoryTProxRequest{
		Request:   listInventoryTwitchRequest,
		ClientID:  &clientID,
		RequestID: &reqID,
	}

	req, err := sc.createTProxRequest(
		"POST",
		requestInput,
		LIST_INVENTORY_OPERATION_NAME,
		reqID,
	)
	if err != nil || req == nil {
		log.WithError(err).WithFields(fields).Error("[ListInventory] createTProxRequest failed: ", err)
		return nil, http.StatusInternalServerError, err
	}

	resp, err := sc.Do(ctx, req, twitchhttp.ReqOpts{})
	defer safeClose(resp, &err)

	if err != nil {
		log.WithError(err).WithFields(fields).Error("[ListInventory] requestCall failed: ", err)
		return nil, http.StatusInternalServerError, err
	}

	// Error if call was okay but it was not a 200
	// (ex: TProx correctly forwards the call to SORDS, but SORDS returns a 500)
	if resp.StatusCode != http.StatusOK {
		var coralError internal_models.CoralError
		err = json.NewDecoder(resp.Body).Decode(&coralError)
		if err != nil {
			log.WithError(err).WithFields(fields).Error("[ListInventory] Error decoding non-200 response into interface: ", err)
			return nil, resp.StatusCode, err
		}
		log.WithError(err).WithFields(fields).Error("[ListInventory] response was not a 200: ", err)
		return nil, resp.StatusCode, errors.New(coralError.ErrorCode)
	}

	var response ListInventoryTProxResponse
	err = json.NewDecoder(resp.Body).Decode(&response)
	if err != nil {
		log.WithError(err).WithFields(fields).Error("[ListInventory] Error decoding response into interface: ", err)
		return nil, resp.StatusCode, err
	}

	log.Info("[ListInventory] TPROX client call succeeded, status code: ", resp.StatusCode)
	log.Info("ListInventory response: ", &response.Output)
	return &response.Output, resp.StatusCode, nil
}
