package clients

import (
	"net/http"
	"testing"

	"github.com/pkg/errors"

	gatewayClient "code.justin.tv/samus/gateway/client"
	"code.justin.tv/samus/gateway/clients/mocks"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"

	"bytes"
	"encoding/json"
	"io/ioutil"

	"github.com/stretchr/testify/mock"
	"golang.org/x/net/context"
)

func TestSamusTProxClient(t *testing.T) {

	Convey("GetCurrentOffersForUser", t, func() {
		httpMock := new(mocks.HttpMock)
		sigV4SignerMock := new(mocks.SigV4SignerMock)
		samusTProxClient := &SamusTProxClient{sigV4SignerMock, httpMock}
		So(samusTProxClient, ShouldNotBeNil)

		// Create fake request
		request := &GetCurrentOffersForUserInCountryRequest{
			TUID:         "testTUID",
			CountryCode:  "testCountryCode",
			Locale:       "testLocale",
			DateOverride: "testDateOverride",
			ClientID:     "testClientID",
		}

		// Create fake http response
		testOfferTitle := "testOfferTitle"
		testOfferID := "testOfferID"
		mockedHttpResponse := TProxGetCurrentOffersForUserInCountryResponse{
			Output: GetCurrentOffersForUserInCountryResponse{
				Offers: []gatewayClient.OfferWithClaimData{
					gatewayClient.OfferWithClaimData{
						ContentCategories:       []string{},
						ExternalOfferUri:        "externalOfferUri",
						PublisherName:           "publisherName",
						ApplicableGame:          "applicableGame",
						ContentDeliveryMethod:   "claimCode",
						Assets:                  []gatewayClient.OfferAsset{},
						ContentClaimInstruction: "claimIntructions",
						Priority:                1,
						OfferDescription:        "offerDescription",
						OfferTitle:              testOfferTitle,
						OfferID:                 testOfferID,
						EndTime:                 "endTime",
						StartTime:               "startTime",
						Tags:                    gatewayClient.OfferTag{},
						OfferClaimMetadata:      gatewayClient.OfferClaimMetadata{},
						OfferClaimData:          "claimData",
					},
				},
				ClaimStatuses: map[string]ClaimStatus{
					testOfferID: {
						ClaimHint: "AVAILABLE",
					},
				},
			},
		}
		buf, err := json.Marshal(mockedHttpResponse)
		if err != nil {
			log.Debug(err)
		}
		mockedResp := new(http.Response)
		mockedResp.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
		mockedResp.StatusCode = http.StatusOK
		httpMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockedResp, nil)

		Convey("returns data when the call to TProx succeeds", func() {
			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, nil)
			resp, s, e := samusTProxClient.GetCurrentOffersForUserInCountry(context.TODO(), request)

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)
			So(len(resp.Offers), ShouldEqual, 1)
			So(resp.Offers[0].OfferID, ShouldEqual, testOfferID)
			So(resp.Offers[0].OfferTitle, ShouldEqual, testOfferTitle)
			So(resp.ClaimStatuses[testOfferID].ClaimHint, ShouldEqual, "AVAILABLE")
			So(s, ShouldEqual, http.StatusOK)
			So(e, ShouldBeNil)
		})

		Convey("errors when the call to TProx fails", func() {
			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, nil)
			mockedResp.StatusCode = http.StatusInternalServerError
			httpMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockedResp, nil)

			resp, s, e := samusTProxClient.GetCurrentOffersForUserInCountry(context.TODO(), request)

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)
			So(resp, ShouldBeNil)
			So(s, ShouldEqual, http.StatusInternalServerError)
			So(e, ShouldNotBeNil)
		})

		Convey("errors when the SigV4Signer fails", func() {
			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, errors.New("Some fool forgot to put in their AWS credentials!"))

			resp, s, e := samusTProxClient.GetCurrentOffersForUserInCountry(context.TODO(), request)

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)
			So(resp, ShouldBeNil)
			So(s, ShouldEqual, http.StatusInternalServerError)
			So(e, ShouldNotBeNil)
		})
	})

	Convey("ClearOfferClaimCodeForUser", t, func() {
		httpMock := new(mocks.HttpMock)
		sigV4SignerMock := new(mocks.SigV4SignerMock)
		samusTProxClient := &SamusTProxClient{sigV4SignerMock, httpMock}
		So(samusTProxClient, ShouldNotBeNil)

		// Create fake request
		request := &ClearOfferClaimCodeForUserRequest{
			TwitchUserID:  "testTUID",
			MarketplaceID: "testMarketplaceID",
			OfferID:       "testOfferID",
			CsAgent:       "testCsAgent",
			CsContactID:   "testCsContactID",
		}

		// Create fake http response
		mockedHttpResponse := TProxClearOfferClaimCodeForUserResponse{
			Output: ClearOfferClaimCodeForUserResponse{
				Success: true,
			},
		}
		buf, err := json.Marshal(mockedHttpResponse)
		if err != nil {
			log.Debug(err)
		}
		mockedResp := new(http.Response)
		mockedResp.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
		mockedResp.StatusCode = http.StatusOK
		httpMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockedResp, nil)

		Convey("returns data when the call to TProx succeeds", func() {
			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, nil)
			resp, s, e := samusTProxClient.ClearOfferClaimCodeForUser(context.TODO(), request)

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)
			So(resp.Success, ShouldEqual, true)
			So(s, ShouldEqual, http.StatusOK)
			So(e, ShouldBeNil)
		})

		Convey("errors when the call to TProx fails", func() {
			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, nil)
			mockedResp.StatusCode = http.StatusInternalServerError
			httpMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockedResp, nil)

			resp, s, e := samusTProxClient.ClearOfferClaimCodeForUser(context.TODO(), request)

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)
			So(resp, ShouldBeNil)
			So(s, ShouldEqual, http.StatusInternalServerError)
			So(e, ShouldNotBeNil)
		})

		Convey("errors when the SigV4Signer fails", func() {
			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, errors.New("Some fool forgot to put in their AWS credentials!"))

			resp, s, e := samusTProxClient.ClearOfferClaimCodeForUser(context.TODO(), request)

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)
			So(resp, ShouldBeNil)
			So(s, ShouldEqual, http.StatusInternalServerError)
			So(e, ShouldNotBeNil)
		})
	})
}

func TestGetEligibilityStatus(t *testing.T) {
	Convey("GetEligibilityStatus", t, func() {
		const (
			TEST_CUSTOMER_ID    = "TEST_CUSTOMER_ID"
			TEST_ACCOUNT_TYPE   = "TEST_ACCOUNT_TYPE"
			TEST_DISPLAY_NAME   = "TEST_DISPLAY_NAME"
			TEST_TWITCH_USER_ID = "TEST_TWITCH_USER_ID"

			TEST_OFFER_ID    = "TEST_OFFER_ID"
			TEST_ITEM_ID     = "TEST_ITEM_ID"
			TEST_OFFER_STATE = "TEST_OFFER_STATE"
			TEST_OFFER_TITLE = "TEST_OFFER_TITLE"
			TEST_ELIGIBLE    = true

			TEST_ELIGIBILITY_RULE        = "TEST_ELIGIBILITY_RULE"
			TEST_ELIGIBILITY_RULE_RESULT = true
		)

		var TEST_ELIGIBILITY_RESULTS = []EligibilityResult{{
			RuleName: TEST_ELIGIBILITY_RULE,
			Eligible: TEST_ELIGIBILITY_RULE_RESULT,
		}}

		var TEST_ACCOUNTS_TO_ENTITLE = []EntitlementAccount{{
			AccountType: TEST_ACCOUNT_TYPE,
			AccountID:   TEST_CUSTOMER_ID,
			DisplayName: TEST_DISPLAY_NAME,
		}}

		var TEST_PRIOR_ENTITLEMENTS = []EntitlementRecord{}

		// Set up the TProx client
		httpMock := new(mocks.HttpMock)
		sigV4SignerMock := new(mocks.SigV4SignerMock)
		samusTProxClient := &SamusTProxClient{
			tproxSigV4Signer: sigV4SignerMock,
			Client:           httpMock,
		}

		// Create fake HTTP request
		mockRequest := &GetEligibilityStatusTwitchRequest{
			TwitchUserID: TEST_TWITCH_USER_ID,
			OfferID:      TEST_OFFER_ID,
		}

		So(samusTProxClient, ShouldNotBeNil)

		Convey("returns data when the call to TProx succeeds", func() {
			// Create fake HTTP response
			mockResponse := GetEligibilityStatusTProxResponse{
				Output: GetEligibilityStatusResponse{
					ActingCustomerID:       TEST_CUSTOMER_ID,
					BenefitOwnerCustomerID: TEST_CUSTOMER_ID,
					OfferID:                TEST_OFFER_ID,
					ItemID:                 TEST_ITEM_ID,
					OfferState:             TEST_OFFER_STATE,
					Title:                  TEST_OFFER_TITLE,
					Eligible:               TEST_ELIGIBLE,
					AccountsToEntitle:      TEST_ACCOUNTS_TO_ENTITLE,
					EligibilityResults:     TEST_ELIGIBILITY_RESULTS,
					PriorEntitlements:      TEST_PRIOR_ENTITLEMENTS,
				},
			}

			buf, err := json.Marshal(mockResponse)
			if err != nil {
				log.Debug(err)
			}
			mockHttpResponse := new(http.Response)
			mockHttpResponse.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
			mockHttpResponse.StatusCode = http.StatusOK
			httpMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockHttpResponse, nil)
			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, nil)
			resp, s, e := samusTProxClient.GetEligibilityStatus(context.TODO(), mockRequest)

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)

			So(resp.ActingCustomerID, ShouldEqual, TEST_CUSTOMER_ID)
			So(resp.BenefitOwnerCustomerID, ShouldEqual, TEST_CUSTOMER_ID)
			So(resp.OfferID, ShouldEqual, TEST_OFFER_ID)
			So(resp.ItemID, ShouldEqual, TEST_ITEM_ID)
			So(resp.OfferState, ShouldEqual, TEST_OFFER_STATE)
			So(resp.Title, ShouldEqual, TEST_OFFER_TITLE)
			So(resp.Eligible, ShouldEqual, TEST_ELIGIBLE)

			So(resp.AccountsToEntitle, ShouldNotBeNil)
			So(resp.AccountsToEntitle[0].AccountType, ShouldEqual, TEST_ACCOUNT_TYPE)
			So(resp.AccountsToEntitle[0].AccountID, ShouldEqual, TEST_CUSTOMER_ID)
			So(resp.AccountsToEntitle[0].DisplayName, ShouldEqual, TEST_DISPLAY_NAME)

			So(resp.EligibilityResults, ShouldNotBeNil)
			So(resp.EligibilityResults[0].RuleName, ShouldEqual, TEST_ELIGIBILITY_RULE)
			So(resp.EligibilityResults[0].Eligible, ShouldEqual, TEST_ELIGIBILITY_RULE_RESULT)

			So(resp.PriorEntitlements, ShouldNotBeNil)
			So(len(resp.PriorEntitlements), ShouldEqual, len(TEST_PRIOR_ENTITLEMENTS))

			So(s, ShouldEqual, http.StatusOK)
			So(e, ShouldBeNil)
		})

		Convey("returns OK request when TProx returns an empty response", func() {
			mockResponse := GetEligibilityStatusTProxResponse{
				Output: GetEligibilityStatusResponse{},
			}
			buf, err := json.Marshal(mockResponse)
			if err != nil {
				log.Debug(err)
			}
			mockHttpResponse := new(http.Response)
			mockHttpResponse.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
			mockHttpResponse.StatusCode = http.StatusOK

			httpMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockHttpResponse, nil)
			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, nil)
			resp, s, e := samusTProxClient.GetEligibilityStatus(context.TODO(), mockRequest)

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)

			So(resp, ShouldNotBeNil)
			So(s, ShouldEqual, http.StatusOK)
			So(e, ShouldBeNil)
		})

		Convey("errors when the call to TProx fails", func() {
			// Create fake HTTP response
			mockResponse := GetEligibilityStatusTProxResponse{}

			buf, err := json.Marshal(mockResponse)
			if err != nil {
				log.Debug(err)
			}
			mockHttpResponse := new(http.Response)
			mockHttpResponse.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
			mockHttpResponse.StatusCode = http.StatusInternalServerError
			httpMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockHttpResponse, nil)
			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, nil)

			resp, s, e := samusTProxClient.GetEligibilityStatus(context.TODO(), mockRequest)

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)
			So(resp, ShouldBeNil)
			So(s, ShouldEqual, http.StatusInternalServerError)
			So(e, ShouldNotBeNil)
		})

		Convey("errors when the SigV4Signer fails", func() {
			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, errors.New("Some fool forgot to put in their AWS credentials!"))

			resp, s, e := samusTProxClient.GetEligibilityStatus(context.TODO(), mockRequest)

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)
			So(resp, ShouldBeNil)
			So(s, ShouldEqual, http.StatusInternalServerError)
			So(e, ShouldNotBeNil)
		})
	})
}

func TestBatchGetEligibilityStatus(t *testing.T) {
	Convey("BatchGetEligibilityStatus", t, func() {
		const (
			TEST_CUSTOMER_ID    = "TEST_CUSTOMER_ID"
			TEST_ACCOUNT_TYPE   = "TEST_ACCOUNT_TYPE"
			TEST_DISPLAY_NAME   = "TEST_DISPLAY_NAME"
			TEST_TWITCH_USER_ID = "TEST_TWITCH_USER_ID"

			TEST_OFFER_ID    = "TEST_OFFER_ID"
			TEST_ITEM_ID     = "TEST_ITEM_ID"
			TEST_OFFER_STATE = "TEST_OFFER_STATE"
			TEST_OFFER_TITLE = "TEST_OFFER_TITLE"
			TEST_ELIGIBLE    = true

			TEST_ELIGIBILITY_RULE        = "TEST_ELIGIBILITY_RULE"
			TEST_ELIGIBILITY_RULE_RESULT = true
		)

		var TEST_ELIGIBILITY_RESULTS = []EligibilityResult{{
			RuleName: TEST_ELIGIBILITY_RULE,
			Eligible: TEST_ELIGIBILITY_RULE_RESULT,
		}}

		var TEST_ACCOUNTS_TO_ENTITLE = []EntitlementAccount{{
			AccountType: TEST_ACCOUNT_TYPE,
			AccountID:   TEST_CUSTOMER_ID,
			DisplayName: TEST_DISPLAY_NAME,
		}}

		var TEST_PRIOR_ENTITLEMENTS = []EntitlementRecord{}

		// Set up the TProx client
		httpMock := new(mocks.HttpMock)
		sigV4SignerMock := new(mocks.SigV4SignerMock)
		samusTProxClient := &SamusTProxClient{
			tproxSigV4Signer: sigV4SignerMock,
			Client:           httpMock,
		}

		// Create fake HTTP request
		mockRequest := &BatchGetEligibilityStatusTwitchRequest{
			TwitchUserID: TEST_TWITCH_USER_ID,
			OfferIDs:     []string{TEST_OFFER_ID},
		}

		So(samusTProxClient, ShouldNotBeNil)

		Convey("returns data when the call to TProx succeeds", func() {
			// Create fake HTTP response
			mockResponse := BatchGetEligibilityStatusTProxResponse{
				Output: BatchGetEligibilityStatusResponse{
					Results: []GetEligibilityStatusResponse{GetEligibilityStatusResponse{
						ActingCustomerID:       TEST_CUSTOMER_ID,
						BenefitOwnerCustomerID: TEST_CUSTOMER_ID,
						OfferID:                TEST_OFFER_ID,
						ItemID:                 TEST_ITEM_ID,
						OfferState:             TEST_OFFER_STATE,
						Title:                  TEST_OFFER_TITLE,
						Eligible:               TEST_ELIGIBLE,
						AccountsToEntitle:      TEST_ACCOUNTS_TO_ENTITLE,
						EligibilityResults:     TEST_ELIGIBILITY_RESULTS,
						PriorEntitlements:      TEST_PRIOR_ENTITLEMENTS,
					}}},
			}

			buf, err := json.Marshal(mockResponse)
			if err != nil {
				log.Debug(err)
			}
			mockHttpResponse := new(http.Response)
			mockHttpResponse.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
			mockHttpResponse.StatusCode = http.StatusOK
			httpMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockHttpResponse, nil)
			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, nil)
			resp, s, e := samusTProxClient.BatchGetEligibilityStatus(context.TODO(), mockRequest)
			results := resp.Results

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)

			So(results[0].ActingCustomerID, ShouldEqual, TEST_CUSTOMER_ID)
			So(results[0].BenefitOwnerCustomerID, ShouldEqual, TEST_CUSTOMER_ID)
			So(results[0].OfferID, ShouldEqual, TEST_OFFER_ID)
			So(results[0].ItemID, ShouldEqual, TEST_ITEM_ID)
			So(results[0].OfferState, ShouldEqual, TEST_OFFER_STATE)
			So(results[0].Title, ShouldEqual, TEST_OFFER_TITLE)
			So(results[0].Eligible, ShouldEqual, TEST_ELIGIBLE)

			So(results[0].AccountsToEntitle, ShouldNotBeNil)
			So(results[0].AccountsToEntitle[0].AccountType, ShouldEqual, TEST_ACCOUNT_TYPE)
			So(results[0].AccountsToEntitle[0].AccountID, ShouldEqual, TEST_CUSTOMER_ID)
			So(results[0].AccountsToEntitle[0].DisplayName, ShouldEqual, TEST_DISPLAY_NAME)

			So(results[0].EligibilityResults, ShouldNotBeNil)
			So(results[0].EligibilityResults[0].RuleName, ShouldEqual, TEST_ELIGIBILITY_RULE)
			So(results[0].EligibilityResults[0].Eligible, ShouldEqual, TEST_ELIGIBILITY_RULE_RESULT)

			So(results[0].PriorEntitlements, ShouldNotBeNil)
			So(len(results[0].PriorEntitlements), ShouldEqual, len(TEST_PRIOR_ENTITLEMENTS))

			So(s, ShouldEqual, http.StatusOK)
			So(e, ShouldBeNil)
		})

		Convey("returns OK request when TProx returns an empty response", func() {
			mockResponse := BatchGetEligibilityStatusTProxResponse{
				Output: BatchGetEligibilityStatusResponse{},
			}

			buf, err := json.Marshal(mockResponse)
			if err != nil {
				log.Debug(err)
			}
			mockHttpResponse := new(http.Response)
			mockHttpResponse.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
			mockHttpResponse.StatusCode = http.StatusOK
			httpMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockHttpResponse, nil)
			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, nil)
			resp, s, e := samusTProxClient.BatchGetEligibilityStatus(context.TODO(), mockRequest)

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)

			So(resp, ShouldNotBeNil)
			So(s, ShouldEqual, http.StatusOK)
			So(e, ShouldBeNil)
		})

		Convey("errors when the call to TProx fails", func() {
			// Create fake HTTP response
			mockResponse := BatchGetEligibilityStatusTProxResponse{}

			buf, err := json.Marshal(mockResponse)
			if err != nil {
				log.Debug(err)
			}
			mockHttpResponse := new(http.Response)
			mockHttpResponse.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
			mockHttpResponse.StatusCode = http.StatusInternalServerError
			httpMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockHttpResponse, nil)

			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, nil)
			resp, s, e := samusTProxClient.BatchGetEligibilityStatus(context.TODO(), mockRequest)

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)
			So(resp, ShouldBeNil)
			So(s, ShouldEqual, http.StatusInternalServerError)
			So(e, ShouldNotBeNil)
		})

		Convey("errors when the SigV4Signer fails", func() {
			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, errors.New("Some fool forgot to put in their AWS credentials!"))

			resp, s, e := samusTProxClient.BatchGetEligibilityStatus(context.TODO(), mockRequest)

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)
			So(resp, ShouldBeNil)
			So(s, ShouldEqual, http.StatusInternalServerError)
			So(e, ShouldNotBeNil)
		})
	})
}

func TestPlaceOrder(t *testing.T) {
	Convey("PlaceOrder", t, func() {
		const (
			TEST_CUSTOMER_ID         = "TEST_CUSTOMER_ID"
			TEST_TWITCH_USER_ID      = "TEST_TWITCH_USER_ID"
			TEST_IDEMPOTENCE_KEY     = "TEST_IDEMPOTENCE_KEY"
			TEST_ATTRIBUTION_CHANNEL = "TEST_ATTRIBUTION_CHANNEL"
			TEST_ORDER_ID            = "TEST_ORDER_ID"
			TEST_OFFER_ID            = "TEST_OFFER_ID"
		)

		// Set up the TProx client
		httpMock := new(mocks.HttpMock)
		sigV4SignerMock := new(mocks.SigV4SignerMock)
		samusTProxClient := &SamusTProxClient{
			tproxSigV4Signer: sigV4SignerMock,
			Client:           httpMock,
		}

		// Create fake HTTP request
		mockRequest := &PlaceOrderTwitchRequest{
			SelectedTwitchUserId: TEST_TWITCH_USER_ID,
			OfferID:              TEST_OFFER_ID,
			IdempotenceKey:       TEST_IDEMPOTENCE_KEY,
			AttributionChannel:   TEST_ATTRIBUTION_CHANNEL,
		}

		So(samusTProxClient, ShouldNotBeNil)

		Convey("returns data when the call to TProx succeeds", func() {
			// Create fake HTTP response
			mockResponse := PlaceOrderTProxResponse{
				Output: PlaceOrderResponse{
					OrderId: TEST_ORDER_ID,
				},
			}
			buf, err := json.Marshal(mockResponse)
			if err != nil {
				log.Debug(err)
			}
			mockHttpResponse := new(http.Response)
			mockHttpResponse.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
			mockHttpResponse.StatusCode = http.StatusOK
			httpMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockHttpResponse, nil)
			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, nil)
			resp, s, e := samusTProxClient.PlaceOrder(context.TODO(), mockRequest)
			results := resp

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)

			So(results.OrderId, ShouldEqual, TEST_ORDER_ID)

			So(s, ShouldEqual, http.StatusOK)
			So(e, ShouldBeNil)
		})

		Convey("returns OK request when TProx returns an empty response", func() {
			mockResponse := PlaceOrderTProxResponse{
				Output: PlaceOrderResponse{},
			}
			buf, err := json.Marshal(mockResponse)
			if err != nil {
				log.Debug(err)
			}
			mockHttpResponse := new(http.Response)
			mockHttpResponse.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
			mockHttpResponse.StatusCode = http.StatusOK
			httpMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockHttpResponse, nil)
			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, nil)
			resp, s, e := samusTProxClient.PlaceOrder(context.TODO(), mockRequest)

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)

			So(resp, ShouldNotBeNil)
			So(s, ShouldEqual, http.StatusOK)
			So(e, ShouldBeNil)
		})

		Convey("errors when the call to TProx fails", func() {
			// Create fake HTTP response
			mockResponse := PlaceOrderTProxResponse{}

			buf, err := json.Marshal(mockResponse)
			if err != nil {
				log.Debug(err)
			}
			mockHttpResponse := new(http.Response)
			mockHttpResponse.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
			mockHttpResponse.StatusCode = http.StatusInternalServerError
			httpMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockHttpResponse, nil)

			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, nil)
			resp, s, e := samusTProxClient.PlaceOrder(context.TODO(), mockRequest)

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)
			So(resp, ShouldBeNil)
			So(s, ShouldEqual, http.StatusInternalServerError)
			So(e, ShouldNotBeNil)
		})

		Convey("errors when the SigV4Signer fails", func() {
			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, errors.New("Some fool forgot to put in their AWS credentials!"))

			resp, s, e := samusTProxClient.PlaceOrder(context.TODO(), mockRequest)

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)
			So(resp, ShouldBeNil)
			So(s, ShouldEqual, http.StatusInternalServerError)
			So(e, ShouldNotBeNil)
		})
	})
}

func TestGetOrdersByCustomer(t *testing.T) {
	Convey("GetOrdersByCustomer", t, func() {
		const (
			TEST_CUSTOMER_ID    = "TEST_CUSTOMER_ID"
			TEST_TWITCH_USER_ID = "TEST_TWITCH_USER_ID"
			TEST_NEXT_TOKEN     = "TEST_NEXT_TOKEN"
			TEST_PAGE_SIZE      = 100
			TEST_ORDER_ID       = "TEST_ORDER_ID"
			TEST_OFFER_ID       = "TEST_OFFER_ID"
		)

		var TEST_ORDER_DOCUMENTS = []gatewayClient.OrderDocument{{}}

		// Set up the TProx client
		httpMock := new(mocks.HttpMock)
		sigV4SignerMock := new(mocks.SigV4SignerMock)
		samusTProxClient := &SamusTProxClient{
			tproxSigV4Signer: sigV4SignerMock,
			Client:           httpMock,
		}

		// Create fake HTTP request
		mockRequest := &GetOrdersByCustomerTwitchRequest{
			TwitchUserID: TEST_TWITCH_USER_ID,
			NextToken:    TEST_NEXT_TOKEN,
			PageSize:     TEST_PAGE_SIZE,
			OfferID:      TEST_OFFER_ID,
			OrderID:      TEST_ORDER_ID,
		}

		So(samusTProxClient, ShouldNotBeNil)

		Convey("returns data when the call to TProx succeeds", func() {
			// Create fake HTTP response
			mockResponse := GetOrdersByCustomerTProxResponse{
				Output: GetOrdersByCustomerResponse{
					OrderDocuments: TEST_ORDER_DOCUMENTS,
				},
			}
			buf, err := json.Marshal(mockResponse)
			if err != nil {
				log.Debug(err)
			}
			mockHttpResponse := new(http.Response)
			mockHttpResponse.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
			mockHttpResponse.StatusCode = http.StatusOK
			httpMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockHttpResponse, nil)
			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, nil)
			resp, s, e := samusTProxClient.GetOrdersByCustomer(context.TODO(), mockRequest)
			results := resp

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)

			So(len(results.OrderDocuments), ShouldEqual, len(TEST_ORDER_DOCUMENTS))

			So(s, ShouldEqual, http.StatusOK)
			So(e, ShouldBeNil)
		})

		Convey("returns OK request when TProx returns an empty response", func() {
			mockResponse := GetOrdersByCustomerTProxResponse{
				Output: GetOrdersByCustomerResponse{},
			}
			buf, err := json.Marshal(mockResponse)
			if err != nil {
				log.Debug(err)
			}
			mockHttpResponse := new(http.Response)
			mockHttpResponse.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
			mockHttpResponse.StatusCode = http.StatusOK
			httpMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockHttpResponse, nil)
			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, nil)
			resp, s, e := samusTProxClient.GetOrdersByCustomer(context.TODO(), mockRequest)

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)

			So(resp, ShouldNotBeNil)
			So(s, ShouldEqual, http.StatusOK)
			So(e, ShouldBeNil)
		})

		Convey("errors when the call to TProx fails", func() {
			// Create fake HTTP response
			mockResponse := GetOrdersByCustomerTProxResponse{}

			buf, err := json.Marshal(mockResponse)
			if err != nil {
				log.Debug(err)
			}
			mockHttpResponse := new(http.Response)
			mockHttpResponse.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
			mockHttpResponse.StatusCode = http.StatusInternalServerError
			httpMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockHttpResponse, nil)

			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, nil)
			resp, s, e := samusTProxClient.GetOrdersByCustomer(context.TODO(), mockRequest)

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)
			So(resp, ShouldBeNil)
			So(s, ShouldEqual, http.StatusInternalServerError)
			So(e, ShouldNotBeNil)
		})

		Convey("errors when the SigV4Signer fails", func() {
			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, errors.New("Some fool forgot to put in their AWS credentials!"))

			resp, s, e := samusTProxClient.GetOrdersByCustomer(context.TODO(), mockRequest)

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)
			So(resp, ShouldBeNil)
			So(s, ShouldEqual, http.StatusInternalServerError)
			So(e, ShouldNotBeNil)
		})
	})
}

func ListInventory(t *testing.T) {
	Convey("ListInventory", t, func() {
		const (
			TEST_CUSTOMER_ID              = "TEST_CUSTOMER_ID"
			TEST_TWITCH_USER_ID           = "TEST_TWITCH_USER_ID"
			TEST_AMAZON_CUSTOMER_ID       = "TEST_AMAZON_CUSTOMER_ID"
			TEST_ENTITLEMENT_ACCOUNT_TYPE = "TEST_ENTITLEMENT_ACCOUNT_TYPE"
			TEST_NEXT_TOKEN               = "TEST_NEXT_TOKEN"
			TEST_PAGE_SIZE                = 100
			TEST_MAX_RESULTS              = 100
		)

		var TEST_ITEM_IDS = []string{"TEST_ITEM_ID"}
		var TEST_ENTITLEMENT_STATUS_FILTERS = []string{"TEST_ENTITLEMENT_STATUS_FILTER"}
		var TEST_INVENTORY = gatewayClient.Inventory{}

		// Set up the TProx client
		httpMock := new(mocks.HttpMock)
		sigV4SignerMock := new(mocks.SigV4SignerMock)
		samusTProxClient := &SamusTProxClient{
			tproxSigV4Signer: sigV4SignerMock,
			Client:           httpMock,
		}

		// Create fake HTTP request
		mockRequest := &ListInventoryTwitchRequest{
			TwitchUserID:             TEST_TWITCH_USER_ID,
			AmazonCustomerID:         TEST_AMAZON_CUSTOMER_ID,
			EntitlementAccountType:   TEST_ENTITLEMENT_ACCOUNT_TYPE,
			NextToken:                TEST_NEXT_TOKEN,
			MaxResults:               TEST_MAX_RESULTS,
			ItemIds:                  TEST_ITEM_IDS,
			EntitlementStatusFilters: TEST_ENTITLEMENT_STATUS_FILTERS,
		}

		So(samusTProxClient, ShouldNotBeNil)

		Convey("returns data when the call to TProx succeeds", func() {
			// Create fake HTTP response
			mockResponse := ListInventoryTProxResponse{
				Output: ListInventoryResponse{
					Inventory: TEST_INVENTORY,
				},
			}
			buf, err := json.Marshal(mockResponse)
			if err != nil {
				log.Debug(err)
			}
			mockHttpResponse := new(http.Response)
			mockHttpResponse.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
			mockHttpResponse.StatusCode = http.StatusOK
			httpMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockHttpResponse, nil)
			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, nil)
			resp, s, e := samusTProxClient.ListInventory(context.TODO(), mockRequest)
			results := resp

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)

			So(len(results.Inventory.Entitlements), ShouldEqual, len(TEST_INVENTORY.Entitlements))

			So(s, ShouldEqual, http.StatusOK)
			So(e, ShouldBeNil)
		})

		Convey("returns OK request when TProx returns an empty response", func() {
			mockResponse := ListInventoryTProxResponse{
				Output: ListInventoryResponse{},
			}
			buf, err := json.Marshal(mockResponse)
			if err != nil {
				log.Debug(err)
			}
			mockHttpResponse := new(http.Response)
			mockHttpResponse.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
			mockHttpResponse.StatusCode = http.StatusOK
			httpMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockHttpResponse, nil)
			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, nil)
			resp, s, e := samusTProxClient.ListInventory(context.TODO(), mockRequest)

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)

			So(resp, ShouldNotBeNil)
			So(s, ShouldEqual, http.StatusOK)
			So(e, ShouldBeNil)
		})

		Convey("errors when the call to TProx fails", func() {
			// Create fake HTTP response
			mockResponse := ListInventoryTProxResponse{}

			buf, err := json.Marshal(mockResponse)
			if err != nil {
				log.Debug(err)
			}
			mockHttpResponse := new(http.Response)
			mockHttpResponse.Body = ioutil.NopCloser(bytes.NewBufferString(string(buf[:])))
			mockHttpResponse.StatusCode = http.StatusInternalServerError
			httpMock.On("Do", mock.Anything, mock.Anything, mock.Anything).Return(mockHttpResponse, nil)

			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, nil)
			resp, s, e := samusTProxClient.ListInventory(context.TODO(), mockRequest)

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)
			So(resp, ShouldBeNil)
			So(s, ShouldEqual, http.StatusInternalServerError)
			So(e, ShouldNotBeNil)
		})

		Convey("errors when the SigV4Signer fails", func() {
			sigV4SignerMock.On("Sign", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(http.Header{}, errors.New("Some fool forgot to put in their AWS credentials!"))

			resp, s, e := samusTProxClient.ListInventory(context.TODO(), mockRequest)

			sigV4SignerMock.AssertCalled(t, "Sign", mock.Anything, mock.Anything, TPROX_SIGV4_SERVICE_NAME, TPROX_SIGV4_AWS_REGION, mock.Anything)
			So(resp, ShouldBeNil)
			So(s, ShouldEqual, http.StatusInternalServerError)
			So(e, ShouldNotBeNil)
		})
	})
}
