package clients

import (
	"time"

	twitchConfig "code.justin.tv/common/config"
	"code.justin.tv/foundation/twitchclient"
	substwirp "code.justin.tv/revenue/subscriptions/twirp"
	"code.justin.tv/samus/gateway/settings"

	"github.com/golang/protobuf/ptypes"
	log "github.com/sirupsen/logrus"
	"golang.org/x/net/context"
)

const (
	SubscriptionsEndpoint = "SubscriptionsEndpoint"
)

// Development Environment config
var DevelopmentSubscriptionsConfig = map[string]string{
	SubscriptionsEndpoint: "https://main.us-west-2.beta.subscriptions.twitch.a2z.com",
}

// Staging Environment config
var StagingSubscriptionsConfig = map[string]string{
	SubscriptionsEndpoint: "https://main.us-west-2.beta.subscriptions.twitch.a2z.com",
}

// Production Environment config
var ProductionSubscriptionsConfig = map[string]string{
	SubscriptionsEndpoint: "https://main.us-west-2.prod.subscriptions.twitch.a2z.com",
}

// Redefine substwirp.Subscriptions functions so we can mock.
type SubscriptionsClienter interface {
	GetChannelProducts(context.Context, *substwirp.GetChannelProductsRequest) (*substwirp.GetChannelProductsResponse, error)
	CreateChatNotificationToken(ctx context.Context, userID string, productID string) (*substwirp.CreateChatNotificationTokenResponse, error)
	GetPaidUpgrades(context.Context, *substwirp.GetPaidUpgradesRequest) (*substwirp.GetPaidUpgradesResponse, error)
}

// SubscriptionsClient - embeds substwirp.Subscriptions.
type SubscriptionsClient struct {
	substwirp.Subscriptions
}

// Creates a revenue/Subscriptions client.
func NewSubscriptionsClient() (SubscriptionsClienter, error) {

	conf := settings.GetConfiguration(DevelopmentSubscriptionsConfig, StagingSubscriptionsConfig, ProductionSubscriptionsConfig)

	c := substwirp.NewSubscriptionsProtobufClient(conf[SubscriptionsEndpoint], twitchclient.NewHTTPClient(twitchclient.ClientConf{
		Host:           conf[SubscriptionsEndpoint],
		TimingXactName: "Subscriptions-Client",
		Stats:          twitchConfig.Statsd(),
	}))

	return &SubscriptionsClient{c}, nil
}

// GetChannelProducts wrapper to proxy to subs client API calls
func (sc *SubscriptionsClient) GetChannelProducts(ctx context.Context, request *substwirp.GetChannelProductsRequest) (*substwirp.GetChannelProductsResponse, error) {
	return sc.Subscriptions.GetChannelProducts(ctx, request)
}

func (sc *SubscriptionsClient) CreateChatNotificationToken(ctx context.Context, userID string, productID string) (*substwirp.CreateChatNotificationTokenResponse, error) {
	logger := log.WithFields(log.Fields{
		"userID":    userID,
		"productID": productID,
	})

	time, err := ptypes.TimestampProto(time.Now().UTC())
	if err != nil {
		logger.Error("[SpendSubscriptionCredit] ", "TimestampProto failed: ", err)
		return nil, err
	}

	chatRequest := &substwirp.CreateChatNotificationTokenRequest{
		OwnerId:     userID,
		ProductId:   productID,
		IsPrime:     &substwirp.OptionalBool{Value: true},
		AvailableAt: time,
	}
	return sc.Subscriptions.CreateChatNotificationToken(ctx, chatRequest)

}
