package dynamo

import (
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/juju/errgo"
	log "github.com/sirupsen/logrus"
)

type AccountLinkTable struct {
}

type AccountLink struct {
	UserID       TwitchUserID
	GameID       string
	AccountId    string
	DisplayName  string
	CreationDate time.Time
}

type AccountLinkDao struct {
	BaseDao
}

type IAccountLinkDao interface {
	Put(accountLink *AccountLink) error
	Get(userID, gameID string) (*AccountLink, error)
	Delete(userID, gameID string) error
}

func (u *AccountLinkTable) GetNodeJsRecordKeySelector() NodeJsRecordKeySelector {
	return "record.dynamodb.Keys.id.S"
}

func (u *AccountLinkTable) ConvertAttributeMapToRecord(attributeMap map[string]*dynamodb.AttributeValue) (DynamoTableRecord, error) {
	return &AccountLink{
		UserID:      TwitchUserID(StringFromAttributes(attributeMap, "twitchUserId")),
		GameID:      StringFromAttributes(attributeMap, "gameId"),
		AccountId:   StringFromAttributes(attributeMap, "accountId"),
		DisplayName: StringFromAttributes(attributeMap, "displayName"),
	}, nil
}

func (u *AccountLink) NewAttributeMap() map[string]*dynamodb.AttributeValue {
	itemMap := make(map[string]*dynamodb.AttributeValue)
	PopulateAttributesString(itemMap, "twitchUserId", string(u.UserID))
	PopulateAttributesString(itemMap, "gameId", string(u.GameID))
	PopulateAttributesString(itemMap, "accountId", u.AccountId)
	PopulateAttributesString(itemMap, "displayName", u.DisplayName)
	return itemMap
}

func (u *AccountLink) NewItemKey() map[string]*dynamodb.AttributeValue {
	keyMap := make(map[string]*dynamodb.AttributeValue)
	keyMap["twitchUserId"] = &dynamodb.AttributeValue{S: aws.String(string(u.UserID))}
	keyMap["gameId"] = &dynamodb.AttributeValue{S: aws.String(string(u.GameID))}
	return keyMap
}

func (u *AccountLink) NewHashKeyEqualsExpression() string {
	return "twitchUserId = :twitchUserId"
}

func (u *AccountLink) NewHashKeyExpressionAttributeValues() map[string]*dynamodb.AttributeValue {
	twitchUserIdAttributeValue := &dynamodb.AttributeValue{
		S: aws.String(string(u.UserID)),
	}
	attributeMap := make(map[string]*dynamodb.AttributeValue)
	attributeMap[":twitchUserId"] = twitchUserIdAttributeValue
	return attributeMap
}

func (u *AccountLink) UpdateWithCurrentTimestamp() {
	u.CreationDate = time.Now()
}

func (u *AccountLink) GetTimestamp() time.Time {
	return u.CreationDate
}

func (u *AccountLink) ApplyTimestamp(timestamp time.Time) {
	u.CreationDate = timestamp
}

func NewAccountLinkDao(client *DynamoClient) IAccountLinkDao {
	dao := &AccountLinkDao{}
	dao.client = client
	dao.table = &AccountLinkTable{}
	return dao
}

func (dao *AccountLinkDao) Get(userID string, gameID string) (*AccountLink, error) {
	var AccountLinkResult *AccountLink

	AccountLinkQuery := &AccountLink{
		UserID: TwitchUserID(userID),
		GameID: gameID,
	}

	result, err := dao.client.GetItem(AccountLinkQuery)
	if err != nil {
		return AccountLinkResult, err
	}

	// Result will be nil if no item is found in dynamo
	// This means that they do not have an account linked
	if result == nil {
		return nil, nil
	}

	AccountLinkResult, isAccountLink := result.(*AccountLink)
	if !isAccountLink {
		msg := "Encountered an unexpected type while converting dynamo result to AccountLink"
		log.Error(msg)
		return AccountLinkResult, errgo.New(msg)
	}

	return AccountLinkResult, nil
}

func (dao *AccountLinkDao) Put(entitlement *AccountLink) error {
	return dao.client.PutItemIfNotExists(entitlement)
}

func (dao *AccountLinkDao) Delete(userID string, gameID string) error {
	AccountLinkQuery := &AccountLink{
		UserID: TwitchUserID(userID),
		GameID: gameID,
	}

	err := dao.client.DeleteItem(AccountLinkQuery)
	if err != nil {
		return err
	}

	return nil
}

func (te *AccountLink) GetTable() DynamoTable {
	return &AccountLinkTable{}
}
