package dynamo

import (
	"errors"
	"fmt"

	"reflect"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/juju/errgo"
	log "github.com/sirupsen/logrus"
)

type DynamoClientConfig struct {
	AwsRegion           string
	TableName           string
	EndpointOverride    string
	CredentialsOverride *credentials.Credentials
}

type DynamoClient struct {
	Dynamo       *dynamodb.DynamoDB
	ClientConfig *DynamoClientConfig
}

type ScanFilter struct {
	Expression *string
	Names      map[string]*string
	Values     map[string]*dynamodb.AttributeValue
}

func NewClient(config *DynamoClientConfig) *DynamoClient {
	awsConfig := &aws.Config{
		Region:      aws.String(config.AwsRegion),
		Credentials: config.CredentialsOverride,
	}
	dynamo := dynamodb.New(session.New(), awsConfig)
	if config.EndpointOverride != "" {
		dynamo.Endpoint = config.EndpointOverride
	}

	return &DynamoClient{
		Dynamo:       dynamo,
		ClientConfig: config,
	}
}

func (c *DynamoClient) PutItem(record DynamoTableRecord) error {
	tableName := c.ClientConfig.TableName
	record.UpdateWithCurrentTimestamp()
	_, err := c.Dynamo.PutItem(NewPutItemInput(tableName, record))
	if err != nil {
		log.Errorf("DynamoClient: Encountered an error calling dynamo PutItem, %v", err)
		return err
	}
	return nil
}

func (c *DynamoClient) PutItemIfNotExists(record DynamoTableRecord) error {
	tableName := c.ClientConfig.TableName
	record.UpdateWithCurrentTimestamp()

	hashKey := record.NewHashKeyExpressionAttributeValues()
	if len(hashKey) != 1 {
		return errors.New(fmt.Sprintf("hashkey=[%+v] for record=[%+v] does not have exactly one value", hashKey, record))
	}

	key := reflect.ValueOf(hashKey).MapKeys()[0].String()

	input := NewPutItemInput(tableName, record)
	input.SetConditionExpression("attribute_not_exists(" + key[1:] + ")")

	_, err := c.Dynamo.PutItem(input)
	if err != nil {
		log.Errorf("DynamoClient: Encountered an error calling dynamo PutItem, %v", err)
		return err
	}

	return nil
}

func (c *DynamoClient) UpdateItem(record DynamoTableRecord) error {
	tableName := c.ClientConfig.TableName
	record.UpdateWithCurrentTimestamp()
	_, err := c.Dynamo.UpdateItem(NewUpdateItemInput(tableName, record))
	if err != nil {
		msg := fmt.Sprintf("DynamoClient: Encountered an error calling dynamo UpdateItem, %v", err)
		log.WithError(err).Error(msg)
		return errgo.Notef(err, msg)
	}
	return nil
}

func (c *DynamoClient) GetItem(record DynamoTableRecord) (DynamoTableRecord, error) {
	tableName := c.ClientConfig.TableName
	result, err := c.Dynamo.GetItem(NewGetItemInput(tableName, record))
	if err != nil {
		msg := fmt.Sprintf("DynamoClient: Encountered an error calling dynamo GetItem, %v", err)
		log.WithError(err).Error(msg)
		return nil, errgo.Notef(err, msg)
	}

	// Result map is empty if no record is found
	if len(result.Item) != 0 {
		lastUpdatedTime, err := TimeFromAttributes(result.Item, "lastUpdated")
		if err != nil {
			msg := fmt.Sprintf("DynamoClient: Encountered an error getting lastUpdated time after dynamo GetItem, %v", err)
			log.WithError(err).Error(msg)
			return nil, errgo.Notef(err, msg)
		}

		record, err := record.GetTable().ConvertAttributeMapToRecord(result.Item)
		if err != nil {
			msg := fmt.Sprintf("DynamoClient: Encountered an error converting dyanmo GetItem result to record, %v", err)
			log.WithError(err).Error(msg)
			return nil, errgo.Notef(err, msg)
		}

		record.ApplyTimestamp(lastUpdatedTime)
		return record, nil
	} else {
		return nil, nil
	}
}

func (c *DynamoClient) DeleteItem(record DynamoTableRecord) error {
	tableName := c.ClientConfig.TableName
	_, err := c.Dynamo.DeleteItem(NewDeleteItemInput(tableName, record))
	if err != nil {
		msg := fmt.Sprintf("DynamoClient: Encountered an error calling dynamo DeleteItem, %v", err)
		log.WithError(err).Error(msg)
		return errgo.Notef(err, msg)
	}
	return nil
}

func (c *DynamoClient) QueryByHashKey(record DynamoTableRecord) ([]DynamoTableRecord, error) {
	tableName := c.ClientConfig.TableName
	queryInput := &dynamodb.QueryInput{
		TableName:                 aws.String(string(tableName)),
		KeyConditionExpression:    aws.String(record.NewHashKeyEqualsExpression()),
		ExpressionAttributeValues: record.NewHashKeyExpressionAttributeValues(),
	}

	queryOutput, err := c.Dynamo.Query(queryInput)
	if err != nil {
		msg := fmt.Sprintf("DynamoClient: Encountered an error calling dynamo Query, %v", err)
		log.WithError(err).Error(msg)
		return nil, errgo.Notef(err, msg)
	}

	results := make([]DynamoTableRecord, 0, len(queryOutput.Items))
	for _, item := range queryOutput.Items {
		lastUpdatedTime, err := TimeFromAttributes(item, "lastUpdated")
		if err != nil {
			msg := fmt.Sprintf("DynamoClient: Encountered an error getting lastUpdated time after dynamo Query, %v", err)
			log.WithError(err).Error(msg)
			return nil, errgo.Notef(err, msg)
		}

		record, err := record.GetTable().ConvertAttributeMapToRecord(item)
		if err != nil {
			msg := fmt.Sprintf("DynamoClient: Encountered an error converting dyanmo Query result item to record, %v", err)
			log.WithError(err).Error(msg)
			return nil, errgo.Notef(err, msg)
		}

		record.ApplyTimestamp(lastUpdatedTime)

		results = append(results, record)
	}
	return results, nil
}

func (c *DynamoClient) ScanTable(table DynamoScanTable, filter ScanFilter) (records []DynamoScanTableRecord, err error) {
	tableName := c.ClientConfig.TableName

	input := &dynamodb.ScanInput{
		TableName:                 aws.String(string(tableName)),
		FilterExpression:          filter.Expression,
		ExpressionAttributeNames:  filter.Names,
		ExpressionAttributeValues: filter.Values,
	}

	output, err := c.Dynamo.Scan(input)
	if err != nil {
		msg := fmt.Sprintf("DynamoClient: Encountered an error calling dynamo Scan, %v", err)
		log.WithError(err).Error(msg)
		return nil, errgo.Notef(err, msg)
	}

	records, err = table.ConvertScanOutputToRecords(output)
	if err != nil {
		msg := fmt.Sprintf("DynamoClient: Failed to parse Scan output, %v", err)
		log.WithError(err).Error(msg)
		return nil, errgo.Notef(err, msg)
	}

	// Continue scanning until we've exhausted the table
	for output.LastEvaluatedKey != nil {
		input.ExclusiveStartKey = output.LastEvaluatedKey
		output, err = c.Dynamo.Scan(input)
		if err != nil {
			msg := fmt.Sprintf("DynamoClient: Encountered an error calling dynamo Scan, %v", err)
			log.WithError(err).Error(msg)
			return nil, errgo.Notef(err, msg)
		}
		toAppend, err := table.ConvertScanOutputToRecords(output)
		if err != nil {
			msg := fmt.Sprintf("DynamoClient: Failed to parse Scan output, %v", err)
			log.WithError(err).Error(msg)
			return nil, errgo.Notef(err, msg)
		}

		records = append(records, toAppend...)
	}

	return records, nil
}
