package dynamo

import (
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/juju/errgo"
	log "github.com/sirupsen/logrus"
)

type TwitchUserID string
type OfferID string

type PrimeEntitlementTable struct {
}

type PrimeEntitlement struct {
	TwitchUserID TwitchUserID
	OfferID      OfferID
	IsEntitled   bool
	ClaimDate    time.Time
}

type PrimeEntitlementDao struct {
	BaseDao
}

type IPrimeEntitlementDao interface {
	Put(entitlement *PrimeEntitlement) error
	Get(userID string, offerID string) (*PrimeEntitlement, error)
}

func (u *PrimeEntitlementTable) GetNodeJsRecordKeySelector() NodeJsRecordKeySelector {
	return "record.dynamodb.Keys.id.S"
}

func (u *PrimeEntitlementTable) ConvertAttributeMapToRecord(attributeMap map[string]*dynamodb.AttributeValue) (DynamoTableRecord, error) {
	return &PrimeEntitlement{
		TwitchUserID: TwitchUserID(StringFromAttributes(attributeMap, "twitchUserId")),
		OfferID:      OfferID(StringFromAttributes(attributeMap, "offerId")),
		IsEntitled:   BoolFromAttributes(attributeMap, "isEntitled"),
	}, nil
}

func (u *PrimeEntitlement) NewAttributeMap() map[string]*dynamodb.AttributeValue {
	itemMap := make(map[string]*dynamodb.AttributeValue)
	PopulateAttributesString(itemMap, "twitchUserId", string(u.TwitchUserID))
	PopulateAttributesString(itemMap, "offerId", string(u.OfferID))
	PopulateAttributesTime(itemMap, "claimDate", u.ClaimDate)
	PopulateAttributesBool(itemMap, "isEntitled", u.IsEntitled)
	return itemMap
}

func (u *PrimeEntitlement) NewItemKey() map[string]*dynamodb.AttributeValue {
	keyMap := make(map[string]*dynamodb.AttributeValue)
	keyMap["twitchUserId"] = &dynamodb.AttributeValue{S: aws.String(string(u.TwitchUserID))}
	keyMap["offerId"] = &dynamodb.AttributeValue{S: aws.String(string(u.OfferID))}
	return keyMap
}

func (u *PrimeEntitlement) NewHashKeyEqualsExpression() string {
	return "twitchUserId = :twitchUserId"
}

func (u *PrimeEntitlement) NewHashKeyExpressionAttributeValues() map[string]*dynamodb.AttributeValue {
	twitchUserIdAttributeValue := &dynamodb.AttributeValue{
		S: aws.String(string(u.TwitchUserID)),
	}
	offerIdAttributeValue := &dynamodb.AttributeValue{
		S: aws.String(string(u.OfferID)),
	}
	attributeMap := make(map[string]*dynamodb.AttributeValue)
	attributeMap[":twitchUserId"] = twitchUserIdAttributeValue
	attributeMap[":offerId"] = offerIdAttributeValue
	return attributeMap
}

func (u *PrimeEntitlement) UpdateWithCurrentTimestamp() {
	u.ClaimDate = time.Now()
}

func (u *PrimeEntitlement) GetTimestamp() time.Time {
	return u.ClaimDate
}

func (u *PrimeEntitlement) ApplyTimestamp(timestamp time.Time) {
	u.ClaimDate = timestamp
}

func NewPrimeEntitlementDao(client *DynamoClient) IPrimeEntitlementDao {
	dao := &PrimeEntitlementDao{}
	dao.client = client
	dao.table = &PrimeEntitlementTable{}
	return dao
}

func (dao *PrimeEntitlementDao) Put(entitlement *PrimeEntitlement) error {
	return dao.client.PutItem(entitlement)
}

func (dao *PrimeEntitlementDao) Get(userID string, offerID string) (*PrimeEntitlement, error) {
	var PrimeEntitlementResult *PrimeEntitlement

	PrimeEntitlementQuery := &PrimeEntitlement{
		TwitchUserID: TwitchUserID(userID),
		OfferID:      OfferID(offerID),
	}

	result, err := dao.client.GetItem(PrimeEntitlementQuery)
	if err != nil {
		return PrimeEntitlementResult, err
	}

	// Result will be nil if no item is found in dynamo
	// This means that they do not have the offer entitlement
	if result == nil {
		PrimeEntitlement := &PrimeEntitlement{
			TwitchUserID: TwitchUserID(userID),
			OfferID:      OfferID(offerID),
			IsEntitled:   false,
		}
		return PrimeEntitlement, nil
	}
	PrimeEntitlementResult, isPrimeEntitlement := result.(*PrimeEntitlement)
	if !isPrimeEntitlement {
		msg := "Encountered an unexpected type while converting dynamo result to PrimeEntitlement"
		log.Error(msg)
		return PrimeEntitlementResult, errgo.New(msg)
	}

	return PrimeEntitlementResult, nil
}

func (te *PrimeEntitlement) GetTable() DynamoTable {
	return &PrimeEntitlementTable{}
}
