package dynamo

import (
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/juju/errgo"

	log "github.com/sirupsen/logrus"
)

type UserId string

type UsersTable struct {
}

type User struct {
	Id                    UserId
	IsSubscriptionMessage bool
	LastUpdated           time.Time
}

type UserDao struct {
	BaseDao
}

type IUserDao interface {
	Put(user *User) error
	GetOrCreate(id UserId) (*User, error)
	Delete(id UserId) error
}

func (u *UsersTable) GetNodeJsRecordKeySelector() NodeJsRecordKeySelector {
	return "record.dynamodb.Keys.id.S"
}

func (u *UsersTable) ConvertAttributeMapToRecord(attributeMap map[string]*dynamodb.AttributeValue) (DynamoTableRecord, error) {
	return &User{
		Id:                    UserId(StringFromAttributes(attributeMap, "id")),
		IsSubscriptionMessage: BoolFromAttributes(attributeMap, "isSubscriptionMessage"),
	}, nil
}

func (u *User) NewAttributeMap() map[string]*dynamodb.AttributeValue {
	itemMap := make(map[string]*dynamodb.AttributeValue)
	PopulateAttributesString(itemMap, "id", string(u.Id))
	PopulateAttributesBool(itemMap, "isSubscriptionMessage", u.IsSubscriptionMessage)
	PopulateAttributesTime(itemMap, "lastUpdated", u.LastUpdated)
	return itemMap
}

func (u *User) NewItemKey() map[string]*dynamodb.AttributeValue {
	keyMap := make(map[string]*dynamodb.AttributeValue)
	keyMap["id"] = &dynamodb.AttributeValue{S: aws.String(string(u.Id))}
	return keyMap
}

func (u *User) NewHashKeyEqualsExpression() string {
	return "id = :id"
}

func (u *User) NewHashKeyExpressionAttributeValues() map[string]*dynamodb.AttributeValue {
	attributeValue := &dynamodb.AttributeValue{
		S: aws.String(string(u.Id)),
	}
	attributeMap := make(map[string]*dynamodb.AttributeValue)
	attributeMap[":id"] = attributeValue
	return attributeMap
}

func (u *User) UpdateWithCurrentTimestamp() {
	u.LastUpdated = time.Now()
}

func (u *User) GetTimestamp() time.Time {
	return u.LastUpdated
}

func (u *User) ApplyTimestamp(timestamp time.Time) {
	u.LastUpdated = timestamp
}

func NewUserDao(client *DynamoClient) IUserDao {
	dao := &UserDao{}
	dao.client = client
	dao.table = &UsersTable{}
	return dao
}

func (dao *UserDao) Put(user *User) error {
	return dao.client.PutItem(user)
}

func (dao *UserDao) GetOrCreate(userId UserId) (*User, error) {
	var userResult *User

	userQuery := &User{
		Id: UserId(userId),
	}

	result, err := dao.client.GetItem(userQuery)
	if err != nil {
		return userResult, err
	}

	// Result will be nil if no item is found in dynamo
	if result == nil {
		user := &User{
			Id:                    UserId(userId),
			IsSubscriptionMessage: true,
		}
		user.UpdateWithCurrentTimestamp()
		err := dao.Put(user)
		if err != nil {
			return nil, err
		}
		return user, nil
	}

	userResult, isUser := result.(*User)
	if !isUser {
		msg := "Encountered an unexpected type while converting dynamo result to user"
		log.Error(msg)
		return userResult, errgo.New(msg)
	}

	return userResult, nil
}

func (dao *UserDao) Delete(userId UserId) error {
	deletionRecord := &User{
		Id: UserId(userId),
	}
	return dao.client.DeleteItem(deletionRecord)
}

func (u *User) GetTable() DynamoTable {
	return &UsersTable{}
}
